<?php

namespace App\Console;

use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Foundation\Console\Kernel as ConsoleKernel;

class Kernel extends ConsoleKernel
{
    /**
     * Define the application's command schedule.
     */
    protected function schedule(Schedule $schedule): void
    {
        // Sync wallet balances with external APIs (every 5 minutes)
        $schedule->command('fintech:sync-wallets')
            ->everyFiveMinutes()
            ->withoutOverlapping()
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/cron-sync-wallets.log'));

        // Update virtual card status (every 10 minutes)
        $schedule->command('fintech:sync-virtual-cards')
            ->everyTenMinutes()
            ->withoutOverlapping()
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/cron-sync-cards.log'));

        // Check for failed transactions and retry (hourly)
        $schedule->command('fintech:retry-failed-transactions')
            ->hourly()
            ->withoutOverlapping()
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/cron-retry-transactions.log'));

        // Update data/airtime order status (every 3 minutes)
        $schedule->command('fintech:sync-bill-payments')
            ->everyThreeMinutes()
            ->withoutOverlapping()
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/cron-sync-bills.log'));

        // Notify users if card or account is inactive (daily at 9 AM)
        $schedule->command('fintech:notify-inactive-accounts')
            ->dailyAt('09:00')
            ->withoutOverlapping()
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/cron-notify-inactive.log'));

        // Clean up old notifications (weekly)
        $schedule->command('fintech:cleanup-notifications')
            ->weekly()
            ->withoutOverlapping()
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/cron-cleanup.log'));

        // Generate daily reports (daily at 1 AM)
        $schedule->command('fintech:generate-daily-reports')
            ->dailyAt('01:00')
            ->withoutOverlapping()
            ->runInBackground()
            ->appendOutputTo(storage_path('logs/cron-reports.log'));

        // Log all cron executions
        $schedule->call(function () {
            \App\Models\CronLog::create([
                'task_name' => 'scheduler_heartbeat',
                'status' => 'success',
                'message' => 'Scheduler is running',
                'executed_at' => now(),
                'execution_time' => 0.001
            ]);
        })->everyMinute();
    }

    /**
     * Register the commands for the application.
     */
    protected function commands(): void
    {
        $this->load(__DIR__.'/Commands');

        require base_path('routes/console.php');
    }
}