<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\User;
use App\Models\Wallet;
use App\Models\Transaction;
use App\Models\KycVerification;
use App\Services\MonnifyService;
use App\Services\AnchorService;
use App\Services\FundingSourceService;
use App\Http\Requests\Admin\AdjustWalletBalanceRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Config;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    /**
     * Display a listing of users with search and filters.
     */
    public function index(Request $request)
    {
        $query = User::with(['wallet', 'kycVerification']);

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'LIKE', "%{$search}%")
                  ->orWhere('last_name', 'LIKE', "%{$search}%")
                  ->orWhere('email', 'LIKE', "%{$search}%")
                  ->orWhere('phone', 'LIKE', "%{$search}%");
            });
        }

        // Status filter
        if ($request->filled('status')) {
            if ($request->status === 'active') {
                $query->where('is_active', true);
            } elseif ($request->status === 'inactive') {
                $query->where('is_active', false);
            } elseif ($request->status === 'blocked') {
                $query->where('is_blocked', true);
            }
        }

        // KYC filter
        if ($request->filled('kyc_status')) {
            $query->whereHas('kycVerification', function ($q) use ($request) {
                $q->where('status', $request->kyc_status);
            });
        }

        // Sorting
        $sortField = $request->get('sort', 'created_at');
        $sortDirection = $request->get('direction', 'desc');
        
        $allowedSorts = ['created_at', 'first_name', 'last_name', 'email', 'phone'];
        if (in_array($sortField, $allowedSorts)) {
            $query->orderBy($sortField, $sortDirection);
        }

        $users = $query->paginate(20)->withQueryString();

        return view('admin.users.index', compact('users'));
    }

    /**
     * Display the specified user details.
     */
    public function show(User $user)
    {
        $user->load(['wallet', 'kycVerification', 'virtualCards', 'virtualAccounts']);
        
        // Get recent transactions
        $transactions = Transaction::where(function ($query) use ($user) {
            $query->where('user_id', $user->id)
                  ->orWhere('recipient_user_id', $user->id);
        })->latest()->limit(10)->get();

        return view('admin.users.show', compact('user', 'transactions'));
    }

    /**
     * Block or unblock a user.
     */
    public function toggleBlock(User $user)
    {
        $user->update([
            'is_blocked' => !$user->is_blocked,
            'blocked_at' => $user->is_blocked ? null : now(),
        ]);

        $action = $user->is_blocked ? 'blocked' : 'unblocked';
        
        return back()->with('success', "User has been {$action} successfully.");
    }

    /**
     * Activate or deactivate a user.
     */
    public function toggleActive(User $user)
    {
        $user->update([
            'is_active' => !$user->is_active,
        ]);

        $action = $user->is_active ? 'activated' : 'deactivated';
        
        return back()->with('success', "User has been {$action} successfully.");
    }

    /**
     * Manually verify a user.
     */
    public function verify(User $user)
    {
        $user->update([
            'email_verified_at' => now(),
            'phone_verified_at' => now(),
        ]);

        return back()->with('success', 'User has been manually verified successfully.');
    }

    /**
     * Reset user password.
     */
    public function resetPassword(Request $request, User $user)
    {
        $request->validate([
            'password' => 'required|string|min:8|confirmed',
        ]);

        $user->update([
            'password' => Hash::make($request->password),
        ]);

        return back()->with('success', 'User password has been reset successfully.');
    }

    /**
     * Reset user 2FA.
     */
    public function reset2FA(User $user)
    {
        $user->update([
            'two_factor_secret' => null,
            'two_factor_confirmed_at' => null,
        ]);

        return back()->with('success', 'User 2FA has been reset successfully.');
    }

    /**
     * Delete a user (soft delete).
     */
    public function destroy(User $user)
    {
        // Check if user has pending transactions
        $pendingTransactions = Transaction::where('user_id', $user->id)
            ->where('status', 'pending')
            ->count();

        if ($pendingTransactions > 0) {
            return back()->withErrors(['error' => 'Cannot delete user with pending transactions.']);
        }

        $user->delete();

        return redirect()->route('admin.users.index')
            ->with('success', 'User has been deleted successfully.');
    }

    /**
     * Show user wallet details.
     */
    public function wallet(User $user)
    {
        $wallet = $user->wallet;
        
        if (!$wallet) {
            return back()->withErrors(['error' => 'User does not have a wallet.']);
        }

        $transactions = Transaction::where('user_id', $user->id)
            ->orWhere('recipient_user_id', $user->id)
            ->latest()
            ->paginate(20);

        return view('admin.users.wallet', compact('user', 'wallet', 'transactions'));
    }

    /**
     * Adjust user wallet balance.
     */
    public function adjustBalance(AdjustWalletBalanceRequest $request, User $user)
    {
        $wallet = $user->wallet;
        
        if (!$wallet) {
            return back()->withErrors(['error' => 'User does not have a wallet.']);
        }

        $amount = $request->validated('amount');
        $type = $request->validated('type');
        $fundingSource = $request->getFundingSource();
        $narration = $request->validated('narration');
        
        if ($type === 'debit' && $wallet->balance < $amount) {
            return back()->withErrors(['error' => 'Insufficient wallet balance for debit.']);
        }

        // Check if approval is required
        if ($request->requiresApproval()) {
            // In a real application, you might queue this for approval
            return back()->withErrors(['error' => 'This funding source requires approval. Please contact your supervisor.']);
        }

        try {
            DB::transaction(function () use ($user, $wallet, $amount, $type, $fundingSource, $narration) {
                // Create transaction record with API source tracking
                $metadata = [
                    'admin_id' => auth('admin')->id(),
                    'admin_name' => auth('admin')->user()->name,
                    'reason' => $narration,
                    'funding_source' => $fundingSource,
                    'funding_source_name' => FundingSourceService::getSourceName($fundingSource),
                ];

                // If using API funding, add API response details
                if ($fundingSource !== FundingSourceService::MANUAL) {
                    $apiResult = $this->processApiFunding($fundingSource, $user, $amount, $type);
                    $metadata['api_result'] = $apiResult;
                    
                    if (!$apiResult['success']) {
                        throw new \Exception($apiResult['message']);
                    }
                }

                $transaction = Transaction::create([
                    'user_id' => $user->id,
                    'wallet_id' => $wallet->id,
                    'type' => $type === 'credit' ? 'credit' : 'debit',
                    'category' => 'admin_adjustment',
                    'amount' => $amount,
                    'fee' => 0,
                    'total_amount' => $amount,
                    'description' => $narration . ' (Admin Adjustment)',
                    'narration' => $narration,
                    'status' => 'completed',
                    'reference' => 'ADJ-' . strtoupper(uniqid()),
                    'metadata' => $metadata,
                    'processed_at' => now(),
                ]);

                // Update wallet balance
                if ($type === 'credit') {
                    $wallet->increment('balance', $amount);
                } else {
                    $wallet->decrement('balance', $amount);
                }

                // Log admin action
                AuditLog::create([
                    'admin_id' => auth('admin')->id(),
                    'action' => 'adjust_wallet_balance',
                    'target_type' => 'Wallet',
                    'target_id' => $wallet->id,
                    'data' => [
                        'type' => $type,
                        'amount' => $amount,
                        'funding_source' => $fundingSource,
                        'transaction_id' => $transaction->id,
                    ],
                    'ip_address' => request()->ip(),
                ]);
            });

            $sourceName = FundingSourceService::getSourceName($fundingSource);
            $sourceText = $fundingSource !== FundingSourceService::MANUAL ? " via {$sourceName}" : "";
            return back()->with('success', "Wallet balance {$type}ed by ₦" . number_format($amount, 2) . $sourceText);
            
        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Failed to adjust balance: ' . $e->getMessage()]);
        }
    }

    /**
     * Process API-based funding using Monnify or Anchor.
     */
    private function processApiFunding(string $source, User $user, float $amount, string $type): array
    {
        try {
            switch ($source) {
                case FundingSourceService::MONNIFY:
                    return $this->processMonnifyFunding($user, $amount, $type);
                
                case FundingSourceService::ANCHOR:
                    return $this->processAnchorFunding($user, $amount, $type);
                
                default:
                    return ['success' => false, 'message' => 'Invalid funding source'];
            }
        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    /**
     * Process Monnify-based funding simulation.
     */
    private function processMonnifyFunding(User $user, float $amount, string $type): array
    {
        $monnifyService = app(MonnifyService::class);
        
        // For admin simulation, we'll create a mock transaction
        // In real implementation, this would trigger actual API calls
        
        if ($type === 'credit') {
            // Simulate funding via Monnify virtual account
            return [
                'success' => true,
                'message' => 'Monnify funding simulation successful',
                'api_reference' => 'MNF-' . uniqid(),
                'provider' => 'monnify',
                'simulated' => true,
                'timestamp' => now(),
            ];
        } else {
            // Simulate withdrawal via Monnify transfer
            return [
                'success' => true,
                'message' => 'Monnify withdrawal simulation successful',
                'api_reference' => 'MNF-WD-' . uniqid(),
                'provider' => 'monnify',
                'simulated' => true,
                'timestamp' => now(),
            ];
        }
    }

    /**
     * Process Anchor-based funding simulation.
     */
    private function processAnchorFunding(User $user, float $amount, string $type): array
    {
        $anchorService = app(AnchorService::class);
        
        // For admin simulation, we'll create a mock transaction
        // In real implementation, this would trigger actual API calls
        
        if ($type === 'credit') {
            // Simulate funding via Anchor wallet funding
            return [
                'success' => true,
                'message' => 'Anchor funding simulation successful',
                'api_reference' => 'ANC-' . uniqid(),
                'provider' => 'anchor',
                'simulated' => true,
                'timestamp' => now(),
            ];
        } else {
            // Simulate withdrawal via Anchor
            return [
                'success' => true,
                'message' => 'Anchor withdrawal simulation successful',
                'api_reference' => 'ANC-WD-' . uniqid(),
                'provider' => 'anchor',
                'simulated' => true,
                'timestamp' => now(),
            ];
        }
    }

    /**
     * Freeze or unfreeze user wallet.
     */
    public function toggleWalletFreeze(User $user)
    {
        $wallet = $user->wallet;
        
        if (!$wallet) {
            return back()->withErrors(['error' => 'User does not have a wallet.']);
        }

        $wallet->update([
            'is_frozen' => !$wallet->is_frozen,
            'frozen_at' => $wallet->is_frozen ? null : now(),
        ]);

        $action = $wallet->is_frozen ? 'frozen' : 'unfrozen';
        
        return back()->with('success', "User wallet has been {$action} successfully.");
    }
}