<?php

namespace App\Services\CardProviders;

use App\Models\ApiConfig;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class BlocService implements CardProviderInterface
{
    protected ?ApiConfig $config;

    public function __construct()
    {
        $this->config = ApiConfig::where('slug', 'bloc-io')->first();
    }

    /**
     * Create a new virtual card for a user.
     */
    public function createCard(array $data): array
    {
        $this->ensureConfigured();

        $payload = [
            'type' => $data['card_type'] ?? 'virtual',
            'brand' => $data['brand'] ?? 'visa',
            'currency' => $data['currency'] ?? 'USD',
            'amount' => $data['amount'] ?? 0,
            'user' => [
                'first_name' => $data['user']['first_name'],
                'last_name' => $data['user']['last_name'],
                'email' => $data['user']['email'],
                'phone' => $data['user']['phone'] ?? null,
            ],
            'limits' => [
                'daily_limit' => $data['daily_limit'],
                'monthly_limit' => $data['monthly_limit'],
            ],
        ];

        try {
            $response = $this->makeApiCall('POST', '/api/v1/cards', $payload);

            if (!$response['success']) {
                throw new \Exception($response['message'] ?? 'Card creation failed');
            }

            return [
                'success' => true,
                'data' => [
                    'card_id' => $response['data']['id'],
                    'masked_pan' => $response['data']['masked_pan'],
                    'card_type' => $response['data']['type'],
                    'brand' => $response['data']['brand'],
                    'currency' => $response['data']['currency'],
                    'balance' => $response['data']['balance'],
                    'status' => $response['data']['status'],
                    'expires_at' => $response['data']['expires_at'],
                    'provider_reference' => $response['data']['reference'],
                    'metadata' => $response['data'],
                ],
            ];
        } catch (\Exception $e) {
            Log::error('Bloc card creation failed', [
                'error' => $e->getMessage(),
                'payload' => $payload,
            ]);

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Get card details from provider.
     */
    public function getCard(string $cardId): array
    {
        $this->ensureConfigured();

        try {
            $response = $this->makeApiCall('GET', "/api/v1/cards/{$cardId}");

            if (!$response['success']) {
                throw new \Exception($response['message'] ?? 'Failed to fetch card details');
            }

            return [
                'success' => true,
                'data' => $response['data'],
            ];
        } catch (\Exception $e) {
            Log::error('Bloc get card failed', [
                'card_id' => $cardId,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Fund a virtual card.
     */
    public function fundCard(string $cardId, float $amount): array
    {
        $this->ensureConfigured();

        $payload = [
            'amount' => $amount,
            'description' => 'Card funding',
        ];

        try {
            $response = $this->makeApiCall('POST', "/api/v1/cards/{$cardId}/fund", $payload);

            if (!$response['success']) {
                throw new \Exception($response['message'] ?? 'Card funding failed');
            }

            return [
                'success' => true,
                'data' => $response['data'],
            ];
        } catch (\Exception $e) {
            Log::error('Bloc card funding failed', [
                'card_id' => $cardId,
                'amount' => $amount,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Freeze a virtual card.
     */
    public function freezeCard(string $cardId): array
    {
        $this->ensureConfigured();

        try {
            $response = $this->makeApiCall('POST', "/api/v1/cards/{$cardId}/freeze");

            return [
                'success' => $response['success'] ?? false,
                'message' => $response['message'] ?? 'Card frozen successfully',
            ];
        } catch (\Exception $e) {
            Log::error('Bloc card freeze failed', [
                'card_id' => $cardId,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Unfreeze a virtual card.
     */
    public function unfreezeCard(string $cardId): array
    {
        $this->ensureConfigured();

        try {
            $response = $this->makeApiCall('POST', "/api/v1/cards/{$cardId}/unfreeze");

            return [
                'success' => $response['success'] ?? false,
                'message' => $response['message'] ?? 'Card unfrozen successfully',
            ];
        } catch (\Exception $e) {
            Log::error('Bloc card unfreeze failed', [
                'card_id' => $cardId,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Terminate a virtual card.
     */
    public function terminateCard(string $cardId): array
    {
        $this->ensureConfigured();

        try {
            $response = $this->makeApiCall('DELETE', "/api/v1/cards/{$cardId}");

            return [
                'success' => $response['success'] ?? false,
                'message' => $response['message'] ?? 'Card terminated successfully',
            ];
        } catch (\Exception $e) {
            Log::error('Bloc card termination failed', [
                'card_id' => $cardId,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Get card transaction history.
     */
    public function getTransactions(string $cardId, array $filters = []): array
    {
        $this->ensureConfigured();

        try {
            $response = $this->makeApiCall('GET', "/api/v1/cards/{$cardId}/transactions", $filters);

            return [
                'success' => $response['success'] ?? false,
                'data' => $response['data'] ?? [],
            ];
        } catch (\Exception $e) {
            Log::error('Bloc get transactions failed', [
                'card_id' => $cardId,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Check if the provider is properly configured.
     */
    public function isConfigured(): bool
    {
        return $this->config && 
               !empty($this->config->api_key) && 
               !empty($this->config->effective_base_url);
    }

    /**
     * Get provider name.
     */
    public function getProviderName(): string
    {
        return 'Bloc.io';
    }

    /**
     * Get provider slug.
     */
    public function getProviderSlug(): string
    {
        return 'bloc-io';
    }

    /**
     * Make API call to Bloc.
     */
    protected function makeApiCall(string $method, string $endpoint, array $data = []): array
    {
        $url = $this->config->effective_base_url . $endpoint;
        
        $headers = [
            'Authorization' => 'Bearer ' . $this->config->api_key,
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
        ];

        $response = Http::withHeaders($headers)
            ->timeout(30)
            ->retry(2, 1000);

        if ($method === 'GET') {
            $response = $response->get($url, $data);
        } elseif ($method === 'POST') {
            $response = $response->post($url, $data);
        } elseif ($method === 'PUT') {
            $response = $response->put($url, $data);
        } elseif ($method === 'DELETE') {
            $response = $response->delete($url, $data);
        }

        if (!$response->successful()) {
            throw new \Exception(
                'API call failed: ' . $response->status() . ' - ' . $response->body()
            );
        }

        try {
            return $response->json();
        } catch (\Throwable $e) {
            throw new \Exception(
                'Failed to parse JSON response: ' . $e->getMessage() . ' | Response body: ' . $response->body(),
                0,
                $e
            );
        }
    }

    /**
     * Ensure the provider is configured.
     */
    protected function ensureConfigured(): void
    {
        if (!$this->config) {
            throw new \Exception('Bloc.io API configuration not found');
        }
        
        if (!$this->isConfigured()) {
            throw new \Exception('Bloc.io provider is not properly configured');
        }
    }
}