<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\VirtualCard;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use ReflectionClass;

class VirtualCardAnalyticsIssueFixTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        $this->artisan('migrate');
    }

    /** @test */
    public function test_month_format_expression_is_string_not_callable()
    {
        $controller = new \App\Http\Controllers\Admin\VirtualCardController();
        
        // Use reflection to test private method
        $reflection = new ReflectionClass($controller);
        $method = $reflection->getMethod('getMonthFormatExpression');
        $method->setAccessible(true);
        
        $result = $method->invoke($controller);
        
        // Should return a string, not a callable
        $this->assertIsString($result);
        $this->assertNotEmpty($result);
        
        // Should contain SQL format expression
        $this->assertMatchesRegularExpression('/strftime|DATE_FORMAT|to_char/', $result);
    }

    /** @test */
    public function test_transaction_month_format_expression_is_string_not_callable()
    {
        $controller = new \App\Http\Controllers\Admin\VirtualCardController();
        
        // Use reflection to test private method
        $reflection = new ReflectionClass($controller);
        $method = $reflection->getMethod('getTransactionMonthFormatExpression');
        $method->setAccessible(true);
        
        $result = $method->invoke($controller);
        
        // Should return a string, not a callable
        $this->assertIsString($result);
        $this->assertNotEmpty($result);
        
        // Should contain SQL format expression with transactions prefix
        $this->assertMatchesRegularExpression('/strftime|DATE_FORMAT|to_char/', $result);
        $this->assertStringContainsString('transactions.created_at', $result);
    }

    /** @test */
    public function test_analytics_method_executes_without_fatal_errors()
    {
        // Create test data
        $user = User::factory()->create();
        
        VirtualCard::factory()->create([
            'user_id' => $user->id,
            'status' => 'active',
            'created_at' => now()->subDays(5)
        ]);

        $controller = new \App\Http\Controllers\Admin\VirtualCardController();
        
        // This should not throw any fatal errors about calling string as function
        $response = $controller->analytics();
        
        $this->assertInstanceOf(\Illuminate\View\View::class, $response);
        
        $data = $response->getData();
        $analytics = $data['analytics'];
        
        // Verify the data is properly structured
        $this->assertIsArray($analytics['cards_by_status']);
        $this->assertIsArray($analytics['monthly_creations']);
        $this->assertIsArray($analytics['transaction_volume']);
        
        // Should have our test data
        $this->assertEquals(1, $analytics['cards_by_status']['active']);
    }

    /** @test */
    public function test_database_compatibility_sqlite()
    {
        // SQLite (default test database)
        $controller = new \App\Http\Controllers\Admin\VirtualCardController();
        
        $reflection = new ReflectionClass($controller);
        $method = $reflection->getMethod('getMonthFormatExpression');
        $method->setAccessible(true);
        
        $result = $method->invoke($controller);
        
        // Should use SQLite strftime function
        $this->assertStringContainsString('strftime', $result);
        $this->assertStringContainsString('%Y-%m', $result);
        $this->assertStringContainsString('created_at', $result);
    }

    /** @test */
    public function test_analytics_resilience_with_missing_columns()
    {
        $controller = new \App\Http\Controllers\Admin\VirtualCardController();
        
        // Even if transactions table has issues, analytics should not crash
        $response = $controller->analytics();
        
        $this->assertInstanceOf(\Illuminate\View\View::class, $response);
        
        $data = $response->getData();
        $analytics = $data['analytics'];
        
        // Transaction volume should be empty array (graceful handling)
        $this->assertIsArray($analytics['transaction_volume']);
        $this->assertEmpty($analytics['transaction_volume']);
    }
}