<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\VirtualCard;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;

class VirtualCardAnalyticsPageTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Migrate the database
        $this->artisan('migrate');
    }

    /** @test */
    public function test_analytics_view_renders_without_errors()
    {
        // Create some test data
        $user = User::factory()->create();
        
        VirtualCard::factory()->create([
            'user_id' => $user->id,
            'status' => 'active',
            'card_type' => 'virtual',
            'created_at' => now()->subDays(5)
        ]);
        
        VirtualCard::factory()->create([
            'user_id' => $user->id,
            'status' => 'frozen',
            'card_type' => 'physical',
            'created_at' => now()->subDays(3)  // Changed to 3 days to stay in current month
        ]);

        $controller = new \App\Http\Controllers\Admin\VirtualCardController();
        $response = $controller->analytics();
        
        // Should return a view response
        $this->assertInstanceOf(\Illuminate\View\View::class, $response);
        
        // Should have the expected view name
        $this->assertEquals('admin.virtual-cards.analytics', $response->getName());
        
        // Should contain the analytics data
        $data = $response->getData();
        $this->assertArrayHasKey('analytics', $data);
        
        $analytics = $data['analytics'];
        
        // Verify the data structure and content
        $this->assertIsArray($analytics['cards_by_status']);
        $this->assertIsArray($analytics['cards_by_type']);
        $this->assertIsArray($analytics['monthly_creations']);
        $this->assertIsArray($analytics['transaction_volume']);
        
        // Check that we have the expected data
        $this->assertEquals(1, $analytics['cards_by_status']['active']);
        $this->assertEquals(1, $analytics['cards_by_status']['frozen']);
        $this->assertEquals(1, $analytics['cards_by_type']['virtual']);
        $this->assertEquals(1, $analytics['cards_by_type']['physical']);
        
        // Should have monthly creation data for current month
        $currentMonth = now()->format('Y-m');
        $this->assertArrayHasKey($currentMonth, $analytics['monthly_creations']);
        $this->assertEquals(2, $analytics['monthly_creations'][$currentMonth]);
    }

    /** @test */
    public function test_analytics_view_renders_with_empty_data()
    {
        // Don't create any data - test empty state handling
        $controller = new \App\Http\Controllers\Admin\VirtualCardController();
        $response = $controller->analytics();
        
        // Should still render successfully
        $this->assertInstanceOf(\Illuminate\View\View::class, $response);
        
        $data = $response->getData();
        $analytics = $data['analytics'];
        
        // All arrays should be empty but still arrays
        $this->assertIsArray($analytics['cards_by_status']);
        $this->assertIsArray($analytics['cards_by_type']);
        $this->assertIsArray($analytics['monthly_creations']);
        $this->assertIsArray($analytics['transaction_volume']);
        
        $this->assertEmpty($analytics['cards_by_status']);
        $this->assertEmpty($analytics['cards_by_type']);
        $this->assertEmpty($analytics['monthly_creations']);
        $this->assertEmpty($analytics['transaction_volume']);
    }
}