<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\User;
use App\Models\Transaction;
use App\Models\Bank;
use App\Services\PaystackService;

/**
 * Transaction API Controller
 * 
 * TODO: Add comprehensive input validation
 * TODO: Add transaction limits and controls
 * TODO: Add real-time transaction monitoring
 * TODO: Add fraud detection for suspicious patterns
 * TODO: Add transaction dispute handling
 */
class TransactionController extends Controller
{
    protected $paystackService;

    public function __construct(PaystackService $paystackService)
    {
        $this->paystackService = $paystackService;
    }

    /**
     * Get user's transaction history
     * 
     * TODO: Add advanced filtering options
     * TODO: Add transaction analytics
     * TODO: Add export functionality
     */
    public function getUserTransactions(Request $request)
    {
        try {
            $user = $request->user();
            
            $query = Transaction::where('user_id', $user->id);

            // Add filters
            if ($request->has('type')) {
                $query->where('type', $request->type);
            }

            if ($request->has('category')) {
                $query->where('category', $request->category);
            }

            if ($request->has('status')) {
                $query->where('status', $request->status);
            }

            if ($request->has('date_from')) {
                $query->whereDate('created_at', '>=', $request->date_from);
            }

            if ($request->has('date_to')) {
                $query->whereDate('created_at', '<=', $request->date_to);
            }

            if ($request->has('amount_min')) {
                $query->where('amount', '>=', $request->amount_min);
            }

            if ($request->has('amount_max')) {
                $query->where('amount', '<=', $request->amount_max);
            }

            $transactions = $query->orderBy('created_at', 'desc')
                ->paginate($request->get('per_page', 20));

            return response()->json([
                'status' => 'success',
                'message' => 'Transactions retrieved successfully',
                'data' => $transactions
            ]);

        } catch (\Exception $e) {
            // TODO: Add proper error logging
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to retrieve transactions',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get single transaction details
     * 
     * TODO: Add transaction receipt generation
     * TODO: Add related transaction information
     */
    public function show(Request $request, $transactionId)
    {
        try {
            $user = $request->user();
            
            $transaction = Transaction::where('id', $transactionId)
                ->where('user_id', $user->id)
                ->first();

            if (!$transaction) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Transaction not found'
                ], 404);
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Transaction details retrieved successfully',
                'data' => $transaction
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to retrieve transaction details',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * P2P Transfer between users
     * 
     * TODO: Add recipient verification
     * TODO: Add transfer scheduling
     * TODO: Add transfer notes/memos
     */
    public function p2pTransfer(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'recipient_phone' => 'required|string|regex:/^(\+234|234|0)[789][01]\d{8}$/',
                'amount' => 'required|numeric|min:100|max:1000000',
                'description' => 'nullable|string|max:255',
                'transaction_pin' => 'required|string|size:4',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $user = $request->user();
            $wallet = $user->wallet;

            // TODO: Verify transaction PIN
            if (!Hash::check($request->transaction_pin, $user->transaction_pin)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid transaction PIN'
                ], 400);
            }

            // Find recipient
            $recipient = User::where('phone', $request->recipient_phone)
                ->where('kyc_verified', true)
                ->first();

            if (!$recipient) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Recipient not found or not verified'
                ], 404);
            }

            if ($recipient->id === $user->id) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Cannot transfer to yourself'
                ], 400);
            }

            // Check balance
            if ($wallet->balance < $request->amount) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Insufficient wallet balance'
                ], 400);
            }

            // TODO: Check daily/monthly transfer limits
            // TODO: Add fraud detection checks

            // Process transfer
            $reference = 'P2P_' . time() . '_' . $user->id;

            // Debit sender
            $wallet->decrement('balance', $request->amount);

            // Credit recipient
            $recipient->wallet->increment('balance', $request->amount);

            // Create sender transaction
            $senderTransaction = Transaction::create([
                'user_id' => $user->id,
                'type' => 'p2p_transfer',
                'category' => 'debit',
                'amount' => $request->amount,
                'description' => $request->description ?? "Transfer to {$recipient->first_name}",
                'reference' => $reference,
                'status' => 'completed',
                'metadata' => json_encode([
                    'recipient_id' => $recipient->id,
                    'recipient_phone' => $recipient->phone
                ])
            ]);

            // Create recipient transaction
            Transaction::create([
                'user_id' => $recipient->id,
                'type' => 'p2p_transfer',
                'category' => 'credit',
                'amount' => $request->amount,
                'description' => "Transfer from {$user->first_name}",
                'reference' => $reference,
                'status' => 'completed',
                'metadata' => json_encode([
                    'sender_id' => $user->id,
                    'sender_phone' => $user->phone
                ])
            ]);

            // TODO: Send transfer notifications
            // TODO: Log transfer event

            return response()->json([
                'status' => 'success',
                'message' => 'Transfer completed successfully',
                'data' => [
                    'transaction' => $senderTransaction,
                    'new_balance' => $wallet->fresh()->balance
                ]
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Transfer failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bank transfer
     * 
     * TODO: Add bank account validation
     * TODO: Add beneficiary management
     * TODO: Add transfer fees calculation
     */
    public function bankTransfer(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'bank_code' => 'required|string|exists:banks,code',
                'account_number' => 'required|string|size:10',
                'account_name' => 'required|string|max:100',
                'amount' => 'required|numeric|min:100|max:1000000',
                'narration' => 'nullable|string|max:255',
                'transaction_pin' => 'required|string|size:4',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $user = $request->user();
            $wallet = $user->wallet;

            // TODO: Verify transaction PIN
            if (!Hash::check($request->transaction_pin, $user->transaction_pin)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid transaction PIN'
                ], 400);
            }

            // TODO: Verify account name with bank
            $bankAccount = $this->paystackService->verifyAccountNumber(
                $request->account_number,
                $request->bank_code
            );

            if (!$bankAccount['success']) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Invalid bank account details',
                    'error' => $bankAccount['message']
                ], 400);
            }

            // TODO: Calculate transfer fees
            $transferFee = 25; // NGN 25 for now
            $totalAmount = $request->amount + $transferFee;

            // Check balance
            if ($wallet->balance < $totalAmount) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Insufficient wallet balance (including transfer fee)'
                ], 400);
            }

            // TODO: Check daily/monthly transfer limits

            // Create transfer recipient
            $recipientData = [
                'type' => 'nuban',
                'name' => $request->account_name,
                'account_number' => $request->account_number,
                'bank_code' => $request->bank_code,
                'currency' => 'NGN'
            ];

            $recipient = $this->paystackService->createTransferRecipient($recipientData);

            if (!$recipient['success']) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to create transfer recipient',
                    'error' => $recipient['message']
                ], 500);
            }

            // Initiate transfer
            $transferData = [
                'source' => 'balance',
                'amount' => $request->amount * 100, // Convert to kobo
                'recipient' => $recipient['data']['recipient_code'],
                'reason' => $request->narration ?? 'Bank transfer',
                'reference' => 'TRF_' . time() . '_' . $user->id
            ];

            $transfer = $this->paystackService->initiateTransfer($transferData);

            if (!$transfer['success']) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Transfer failed',
                    'error' => $transfer['message']
                ], 500);
            }

            // Debit wallet
            $wallet->decrement('balance', $totalAmount);

            // Create transaction record
            $transaction = Transaction::create([
                'user_id' => $user->id,
                'type' => 'bank_transfer',
                'category' => 'debit',
                'amount' => $request->amount,
                'fee' => $transferFee,
                'description' => "Bank transfer to {$request->account_name}",
                'reference' => $transferData['reference'],
                'provider_reference' => $transfer['data']['transfer_code'],
                'status' => 'pending',
                'metadata' => json_encode([
                    'bank_code' => $request->bank_code,
                    'account_number' => $request->account_number,
                    'account_name' => $request->account_name
                ])
            ]);

            // TODO: Send transfer notification
            // TODO: Log transfer event

            return response()->json([
                'status' => 'success',
                'message' => 'Bank transfer initiated successfully',
                'data' => [
                    'transaction' => $transaction,
                    'new_balance' => $wallet->fresh()->balance
                ]
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Bank transfer failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get transaction statistics
     * 
     * TODO: Add customizable date ranges
     * TODO: Add comparison with previous periods
     */
    public function getStatistics(Request $request)
    {
        try {
            $user = $request->user();
            
            $dateFrom = $request->get('date_from', now()->startOfMonth());
            $dateTo = $request->get('date_to', now());

            $stats = [
                'total_transactions' => Transaction::where('user_id', $user->id)
                    ->whereBetween('created_at', [$dateFrom, $dateTo])
                    ->count(),
                
                'total_spent' => Transaction::where('user_id', $user->id)
                    ->where('category', 'debit')
                    ->whereBetween('created_at', [$dateFrom, $dateTo])
                    ->sum('amount'),
                
                'total_received' => Transaction::where('user_id', $user->id)
                    ->where('category', 'credit')
                    ->whereBetween('created_at', [$dateFrom, $dateTo])
                    ->sum('amount'),
                
                'pending_transactions' => Transaction::where('user_id', $user->id)
                    ->where('status', 'pending')
                    ->count(),
                
                'failed_transactions' => Transaction::where('user_id', $user->id)
                    ->where('status', 'failed')
                    ->whereBetween('created_at', [$dateFrom, $dateTo])
                    ->count(),
            ];

            return response()->json([
                'status' => 'success',
                'message' => 'Transaction statistics retrieved successfully',
                'data' => $stats
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to retrieve transaction statistics',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}