<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Models\Transaction;
use App\Models\Service;
use App\Services\VTPassService;

class BillPaymentController extends Controller
{
    protected $vtpassService;

    public function __construct(VTPassService $vtpassService)
    {
        $this->vtpassService = $vtpassService;
    }

    /**
     * Show bill payment dashboard.
     */
    public function index()
    {
        $services = Service::active()
            ->select('category')
            ->distinct()
            ->pluck('category');
            
        return view('bills.index', compact('services'));
    }

    /**
     * Show airtime purchase form.
     */
    public function showAirtimeForm()
    {
        $networks = ['MTN', 'Glo', 'Airtel', '9mobile'];
        return view('bills.airtime', compact('networks'));
    }

    /**
     * Purchase airtime.
     */
    public function purchaseAirtime(Request $request)
    {
        $request->validate([
            'network' => 'required|in:MTN,Glo,Airtel,9mobile',
            'phone' => 'required|string|regex:/^(\+234|234|0)[789][01]\d{8}$/',
            'amount' => 'required|numeric|min:50|max:10000',
            'transaction_pin' => 'required|string',
        ]);

        return $this->processBillPayment($request, 'airtime_purchase', function ($data) {
            return $this->vtpassService->purchaseAirtime([
                'network' => $data['network'],
                'phone' => $data['phone'],
                'amount' => $data['amount'],
                'request_id' => $data['transaction_reference'],
            ]);
        });
    }

    /**
     * Show data purchase form.
     */
    public function showDataForm()
    {
        $networks = ['MTN', 'Glo', 'Airtel', '9mobile'];
        return view('bills.data', compact('networks'));
    }

    /**
     * Purchase data bundle.
     */
    public function purchaseData(Request $request)
    {
        $request->validate([
            'network' => 'required|in:MTN,Glo,Airtel,9mobile',
            'phone' => 'required|string|regex:/^(\+234|234|0)[789][01]\d{8}$/',
            'plan_code' => 'required|string',
            'amount' => 'required|numeric|min:50|max:20000',
            'transaction_pin' => 'required|string',
        ]);

        return $this->processBillPayment($request, 'data_purchase', function ($data) {
            return $this->vtpassService->purchaseData([
                'network' => $data['network'],
                'phone' => $data['phone'],
                'plan_code' => $data['plan_code'],
                'amount' => $data['amount'],
                'request_id' => $data['transaction_reference'],
            ]);
        });
    }

    /**
     * Show electricity payment form.
     */
    public function showElectricityForm()
    {
        $providers = [
            'eko-electric' => 'Eko Electricity',
            'ikeja-electric' => 'Ikeja Electric',
            'abuja-electric' => 'Abuja Electricity',
            'kano-electric' => 'Kano Electricity',
            'portharcourt-electric' => 'Port Harcourt Electricity',
        ];
        
        return view('bills.electricity', compact('providers'));
    }

    /**
     * Pay electricity bill.
     */
    public function payElectricity(Request $request)
    {
        $request->validate([
            'provider' => 'required|string',
            'meter_number' => 'required|string|min:10|max:13',
            'meter_type' => 'required|in:prepaid,postpaid',
            'amount' => 'required|numeric|min:500|max:100000',
            'phone' => 'required|string|regex:/^(\+234|234|0)[789][01]\d{8}$/',
            'transaction_pin' => 'required|string',
        ]);

        // Verify meter number first
        $verification = $this->vtpassService->verifyCustomer(
            $request->provider,
            $request->meter_number
        );

        if (!$verification['success']) {
            return back()->withErrors(['meter_number' => 'Invalid meter number or provider.']);
        }

        return $this->processBillPayment($request, 'bill_payment', function ($data) use ($verification) {
            return $this->vtpassService->payElectricity([
                'service_id' => $data['provider'],
                'meter_number' => $data['meter_number'],
                'meter_type' => $data['meter_type'],
                'amount' => $data['amount'],
                'phone' => $data['phone'],
                'request_id' => $data['transaction_reference'],
                'customer_name' => $verification['customer_name'],
            ]);
        }, [
            'provider' => $request->provider,
            'meter_number' => $request->meter_number,
            'meter_type' => $request->meter_type,
            'customer_name' => $verification['customer_name'],
        ]);
    }

    /**
     * Process bill payment transaction.
     */
    protected function processBillPayment(Request $request, string $category, callable $serviceCall, array $additionalMetadata = [])
    {
        $user = Auth::user();
        $wallet = $user->wallet;

        // Verify transaction PIN
        if (!$user->transaction_pin || !Hash::check($request->transaction_pin, $user->transaction_pin)) {
            return back()->withErrors(['transaction_pin' => 'Invalid transaction PIN.']);
        }

        $amount = $request->amount;
        $fee = $this->calculateBillPaymentFee($amount, $category);
        $totalAmount = $amount + $fee;

        // Check if user can perform transaction
        if (!$wallet->canTransact($totalAmount)) {
            return back()->with('error', 'Insufficient balance or transaction limit exceeded.');
        }

        DB::beginTransaction();
        try {
            // Create transaction
            $transaction = Transaction::create([
                'user_id' => $user->id,
                'wallet_id' => $wallet->id,
                'type' => 'debit',
                'category' => $category,
                'amount' => $amount,
                'fee' => $fee,
                'total_amount' => $totalAmount,
                'currency' => 'NGN',
                'status' => 'processing',
                'description' => $this->generateBillDescription($category, $request->all()),
                'external_reference' => $transaction->reference ?? null,
                'metadata' => array_merge([
                    'service_type' => $category,
                    'provider' => $request->network ?? $request->provider ?? 'unknown',
                    'recipient' => $request->phone ?? $request->meter_number ?? null,
                ], $additionalMetadata),
            ]);

            // Call external service
            $serviceResponse = $serviceCall(array_merge($request->all(), [
                'transaction_reference' => $transaction->reference,
            ]));

            if ($serviceResponse['success']) {
                // Debit wallet
                $wallet->debit($totalAmount);
                
                // Update transaction with external reference
                $transaction->update([
                    'external_reference' => $serviceResponse['reference'],
                    'metadata' => array_merge($transaction->metadata, [
                        'external_transaction_id' => $serviceResponse['transaction_id'] ?? null,
                        'service_status' => $serviceResponse['status'] ?? 'pending',
                    ]),
                ]);

                // Mark as completed if service delivery is immediate
                if (isset($serviceResponse['status']) && 
                    in_array($serviceResponse['status'], ['delivered', 'successful', 'completed'])) {
                    $transaction->markAsCompleted();
                }

                DB::commit();

                $successMessage = $this->generateSuccessMessage($category, $amount, $request->all());
                return redirect()->route('transactions.index')->with('success', $successMessage);
            } else {
                DB::rollBack();
                return back()->with('error', $serviceResponse['message'] ?? 'Service request failed.');
            }
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Transaction failed. Please try again.');
        }
    }

    /**
     * Calculate bill payment fee.
     */
    protected function calculateBillPaymentFee(float $amount, string $category): float
    {
        return match($category) {
            'airtime_purchase' => min($amount * 0.005, 20), // 0.5% capped at ₦20
            'data_purchase' => 0, // Free data purchases
            'bill_payment' => min($amount * 0.01, 100), // 1% capped at ₦100
            default => 0
        };
    }

    /**
     * Generate bill payment description.
     */
    protected function generateBillDescription(string $category, array $data): string
    {
        return match($category) {
            'airtime_purchase' => "₦{$data['amount']} {$data['network']} airtime for {$data['phone']}",
            'data_purchase' => "₦{$data['amount']} {$data['network']} data for {$data['phone']}",
            'bill_payment' => "₦{$data['amount']} electricity payment for {$data['meter_number']}",
            default => "Bill payment: ₦{$data['amount']}"
        };
    }

    /**
     * Generate success message.
     */
    protected function generateSuccessMessage(string $category, float $amount, array $data): string
    {
        return match($category) {
            'airtime_purchase' => "Successfully purchased ₦{$amount} {$data['network']} airtime for {$data['phone']}",
            'data_purchase' => "Successfully purchased ₦{$amount} {$data['network']} data for {$data['phone']}",
            'bill_payment' => "Successfully paid ₦{$amount} electricity bill for meter {$data['meter_number']}",
            default => "Bill payment of ₦{$amount} completed successfully"
        };
    }
}