<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use App\Services\WebhookVerificationService;
use Symfony\Component\HttpFoundation\Response;

class VerifyWebhookSignature
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next, string $provider = null): Response
    {
        // Skip verification in development/testing
        if (app()->environment(['local', 'testing'])) {
            return $next($request);
        }

        $provider = $provider ?: $this->detectProvider($request);
        $verified = false;

        switch ($provider) {
            case 'paystack':
                $verified = WebhookVerificationService::verifyPaystackSignature($request);
                break;
                
            case 'monnify':
                $verified = WebhookVerificationService::verifyMonnifySignature($request);
                break;
                
            case 'anchor':
                $verified = WebhookVerificationService::verifyAnchorSignature($request);
                break;
                
            case 'vtpass':
                $verified = WebhookVerificationService::verifyVTPassSignature($request);
                break;
                
            default:
                // Log unknown provider
                WebhookVerificationService::logVerificationAttempt(
                    $provider ?: 'unknown',
                    false,
                    $request,
                    ['error' => 'Unknown webhook provider']
                );
                return response()->json([
                    'success' => false,
                    'message' => 'Unknown webhook provider'
                ], 400);
        }

        // Log verification attempt
        WebhookVerificationService::logVerificationAttempt(
            $provider,
            $verified,
            $request
        );

        if (!$verified) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid webhook signature'
            ], 401);
        }

        return $next($request);
    }

    /**
     * Detect webhook provider from request path.
     */
    private function detectProvider(Request $request): ?string
    {
        $path = $request->path();
        
        if (str_contains($path, 'paystack')) {
            return 'paystack';
        }
        
        if (str_contains($path, 'monnify')) {
            return 'monnify';
        }
        
        if (str_contains($path, 'anchor')) {
            return 'anchor';
        }
        
        if (str_contains($path, 'vtpass')) {
            return 'vtpass';
        }
        
        return null;
    }
}