<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

// TODO: Add soft deletes for user records
// TODO: Implement user verification status tracking
// TODO: Add user profile completion tracking
// TODO: Consider adding user tiers/levels for transaction limits

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'first_name',
        'last_name',
        'email',
        'phone',
        'password',
        'date_of_birth',
        'gender',
        'address',
        'city',
        'state',
        'country',
        'profile_photo_path',
        'transaction_pin',
        'is_active',
        'last_login_at',
        'last_login_ip',
        // TODO: Add profile completion percentage field
        // TODO: Add user tier/level field for transaction limits
        // TODO: Add referral code field
        // TODO: Add two-factor authentication fields
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'transaction_pin',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'phone_verified_at' => 'datetime',
            'password' => 'hashed',
            'date_of_birth' => 'date',
            'transaction_pin_set_at' => 'datetime',
            'last_login_at' => 'datetime',
            'is_active' => 'boolean',
        ];
    }

    /**
     * Get the user's full name.
     */
    public function getFullNameAttribute(): string
    {
        return $this->first_name . ' ' . $this->last_name;
    }

    /**
     * Check if user has verified phone.
     */
    public function hasVerifiedPhone(): bool
    {
        return !is_null($this->phone_verified_at);
    }

    /**
     * Check if user has completed KYC verification.
     * 
     * TODO: Enhance KYC verification logic
     * TODO: Check for expiration of KYC documents
     * TODO: Consider different KYC levels (Tier 1, 2, 3)
     */
    public function hasCompletedKyc(): bool
    {
        return $this->kycVerifications()
            ->where('status', 'verified')
            ->where('type', 'bvn') // TODO: Consider multiple verification types
            ->exists();
        
        // TODO: Add check for document expiration
        // TODO: Add check for different verification levels
        // TODO: Add cache for frequently accessed KYC status
    }

    /**
     * Get the user's primary wallet.
     */
    public function wallet(): HasOne
    {
        return $this->hasOne(Wallet::class)->where('currency', 'NGN');
    }

    /**
     * Get all user's wallets.
     */
    public function wallets(): HasMany
    {
        return $this->hasMany(Wallet::class);
    }

    /**
     * Get all user's transactions.
     */
    public function transactions(): HasMany
    {
        return $this->hasMany(Transaction::class);
    }

    /**
     * Get all user's virtual accounts.
     */
    public function virtualAccounts(): HasMany
    {
        return $this->hasMany(VirtualAccount::class);
    }

    /**
     * Get all user's KYC verifications.
     */
    public function kycVerifications(): HasMany
    {
        return $this->hasMany(KycVerification::class);
    }

    /**
     * Get all user's virtual cards.
     */
    public function virtualCards(): HasMany
    {
        return $this->hasMany(VirtualCard::class);
    }
    
    // TODO: Add relationship for user notifications
    // TODO: Add relationship for user referrals
    // TODO: Add relationship for user activity logs
    // TODO: Add relationship for user devices/sessions
    // TODO: Add relationship for user transaction limits
}