<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

// TODO: Add audit logging for wallet balance changes
// TODO: Implement wallet freezing/unfreezing functionality
// TODO: Add multi-signature support for high-value wallets
// TODO: Add wallet backup and recovery mechanisms

class Wallet extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'currency',
        'balance',
        'locked_balance',
        'daily_transaction_limit',
        'monthly_transaction_limit',
        'daily_spent',
        'monthly_spent',
        'daily_spent_reset_date',
        'monthly_spent_reset_date',
        'is_active',
        // TODO: Add wallet type (personal, business, escrow)
        // TODO: Add wallet nickname/name
        // TODO: Add wallet description
        // TODO: Add last_transaction_at timestamp
    ];

    protected function casts(): array
    {
        return [
            'balance' => 'decimal:2',
            'locked_balance' => 'decimal:2',
            'daily_transaction_limit' => 'decimal:2',
            'monthly_transaction_limit' => 'decimal:2',
            'daily_spent' => 'decimal:2',
            'monthly_spent' => 'decimal:2',
            'daily_spent_reset_date' => 'date',
            'monthly_spent_reset_date' => 'date',
            'is_active' => 'boolean',
        ];
    }

    /**
     * Get the user that owns the wallet.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get all transactions for this wallet.
     */
    public function transactions(): HasMany
    {
        return $this->hasMany(Transaction::class);
    }

    /**
     * Get the available balance (total balance minus locked balance).
     */
    public function getAvailableBalanceAttribute(): float
    {
        return $this->balance - $this->locked_balance;
    }

    /**
     * Get remaining daily transaction limit.
     */
    public function getRemainingDailyLimitAttribute(): float
    {
        return max(0, $this->daily_transaction_limit - $this->daily_spent);
    }

    /**
     * Get remaining monthly transaction limit.
     */
    public function getRemainingMonthlyLimitAttribute(): float
    {
        return max(0, $this->monthly_transaction_limit - $this->monthly_spent);
    }

    /**
     * Check if wallet can perform a transaction of given amount.
     * 
     * TODO: Add fraud detection checks
     * TODO: Add velocity checks (number of transactions in time period)
     * TODO: Add location-based restrictions
     */
    public function canTransact(float $amount): bool
    {
        // TODO: Add reset logic for daily/monthly limits if dates have passed
        // TODO: Add transaction pattern analysis
        // TODO: Add user behavior scoring
        
        return $this->is_active &&
               $this->available_balance >= $amount &&
               $this->remaining_daily_limit >= $amount &&
               $this->remaining_monthly_limit >= $amount;
    }

    /**
     * Lock balance for pending transaction.
     */
    public function lockBalance(float $amount): bool
    {
        if ($this->available_balance >= $amount) {
            $this->increment('locked_balance', $amount);
            return true;
        }
        return false;
    }

    /**
     * Unlock previously locked balance.
     */
    public function unlockBalance(float $amount): void
    {
        $this->decrement('locked_balance', min($this->locked_balance, $amount));
    }

    /**
     * Credit wallet balance.
     */
    public function credit(float $amount): void
    {
        $this->increment('balance', $amount);
    }

    /**
     * Debit wallet balance.
     * 
     * TODO: Add transaction atomicity with database transactions
     * TODO: Add real-time balance validation
     * TODO: Implement transaction rollback mechanisms
     */
    public function debit(float $amount): bool
    {
        if ($this->balance >= $amount) {
            // TODO: Wrap in database transaction for atomicity
            // TODO: Add balance change logging
            // TODO: Trigger real-time balance update events
            
            $this->decrement('balance', $amount);
            $this->increment('daily_spent', $amount);
            $this->increment('monthly_spent', $amount);
            return true;
        }
        return false;
    }
    
    // TODO: Add method for wallet statements generation
    // TODO: Add method for balance history tracking
    // TODO: Add method for transaction categorization
    // TODO: Add method for spending analytics
}