<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class MonnifyService
{
    protected $apiKey;
    protected $secretKey;
    protected $contractCode;
    protected $baseUrl;

    public function __construct()
    {
        $this->apiKey = config('services.monnify.api_key');
        $this->secretKey = config('services.monnify.secret_key');
        $this->contractCode = config('services.monnify.contract_code');
        $this->baseUrl = config('services.monnify.base_url');
    }

    /**
     * Create virtual account for user.
     */
    public function createVirtualAccount(array $data): array
    {
        try {
            $accessToken = $this->getAccessToken();

            $response = $this->makeApiCall('/api/v1/bank-transfer/reserved-accounts', [
                'accountReference' => $data['account_reference'],
                'accountName' => $data['account_name'],
                'currencyCode' => $data['currency'] ?? 'NGN',
                'contractCode' => $this->contractCode,
                'customerEmail' => $data['email'],
                'customerName' => $data['customer_name'],
                'getAllAvailableBanks' => true,
            ], $accessToken);

            if ($response['requestSuccessful']) {
                $accounts = $response['responseBody']['accounts'] ?? [];
                
                return [
                    'success' => true,
                    'accounts' => array_map(function ($account) {
                        return [
                            'account_number' => $account['accountNumber'],
                            'account_name' => $account['accountName'],
                            'bank_code' => $account['bankCode'],
                            'bank_name' => $account['bankName'],
                        ];
                    }, $accounts),
                    'reference' => $response['responseBody']['accountReference'],
                ];
            }

            return [
                'success' => false,
                'message' => $response['responseMessage'] ?? 'Virtual account creation failed',
            ];
        } catch (\Exception $e) {
            Log::error('Monnify virtual account creation failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Virtual account creation failed. Please try again.',
            ];
        }
    }

    /**
     * Get virtual account details.
     */
    public function getVirtualAccount(string $accountReference): array
    {
        try {
            $accessToken = $this->getAccessToken();

            $response = $this->makeApiCall(
                "/api/v1/bank-transfer/reserved-accounts/{$accountReference}",
                [],
                $accessToken,
                'GET'
            );

            if ($response['requestSuccessful']) {
                $data = $response['responseBody'];
                
                return [
                    'success' => true,
                    'account_reference' => $data['accountReference'],
                    'account_name' => $data['accountName'],
                    'accounts' => array_map(function ($account) {
                        return [
                            'account_number' => $account['accountNumber'],
                            'bank_code' => $account['bankCode'],
                            'bank_name' => $account['bankName'],
                        ];
                    }, $data['accounts']),
                ];
            }

            return [
                'success' => false,
                'message' => 'Virtual account not found',
            ];
        } catch (\Exception $e) {
            Log::error('Monnify get virtual account failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Could not fetch virtual account details.',
            ];
        }
    }

    /**
     * Initiate single transfer.
     */
    public function initiateSingleTransfer(array $data): array
    {
        try {
            $accessToken = $this->getAccessToken();

            $response = $this->makeApiCall('/api/v1/disbursements/single', [
                'amount' => $data['amount'],
                'reference' => $data['reference'],
                'narration' => $data['narration'] ?? 'Transfer',
                'destinationBankCode' => $data['bank_code'],
                'destinationAccountNumber' => $data['account_number'],
                'currency' => $data['currency'] ?? 'NGN',
                'sourceAccountNumber' => $data['source_account'] ?? null,
            ], $accessToken);

            if ($response['requestSuccessful']) {
                return [
                    'success' => true,
                    'reference' => $response['responseBody']['reference'],
                    'status' => $response['responseBody']['status'],
                    'fee' => $response['responseBody']['fee'],
                ];
            }

            return [
                'success' => false,
                'message' => $response['responseMessage'] ?? 'Transfer initiation failed',
            ];
        } catch (\Exception $e) {
            Log::error('Monnify transfer initiation failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Transfer initiation failed.',
            ];
        }
    }

    /**
     * Get transaction status.
     */
    public function getTransactionStatus(string $reference): array
    {
        try {
            $accessToken = $this->getAccessToken();

            $response = $this->makeApiCall(
                "/api/v1/transactions/{$reference}",
                [],
                $accessToken,
                'GET'
            );

            if ($response['requestSuccessful']) {
                $data = $response['responseBody'];
                
                return [
                    'success' => true,
                    'status' => $data['paymentStatus'],
                    'amount' => $data['amountPaid'],
                    'reference' => $data['transactionReference'],
                    'customer' => [
                        'name' => $data['customer']['name'],
                        'email' => $data['customer']['email'],
                    ],
                ];
            }

            return [
                'success' => false,
                'message' => 'Transaction not found',
            ];
        } catch (\Exception $e) {
            Log::error('Monnify get transaction status failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Could not fetch transaction status.',
            ];
        }
    }

    /**
     * Get access token.
     */
    protected function getAccessToken(): string
    {
        if (config('app.env') === 'local') {
            return 'mock_access_token';
        }

        $credentials = base64_encode($this->apiKey . ':' . $this->secretKey);

        $response = Http::withHeaders([
            'Authorization' => 'Basic ' . $credentials,
            'Content-Type' => 'application/json',
        ])->post($this->baseUrl . '/api/v1/auth/login');

        if ($response->successful()) {
            $data = $response->json();
            return $data['responseBody']['accessToken'];
        }

        throw new \Exception('Failed to get Monnify access token');
    }

    /**
     * Make API call to Monnify.
     */
    protected function makeApiCall(string $endpoint, array $data = [], string $accessToken = '', string $method = 'POST'): array
    {
        if (config('app.env') === 'local') {
            // Return mock response in local environment
            Log::info("Monnify API Call (MOCK): {$method} {$endpoint}", $data);
            
            // Return mock successful response based on endpoint
            if (str_contains($endpoint, 'reserved-accounts')) {
                return [
                    'requestSuccessful' => true,
                    'responseBody' => [
                        'accountReference' => $data['accountReference'] ?? 'mock_ref',
                        'accountName' => $data['accountName'] ?? 'Mock Account',
                        'accounts' => [
                            [
                                'accountNumber' => '1234567890',
                                'accountName' => $data['accountName'] ?? 'Mock Account',
                                'bankCode' => '232',
                                'bankName' => 'Sterling Bank',
                            ]
                        ]
                    ]
                ];
            }
            
            return ['requestSuccessful' => true, 'responseBody' => ['mock' => true]];
        }

        $http = Http::withHeaders([
            'Authorization' => 'Bearer ' . $accessToken,
            'Content-Type' => 'application/json',
        ]);

        $response = $method === 'GET' 
            ? $http->get($this->baseUrl . $endpoint)
            : $http->post($this->baseUrl . $endpoint, $data);

        if ($response->successful()) {
            return $response->json();
        }

        throw new \Exception('Monnify API call failed: ' . $response->body());
    }
}