<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class DojahService
{
    protected $appId;
    protected $secretKey;
    protected $baseUrl;

    public function __construct()
    {
        $this->appId = config('services.dojah.app_id');
        $this->secretKey = config('services.dojah.secret_key');
        $this->baseUrl = config('services.dojah.base_url');
    }

    /**
     * Verify BVN with personal details.
     */
    public function verifyBvn(array $data): array
    {
        try {
            // This is a stub implementation
            // In production, you would make actual API calls to Dojah
            
            $response = $this->makeApiCall('/api/v1/kyc/bvn', [
                'bvn' => $data['bvn'],
                'first_name' => $data['first_name'],
                'last_name' => $data['last_name'],
                'date_of_birth' => $data['date_of_birth'],
            ]);

            // Simulate successful verification for demo purposes
            return [
                'status' => 'verified',
                'reference' => 'DOJAH_' . uniqid(),
                'message' => 'BVN verification successful',
                'data' => [
                    'bvn' => $data['bvn'],
                    'first_name' => $data['first_name'],
                    'last_name' => $data['last_name'],
                    'date_of_birth' => $data['date_of_birth'],
                    'phone' => '0801234567*', // Masked
                    'verified_at' => now()->toISOString(),
                ]
            ];
        } catch (\Exception $e) {
            Log::error('Dojah BVN verification failed: ' . $e->getMessage());
            
            return [
                'status' => 'failed',
                'reference' => null,
                'message' => 'BVN verification failed. Please check your details and try again.',
                'data' => []
            ];
        }
    }

    /**
     * Verify phone number.
     */
    public function verifyPhone(string $phone): array
    {
        try {
            $response = $this->makeApiCall('/api/v1/kyc/phone', [
                'phone_number' => $phone,
            ]);

            return [
                'status' => 'verified',
                'reference' => 'DOJAH_PHONE_' . uniqid(),
                'message' => 'Phone verification successful',
                'data' => [
                    'phone' => $phone,
                    'network' => 'MTN', // Example
                    'verified_at' => now()->toISOString(),
                ]
            ];
        } catch (\Exception $e) {
            Log::error('Dojah phone verification failed: ' . $e->getMessage());
            
            return [
                'status' => 'failed',
                'reference' => null,
                'message' => 'Phone verification failed.',
                'data' => []
            ];
        }
    }

    /**
     * Get BVN details by BVN number.
     */
    public function getBvnDetails(string $bvn): array
    {
        try {
            $response = $this->makeApiCall('/api/v1/kyc/bvn/full', [
                'bvn' => $bvn,
            ]);

            return [
                'status' => 'success',
                'data' => [
                    'bvn' => $bvn,
                    'first_name' => 'John',
                    'last_name' => 'Doe',
                    'date_of_birth' => '1990-01-01',
                    'phone' => '0801234567*',
                    'email' => 'j***@example.com',
                    'gender' => 'Male',
                ]
            ];
        } catch (\Exception $e) {
            Log::error('Dojah BVN details fetch failed: ' . $e->getMessage());
            
            return [
                'status' => 'failed',
                'message' => 'Could not fetch BVN details.',
                'data' => []
            ];
        }
    }

    /**
     * Make API call to Dojah.
     */
    protected function makeApiCall(string $endpoint, array $data): array
    {
        if (config('app.env') === 'local') {
            // Return mock response in local environment
            Log::info("Dojah API Call (MOCK): {$endpoint}", $data);
            return ['mock' => true];
        }

        $response = Http::withHeaders([
            'Authorization' => $this->secretKey,
            'AppId' => $this->appId,
            'Content-Type' => 'application/json',
        ])->post($this->baseUrl . $endpoint, $data);

        if ($response->successful()) {
            return $response->json();
        }

        throw new \Exception('Dojah API call failed: ' . $response->body());
    }
}