<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class AirtimeOrder extends Model
{
    use HasFactory;

    protected $fillable = [
        'reference',
        'user_id',
        'transaction_id',
        'airtime_provider_id',
        'service_type',
        'network',
        'phone_number',
        'amount',
        'discount_amount',
        'margin_amount',
        'provider_cost',
        'data_plan_id',
        'data_plan_name',
        'data_volume_mb',
        'validity_days',
        'status',
        'provider_reference',
        'provider_response',
        'failure_reason',
        'processed_at',
        'completed_at',
        'api_endpoint_used',
        'retry_count',
        'retry_log',
    ];

    protected function casts(): array
    {
        return [
            'amount' => 'decimal:2',
            'discount_amount' => 'decimal:2',
            'margin_amount' => 'decimal:2',
            'provider_cost' => 'decimal:2',
            'provider_response' => 'array',
            'retry_log' => 'array',
            'processed_at' => 'datetime',
            'completed_at' => 'datetime',
            'retry_count' => 'integer',
            'data_volume_mb' => 'integer',
            'validity_days' => 'integer',
        ];
    }

    /**
     * Get the user that owns the order.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the transaction associated with this order.
     */
    public function transaction(): BelongsTo
    {
        return $this->belongsTo(Transaction::class);
    }

    /**
     * Get the provider used for this order.
     */
    public function provider(): BelongsTo
    {
        return $this->belongsTo(AirtimeProvider::class, 'airtime_provider_id');
    }

    /**
     * Scope to get orders by service type.
     */
    public function scopeByServiceType($query, $serviceType)
    {
        return $query->where('service_type', $serviceType);
    }

    /**
     * Scope to get orders by network.
     */
    public function scopeByNetwork($query, $network)
    {
        return $query->where('network', $network);
    }

    /**
     * Scope to get orders by status.
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope to get recent orders.
     */
    public function scopeRecent($query, $days = 30)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    /**
     * Check if the order is for airtime.
     */
    public function isAirtime(): bool
    {
        return $this->service_type === 'airtime';
    }

    /**
     * Check if the order is for data.
     */
    public function isData(): bool
    {
        return $this->service_type === 'data';
    }

    /**
     * Check if the order is completed.
     */
    public function isCompleted(): bool
    {
        return $this->status === 'completed';
    }

    /**
     * Check if the order has failed.
     */
    public function isFailed(): bool
    {
        return $this->status === 'failed';
    }

    /**
     * Check if the order is pending.
     */
    public function isPending(): bool
    {
        return in_array($this->status, ['pending', 'processing']);
    }

    /**
     * Get the formatted phone number.
     */
    public function getFormattedPhoneAttribute(): string
    {
        $phone = $this->phone_number;
        if (strlen($phone) === 11 && substr($phone, 0, 1) === '0') {
            return '+234' . substr($phone, 1);
        }
        return $phone;
    }

    /**
     * Get the network display name.
     */
    public function getNetworkDisplayNameAttribute(): string
    {
        $networks = [
            'mtn' => 'MTN',
            'glo' => 'Glo',
            'airtel' => 'Airtel',
            '9mobile' => '9mobile',
        ];

        return $networks[$this->network] ?? strtoupper($this->network);
    }

    /**
     * Get the status badge color.
     */
    public function getStatusColorAttribute(): string
    {
        return match($this->status) {
            'completed' => 'green',
            'failed', 'cancelled' => 'red',
            'processing' => 'yellow',
            'pending' => 'blue',
            default => 'gray',
        };
    }
}
