<?php

namespace App\Services\Airtime\Providers;

use App\Services\Airtime\BaseAirtimeProvider;

class CustomProvider extends BaseAirtimeProvider
{
    protected array $endpoints = [];
    protected array $customHeaders = [];

    /**
     * Configure Custom provider.
     */
    public function configure(array $credentials): void
    {
        parent::configure($credentials);
        
        $this->endpoints = $credentials['endpoints'] ?? [];
        $this->customHeaders = $credentials['headers'] ?? [];
        
        if (empty($this->baseUrl)) {
            throw new \InvalidArgumentException('Base URL is required for custom provider');
        }
    }

    /**
     * Purchase airtime via custom provider.
     */
    public function purchaseAirtime(array $params): array
    {
        $this->validateParameters($params, ['phone_number', 'amount', 'network']);

        $endpoint = $this->endpoints['airtime'] ?? '/airtime';
        $requestId = $this->generateReference('CUSTOM_AIR');

        $data = [
            'reference' => $requestId,
            'phone_number' => $this->formatPhoneNumber($params['phone_number']),
            'amount' => $params['amount'],
            'network' => $params['network'],
        ];

        $response = $this->makeRequest('POST', $endpoint, $data, $this->customHeaders);

        if (!$response['success']) {
            return [
                'success' => false,
                'message' => 'Network error: ' . ($response['error'] ?? 'Unknown error'),
                'reference' => $requestId,
                'provider_response' => $response,
            ];
        }

        $responseData = $response['data'];
        $success = $this->determineSuccess($responseData);

        return [
            'success' => $success,
            'message' => $this->extractMessage($responseData, $success),
            'reference' => $requestId,
            'provider_reference' => $this->extractProviderReference($responseData, $requestId),
            'amount' => $params['amount'],
            'phone_number' => $params['phone_number'],
            'network' => $params['network'],
            'provider_response' => $response,
        ];
    }

    /**
     * Purchase data bundle via custom provider.
     */
    public function purchaseData(array $params): array
    {
        $this->validateParameters($params, ['phone_number', 'data_plan', 'network']);

        $endpoint = $this->endpoints['data'] ?? '/data';
        $requestId = $this->generateReference('CUSTOM_DATA');

        $data = [
            'reference' => $requestId,
            'phone_number' => $this->formatPhoneNumber($params['phone_number']),
            'data_plan' => $params['data_plan'],
            'network' => $params['network'],
            'amount' => $params['amount'] ?? null,
        ];

        $response = $this->makeRequest('POST', $endpoint, $data, $this->customHeaders);

        if (!$response['success']) {
            return [
                'success' => false,
                'message' => 'Network error: ' . ($response['error'] ?? 'Unknown error'),
                'reference' => $requestId,
                'provider_response' => $response,
            ];
        }

        $responseData = $response['data'];
        $success = $this->determineSuccess($responseData);

        return [
            'success' => $success,
            'message' => $this->extractMessage($responseData, $success),
            'reference' => $requestId,
            'provider_reference' => $this->extractProviderReference($responseData, $requestId),
            'amount' => $responseData['amount'] ?? $params['amount'],
            'phone_number' => $params['phone_number'],
            'network' => $params['network'],
            'data_plan' => $params['data_plan'],
            'provider_response' => $response,
        ];
    }

    /**
     * Get available data plans for a network.
     */
    public function getDataPlans(string $network): array
    {
        $endpoint = $this->endpoints['plans'] ?? '/data-plans';
        
        $response = $this->makeRequest('GET', $endpoint, ['network' => $network], $this->customHeaders);

        if (!$response['success']) {
            return [];
        }

        $responseData = $response['data'];
        
        // Try different common response structures
        $plans = $responseData['plans'] ?? $responseData['data'] ?? $responseData ?? [];
        
        if (!is_array($plans)) {
            return [];
        }

        return array_map(function($plan) {
            return [
                'id' => $plan['id'] ?? $plan['plan_id'] ?? $plan['code'] ?? '',
                'name' => $plan['name'] ?? $plan['plan_name'] ?? $plan['description'] ?? '',
                'amount' => $plan['amount'] ?? $plan['price'] ?? 0,
                'validity' => $plan['validity'] ?? $plan['duration'] ?? null,
            ];
        }, $plans);
    }

    /**
     * Check transaction status.
     */
    public function checkTransactionStatus(string $reference): array
    {
        $endpoint = $this->endpoints['status'] ?? '/status';
        
        $response = $this->makeRequest('GET', $endpoint, ['reference' => $reference], $this->customHeaders);

        if (!$response['success']) {
            return [
                'success' => false,
                'status' => 'unknown',
                'message' => 'Failed to query status',
            ];
        }

        $responseData = $response['data'];
        $status = $this->mapStatus($responseData['status'] ?? 'pending');

        return [
            'success' => true,
            'status' => $status,
            'message' => $responseData['message'] ?? 'Status checked',
            'provider_response' => $response,
        ];
    }

    /**
     * Get account balance.
     */
    public function getBalance(): array
    {
        $endpoint = $this->endpoints['balance'] ?? '/balance';
        
        $response = $this->makeRequest('GET', $endpoint, [], $this->customHeaders);

        if (!$response['success']) {
            return [
                'success' => false,
                'balance' => 0,
                'message' => 'Failed to get balance',
            ];
        }

        $responseData = $response['data'];
        
        return [
            'success' => true,
            'balance' => $responseData['balance'] ?? $responseData['amount'] ?? 0,
            'currency' => $responseData['currency'] ?? 'NGN',
            'message' => 'Balance retrieved successfully',
        ];
    }

    /**
     * Test connection to custom provider.
     */
    public function testConnection(): array
    {
        try {
            $balanceResponse = $this->getBalance();
            
            return [
                'success' => $balanceResponse['success'],
                'message' => $balanceResponse['success'] ? 'Connection successful' : 'Connection failed',
                'response_time' => 0,
                'provider_response' => $balanceResponse,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Connection test failed: ' . $e->getMessage(),
                'response_time' => 0,
            ];
        }
    }

    /**
     * Verify phone number and detect network.
     */
    public function verifyPhoneNumber(string $phoneNumber): array
    {
        $network = $this->detectNetwork($phoneNumber);
        
        return [
            'success' => !is_null($network),
            'phone_number' => $this->formatPhoneNumber($phoneNumber),
            'network' => $network,
            'message' => $network ? 'Phone number verified' : 'Network not detected',
        ];
    }

    /**
     * Determine if the response indicates success.
     */
    protected function determineSuccess(array $responseData): bool
    {
        // Common success indicators
        if (isset($responseData['success'])) {
            return (bool) $responseData['success'];
        }

        if (isset($responseData['status'])) {
            $status = strtolower($responseData['status']);
            return in_array($status, ['success', 'successful', 'completed', 'ok', 'true']);
        }

        if (isset($responseData['code'])) {
            return in_array($responseData['code'], ['200', '000', 0, 200]);
        }

        // Default to false if we can't determine
        return false;
    }

    /**
     * Extract message from response.
     */
    protected function extractMessage(array $responseData, bool $success): string
    {
        $message = $responseData['message'] ?? 
                  $responseData['description'] ?? 
                  $responseData['response_description'] ?? 
                  ($success ? 'Transaction successful' : 'Transaction failed');

        return $message;
    }

    /**
     * Extract provider reference from response.
     */
    protected function extractProviderReference(array $responseData, string $fallback): string
    {
        return $responseData['reference'] ?? 
               $responseData['provider_reference'] ?? 
               $responseData['transaction_id'] ?? 
               $responseData['id'] ?? 
               $fallback;
    }

    /**
     * Map provider status to standard status.
     */
    protected function mapStatus(string $providerStatus): string
    {
        $status = strtolower($providerStatus);
        
        return match($status) {
            'success', 'successful', 'completed', 'delivered' => 'completed',
            'failed', 'error', 'cancelled' => 'failed',
            'processing', 'pending' => 'pending',
            default => 'pending',
        };
    }

    /**
     * Get custom headers including authentication.
     */
    protected function getDefaultHeaders(): array
    {
        return array_merge(parent::getDefaultHeaders(), $this->customHeaders);
    }
}