@extends('layouts.mobile')

@section('title', 'Pay Electricity Bill')

@section('content')
<div class="container-fluid py-6">
    <!-- Header -->
    <div class="flex justify-between items-center mb-6">
        <div class="flex items-center">
            <a href="{{ route('bills.index') }}" class="w-10 h-10 bg-dark-800 rounded-xl flex items-center justify-center mr-4 hover:bg-dark-700 transition-colors">
                <i class="fas fa-arrow-left text-gray-300"></i>
            </a>
            <div>
                <h1 class="text-xl font-semibold text-white">Pay Electricity Bill</h1>
                <p class="text-sm text-gray-400">Pay your electricity bills instantly</p>
            </div>
        </div>
    </div>

    @if($errors->any())
        <div class="bg-red-500/20 border border-red-500 rounded-lg p-4 mb-6">
            <div class="flex items-center">
                <i class="fas fa-exclamation-triangle text-red-400 mr-2"></i>
                <span class="text-red-400 text-sm">Please fix the following errors:</span>
            </div>
            <ul class="mt-2 text-red-400 text-sm">
                @foreach ($errors->all() as $error)
                    <li class="ml-4">• {{ $error }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    <form method="POST" action="{{ route('bills.electricity') }}" class="space-y-6">
        @csrf

        <!-- Provider Selection -->
        <div class="space-y-3">
            <label class="text-white font-medium">Select Provider</label>
            <div class="space-y-3">
                @foreach($providers as $key => $provider)
                    <input type="radio" id="provider_{{ $key }}" name="provider" value="{{ $key }}" class="hidden provider-radio" 
                           {{ old('provider') == $key ? 'checked' : '' }}>
                    <label for="provider_{{ $key }}" class="provider-option card-gradient border border-gray-600 rounded-xl p-4 cursor-pointer transition-all duration-200 hover:border-primary-500 block">
                        <div class="flex items-center">
                            <div class="w-12 h-12 bg-gradient-to-br from-yellow-500 to-orange-500 rounded-full flex items-center justify-center mr-4">
                                <i class="fas fa-bolt text-white text-lg"></i>
                            </div>
                            <div>
                                <p class="text-white font-medium">{{ $provider }}</p>
                                <p class="text-gray-400 text-sm">Electricity provider</p>
                            </div>
                        </div>
                    </label>
                @endforeach
            </div>
        </div>

        <!-- Meter Type -->
        <div class="space-y-3">
            <label class="text-white font-medium">Meter Type</label>
            <div class="grid grid-cols-2 gap-3">
                <input type="radio" id="meter_prepaid" name="meter_type" value="prepaid" class="hidden meter-type-radio" 
                       {{ old('meter_type', 'prepaid') == 'prepaid' ? 'checked' : '' }}>
                <label for="meter_prepaid" class="meter-type-option card-gradient border border-gray-600 rounded-xl p-4 text-center cursor-pointer transition-all duration-200 hover:border-primary-500">
                    <div class="flex flex-col items-center">
                        <div class="w-12 h-12 bg-green-500 rounded-full flex items-center justify-center mb-2">
                            <i class="fas fa-credit-card text-white"></i>
                        </div>
                        <span class="text-white font-medium">Prepaid</span>
                        <span class="text-gray-400 text-xs">Pay before use</span>
                    </div>
                </label>

                <input type="radio" id="meter_postpaid" name="meter_type" value="postpaid" class="hidden meter-type-radio"
                       {{ old('meter_type') == 'postpaid' ? 'checked' : '' }}>
                <label for="meter_postpaid" class="meter-type-option card-gradient border border-gray-600 rounded-xl p-4 text-center cursor-pointer transition-all duration-200 hover:border-primary-500">
                    <div class="flex flex-col items-center">
                        <div class="w-12 h-12 bg-blue-500 rounded-full flex items-center justify-center mb-2">
                            <i class="fas fa-file-invoice text-white"></i>
                        </div>
                        <span class="text-white font-medium">Postpaid</span>
                        <span class="text-gray-400 text-xs">Pay after use</span>
                    </div>
                </label>
            </div>
        </div>

        <!-- Meter Number -->
        <div class="space-y-2">
            <label for="meter_number" class="text-white font-medium">Meter Number</label>
            <div class="relative">
                <input type="text" 
                       id="meter_number" 
                       name="meter_number" 
                       value="{{ old('meter_number') }}"
                       placeholder="Enter meter number" 
                       class="w-full p-4 bg-dark-800 border border-gray-600 rounded-xl text-white placeholder-gray-400 focus:border-primary-500 focus:outline-none"
                       minlength="10" 
                       maxlength="13"
                       required>
                <button type="button" id="verify-meter" class="absolute right-3 top-1/2 transform -translate-y-1/2 text-primary-500 hover:text-primary-400 text-sm font-medium" style="display: none;">
                    Verify
                </button>
            </div>
            <p class="text-xs text-gray-400">Enter 10-13 digit meter number</p>
            
            <!-- Meter verification result -->
            <div id="meter-verification" style="display: none;">
                <div class="bg-green-500/20 border border-green-500 rounded-lg p-3">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-400 mr-2"></i>
                        <span class="text-green-400 text-sm font-medium">Meter verified successfully</span>
                    </div>
                    <p class="text-green-400 text-sm mt-1">Customer: <span id="customer-name"></span></p>
                </div>
            </div>
        </div>

        <!-- Phone Number -->
        <div class="space-y-2">
            <label for="phone" class="text-white font-medium">Phone Number</label>
            <input type="tel" 
                   id="phone" 
                   name="phone" 
                   value="{{ old('phone') }}"
                   placeholder="08012345678" 
                   class="w-full p-4 bg-dark-800 border border-gray-600 rounded-xl text-white placeholder-gray-400 focus:border-primary-500 focus:outline-none"
                   maxlength="11" 
                   pattern="[0-9]{11}"
                   required>
            <p class="text-xs text-gray-400">Phone number for transaction confirmation</p>
        </div>

        <!-- Amount -->
        <div class="space-y-3">
            <label class="text-white font-medium">Select Amount</label>
            <div class="grid grid-cols-3 gap-3">
                @foreach([1000, 2000, 5000, 10000, 15000, 20000] as $amount)
                    <input type="radio" id="amount_{{ $amount }}" name="amount_preset" value="{{ $amount }}" class="hidden amount-radio"
                           {{ old('amount') == $amount ? 'checked' : '' }}>
                    <label for="amount_{{ $amount }}" class="amount-option card-gradient border border-gray-600 rounded-xl p-3 text-center cursor-pointer transition-all duration-200 hover:border-primary-500">
                        <span class="text-white font-medium">₦{{ number_format($amount) }}</span>
                    </label>
                @endforeach
            </div>
        </div>

        <!-- Custom Amount -->
        <div class="space-y-2">
            <label for="amount" class="text-white font-medium">Or Enter Custom Amount</label>
            <div class="relative">
                <span class="absolute left-4 top-1/2 transform -translate-y-1/2 text-gray-400">₦</span>
                <input type="number" 
                       id="amount" 
                       name="amount" 
                       value="{{ old('amount') }}"
                       placeholder="0" 
                       min="500" 
                       max="100000"
                       class="w-full pl-8 pr-4 py-4 bg-dark-800 border border-gray-600 rounded-xl text-white placeholder-gray-400 focus:border-primary-500 focus:outline-none"
                       required>
            </div>
            <p class="text-xs text-gray-400">Min: ₦500, Max: ₦100,000</p>
        </div>

        <!-- Transaction PIN -->
        <div class="space-y-2">
            <label for="transaction_pin" class="text-white font-medium">Transaction PIN</label>
            <input type="password" 
                   id="transaction_pin" 
                   name="transaction_pin" 
                   placeholder="Enter 4-digit PIN" 
                   maxlength="4"
                   pattern="[0-9]{4}"
                   class="w-full p-4 bg-dark-800 border border-gray-600 rounded-xl text-white placeholder-gray-400 focus:border-primary-500 focus:outline-none"
                   required>
        </div>

        <!-- Summary -->
        <div class="card-gradient rounded-xl p-4 space-y-2" id="transaction-summary" style="display: none;">
            <h3 class="text-white font-medium mb-3">Transaction Summary</h3>
            <div class="flex justify-between text-sm">
                <span class="text-gray-400">Provider:</span>
                <span class="text-white" id="summary-provider">-</span>
            </div>
            <div class="flex justify-between text-sm">
                <span class="text-gray-400">Meter Number:</span>
                <span class="text-white" id="summary-meter">-</span>
            </div>
            <div class="flex justify-between text-sm">
                <span class="text-gray-400">Meter Type:</span>
                <span class="text-white" id="summary-meter-type">-</span>
            </div>
            <div class="flex justify-between text-sm">
                <span class="text-gray-400">Amount:</span>
                <span class="text-white" id="summary-amount">₦0</span>
            </div>
            <div class="flex justify-between text-sm">
                <span class="text-gray-400">Fee:</span>
                <span class="text-white" id="summary-fee">₦0</span>
            </div>
            <hr class="border-gray-600 my-2">
            <div class="flex justify-between font-medium">
                <span class="text-white">Total:</span>
                <span class="text-primary-400" id="summary-total">₦0</span>
            </div>
        </div>

        <!-- Submit Button -->
        <button type="submit" 
                class="w-full bg-gradient-to-r from-primary-600 to-primary-700 hover:from-primary-700 hover:to-primary-800 text-white font-semibold py-4 rounded-xl transition-all duration-200 disabled:opacity-50"
                id="submit-btn" 
                disabled>
            <i class="fas fa-bolt mr-2"></i>
            Pay Electricity Bill
        </button>
    </form>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const providerRadios = document.querySelectorAll('.provider-radio');
    const meterTypeRadios = document.querySelectorAll('.meter-type-radio');
    const amountRadios = document.querySelectorAll('.amount-radio');
    const meterNumberInput = document.getElementById('meter_number');
    const phoneInput = document.getElementById('phone');
    const amountInput = document.getElementById('amount');
    const submitBtn = document.getElementById('submit-btn');
    const summary = document.getElementById('transaction-summary');
    const verifyBtn = document.getElementById('verify-meter');
    const meterVerification = document.getElementById('meter-verification');
    
    const providers = @json($providers);
    
    function updateSummary() {
        const selectedProvider = document.querySelector('.provider-radio:checked')?.value;
        const selectedMeterType = document.querySelector('.meter-type-radio:checked')?.value;
        const selectedAmount = document.querySelector('.amount-radio:checked')?.value || amountInput.value;
        const meterNumber = meterNumberInput.value;
        const phone = phoneInput.value;
        
        if (selectedProvider && selectedMeterType && selectedAmount && meterNumber && phone) {
            const amount = parseFloat(selectedAmount);
            const fee = Math.min(amount * 0.01, 100); // 1% capped at ₦100
            const total = amount + fee;
            
            document.getElementById('summary-provider').textContent = providers[selectedProvider];
            document.getElementById('summary-meter').textContent = meterNumber;
            document.getElementById('summary-meter-type').textContent = selectedMeterType.charAt(0).toUpperCase() + selectedMeterType.slice(1);
            document.getElementById('summary-amount').textContent = '₦' + amount.toLocaleString();
            document.getElementById('summary-fee').textContent = '₦' + fee.toLocaleString();
            document.getElementById('summary-total').textContent = '₦' + total.toLocaleString();
            
            summary.style.display = 'block';
            submitBtn.disabled = false;
        } else {
            summary.style.display = 'none';
            submitBtn.disabled = true;
        }
    }
    
    // Style selection handlers
    providerRadios.forEach(radio => {
        radio.addEventListener('change', function() {
            document.querySelectorAll('.provider-option').forEach(option => {
                option.classList.remove('border-primary-500', 'bg-primary-500/20');
            });
            if (this.checked) {
                this.nextElementSibling.classList.add('border-primary-500', 'bg-primary-500/20');
            }
            updateSummary();
        });
    });
    
    meterTypeRadios.forEach(radio => {
        radio.addEventListener('change', function() {
            document.querySelectorAll('.meter-type-option').forEach(option => {
                option.classList.remove('border-primary-500', 'bg-primary-500/20');
            });
            if (this.checked) {
                this.nextElementSibling.classList.add('border-primary-500', 'bg-primary-500/20');
            }
            updateSummary();
        });
    });
    
    amountRadios.forEach(radio => {
        radio.addEventListener('change', function() {
            document.querySelectorAll('.amount-option').forEach(option => {
                option.classList.remove('border-primary-500', 'bg-primary-500/20');
            });
            if (this.checked) {
                this.nextElementSibling.classList.add('border-primary-500', 'bg-primary-500/20');
                amountInput.value = this.value;
            }
            updateSummary();
        });
    });
    
    amountInput.addEventListener('input', function() {
        if (this.value) {
            amountRadios.forEach(radio => {
                radio.checked = false;
                radio.nextElementSibling.classList.remove('border-primary-500', 'bg-primary-500/20');
            });
        }
        updateSummary();
    });
    
    meterNumberInput.addEventListener('input', function() {
        if (this.value.length >= 10) {
            verifyBtn.style.display = 'block';
        } else {
            verifyBtn.style.display = 'none';
            meterVerification.style.display = 'none';
        }
        updateSummary();
    });
    
    verifyBtn.addEventListener('click', function() {
        const meterNumber = meterNumberInput.value;
        const provider = document.querySelector('.provider-radio:checked')?.value;
        
        if (meterNumber && provider) {
            // Simulate meter verification
            verifyBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
            
            setTimeout(() => {
                document.getElementById('customer-name').textContent = 'John Doe';
                meterVerification.style.display = 'block';
                verifyBtn.innerHTML = 'Verified';
                verifyBtn.style.color = '#10b981';
            }, 2000);
        }
    });
    
    phoneInput.addEventListener('input', updateSummary);
    
    // Initialize selected styles
    document.querySelectorAll('.provider-radio:checked').forEach(radio => {
        radio.nextElementSibling.classList.add('border-primary-500', 'bg-primary-500/20');
    });
    
    document.querySelectorAll('.meter-type-radio:checked').forEach(radio => {
        radio.nextElementSibling.classList.add('border-primary-500', 'bg-primary-500/20');
    });
    
    document.querySelectorAll('.amount-radio:checked').forEach(radio => {
        radio.nextElementSibling.classList.add('border-primary-500', 'bg-primary-500/20');
    });
    
    updateSummary();
});
</script>
@endsection