@extends('layouts.responsive')

@section('title', 'Document Upload')

@section('content')
<div class="space-y-8">
    <!-- Header Section -->
    <div class="text-center">
        <div class="w-20 h-20 bg-gradient-to-br from-purple-500 to-purple-700 rounded-2xl flex items-center justify-center mx-auto mb-6">
            <i class="fas fa-file-upload text-white text-3xl"></i>
        </div>
        <h2 class="text-3xl font-bold text-white mb-4">Document Upload</h2>
        <p class="text-gray-300 max-w-md mx-auto">
            Upload your government-issued ID and take a selfie for secure identity verification
        </p>
    </div>
    
    <!-- Upload Form -->
    <div class="bg-dark-800/50 backdrop-blur-sm border border-dark-700/50 rounded-2xl p-6">
        <form method="POST" action="{{ route('kyc.document') }}" enctype="multipart/form-data" id="documentForm">
            @csrf
            
            <div class="space-y-6">
                <!-- Document Type -->
                <div>
                    <label for="document_type" class="form-label">
                        <i class="fas fa-id-card mr-2"></i>
                        Document Type
                    </label>
                    <select id="document_type" name="document_type" 
                            class="form-input @error('document_type') border-red-500 focus:ring-red-500 @enderror" 
                            required>
                        <option value="">Select document type</option>
                        <option value="passport" {{ old('document_type') == 'passport' ? 'selected' : '' }}>
                            International Passport
                        </option>
                        <option value="drivers_license" {{ old('document_type') == 'drivers_license' ? 'selected' : '' }}>
                            Driver's License
                        </option>
                        <option value="voters_card" {{ old('document_type') == 'voters_card' ? 'selected' : '' }}>
                            Voter's Card
                        </option>
                        <option value="nin_slip" {{ old('document_type') == 'nin_slip' ? 'selected' : '' }}>
                            NIN Slip
                        </option>
                    </select>
                    @error('document_type')
                        <p class="text-red-400 text-sm mt-2 flex items-center">
                            <i class="fas fa-exclamation-circle mr-2"></i>{{ $message }}
                        </p>
                    @enderror
                </div>

                <!-- Document Upload -->
                <div>
                    <label class="form-label">
                        <i class="fas fa-file-image mr-2"></i>
                        Document Image
                    </label>
                    <div class="mt-3 flex justify-center px-6 pt-8 pb-8 border-2 border-dark-600 border-dashed rounded-2xl hover:border-primary-500 transition-all duration-200 relative" id="documentDropZone">
                        <div class="space-y-4 text-center">
                            <div class="w-16 h-16 bg-purple-500/20 rounded-2xl flex items-center justify-center mx-auto">
                                <i class="fas fa-cloud-upload-alt text-purple-400 text-2xl"></i>
                            </div>
                            <div class="text-gray-300">
                                <label for="document" class="relative cursor-pointer bg-purple-600 hover:bg-purple-700 rounded-xl px-4 py-2 font-semibold transition-colors duration-200">
                                    <span>Choose file</span>
                                    <input id="document" name="document" type="file" 
                                           class="sr-only @error('document') border-red-500 @enderror" 
                                           accept=".pdf,.jpg,.jpeg,.png"
                                           required>
                                </label>
                                <span class="ml-2">or drag and drop</span>
                            </div>
                            <p class="text-sm text-gray-400">
                                PDF, PNG, JPG up to 5MB
                            </p>
                        </div>
                        
                        <!-- File Preview -->
                        <div id="documentPreview" class="hidden absolute inset-0 rounded-2xl bg-dark-800 border-2 border-primary-500">
                            <div class="relative h-full">
                                <img id="documentImage" src="" alt="Document preview" class="w-full h-full object-contain rounded-2xl">
                                <button type="button" onclick="removeDocument()" class="absolute top-2 right-2 w-8 h-8 bg-red-600 hover:bg-red-700 rounded-full flex items-center justify-center transition-colors">
                                    <i class="fas fa-times text-white text-sm"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                    @error('document')
                        <p class="text-red-400 text-sm mt-2 flex items-center">
                            <i class="fas fa-exclamation-circle mr-2"></i>{{ $message }}
                        </p>
                    @enderror
                </div>

                <!-- Selfie Capture -->
                <div>
                    <label class="form-label">
                        <i class="fas fa-camera mr-2"></i>
                        Take a Selfie
                    </label>
                    <div class="mt-3 bg-dark-700/50 rounded-2xl p-6">
                        <div id="cameraContainer" class="relative">
                            <video id="cameraPreview" class="w-full h-64 bg-dark-800 rounded-2xl object-cover hidden" autoplay muted playsinline></video>
                            <canvas id="selfieCanvas" class="w-full h-64 bg-dark-800 rounded-2xl hidden"></canvas>
                            
                            <!-- Camera Controls -->
                            <div id="cameraControls" class="text-center mt-4">
                                <button type="button" onclick="startCamera()" id="startCameraBtn" class="px-6 py-3 bg-green-600 hover:bg-green-700 text-white font-semibold rounded-xl transition-colors duration-200">
                                    <i class="fas fa-camera mr-2"></i>Start Camera
                                </button>
                                <button type="button" onclick="capturePhoto()" id="captureBtn" class="px-6 py-3 bg-blue-600 hover:bg-blue-700 text-white font-semibold rounded-xl transition-colors duration-200 hidden">
                                    <i class="fas fa-camera mr-2"></i>Take Photo
                                </button>
                                <button type="button" onclick="retakePhoto()" id="retakeBtn" class="px-6 py-3 bg-orange-600 hover:bg-orange-700 text-white font-semibold rounded-xl transition-colors duration-200 hidden ml-3">
                                    <i class="fas fa-redo mr-2"></i>Retake
                                </button>
                            </div>
                        </div>
                        
                        <!-- Selfie Guidelines -->
                        <div class="mt-4 p-4 bg-blue-500/20 border border-blue-500/30 rounded-xl">
                            <h5 class="text-blue-400 font-semibold mb-2">Selfie Guidelines:</h5>
                            <ul class="text-sm text-gray-300 space-y-1">
                                <li class="flex items-center"><i class="fas fa-check text-green-400 mr-2"></i>Look directly at the camera</li>
                                <li class="flex items-center"><i class="fas fa-check text-green-400 mr-2"></i>Ensure good lighting on your face</li>
                                <li class="flex items-center"><i class="fas fa-check text-green-400 mr-2"></i>Remove glasses or hats if possible</li>
                                <li class="flex items-center"><i class="fas fa-check text-green-400 mr-2"></i>Keep a neutral expression</li>
                            </ul>
                        </div>
                        
                        <input type="hidden" name="selfie" id="selfieData">
                    </div>
                </div>
            </div>

            <div class="flex flex-col sm:flex-row gap-4 mt-8">
                <a href="{{ route('kyc.index') }}" class="flex-1 py-4 bg-dark-700 hover:bg-dark-600 text-white font-semibold rounded-2xl transition-colors duration-200 text-center">
                    <i class="fas fa-arrow-left mr-2"></i>Back to KYC
                </a>
                <button type="submit" class="flex-1 py-4 bg-purple-600 hover:bg-purple-700 text-white font-semibold rounded-2xl transition-colors duration-200 flex items-center justify-center">
                    <i class="fas fa-upload mr-2"></i>
                    <span id="submit-text">Upload Documents</span>
                </button>
            </div>
        </form>
    </div>

    <!-- Information Cards -->
    <div class="grid gap-6 md:grid-cols-2">
        <!-- Requirements -->
        <div class="bg-gradient-to-r from-purple-500/20 to-purple-600/20 border border-purple-500/30 rounded-2xl p-6">
            <div class="flex items-start">
                <div class="w-12 h-12 bg-purple-500/20 rounded-full flex items-center justify-center mr-4 flex-shrink-0">
                    <i class="fas fa-clipboard-check text-purple-400 text-xl"></i>
                </div>
                <div>
                    <h4 class="text-purple-400 font-bold text-lg mb-3">Document Requirements</h4>
                    <ul class="text-gray-300 text-sm space-y-2">
                        <li class="flex items-center">
                            <i class="fas fa-check text-green-400 mr-2"></i>
                            Clear, high-quality image or scan
                        </li>
                        <li class="flex items-center">
                            <i class="fas fa-check text-green-400 mr-2"></i>
                            All text must be clearly readable
                        </li>
                        <li class="flex items-center">
                            <i class="fas fa-check text-green-400 mr-2"></i>
                            Document must be valid and not expired
                        </li>
                        <li class="flex items-center">
                            <i class="fas fa-check text-green-400 mr-2"></i>
                            File size should not exceed 5MB
                        </li>
                    </ul>
                </div>
            </div>
        </div>

        <!-- Processing Time -->
        <div class="bg-gradient-to-r from-blue-500/20 to-blue-600/20 border border-blue-500/30 rounded-2xl p-6">
            <div class="flex items-start">
                <div class="w-12 h-12 bg-blue-500/20 rounded-full flex items-center justify-center mr-4 flex-shrink-0">
                    <i class="fas fa-clock text-blue-400 text-xl"></i>
                </div>
                <div>
                    <h4 class="text-blue-400 font-bold text-lg mb-3">Processing Time</h4>
                    <p class="text-gray-300 text-sm mb-3">
                        Document verification typically takes 24-48 hours. You'll receive notifications once the review is complete.
                    </p>
                    <div class="flex items-center text-sm text-gray-400">
                        <i class="fas fa-shield-alt text-blue-400 mr-2"></i>
                        All documents are processed securely and confidentially
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
let mediaStream = null;
let selfieCapture = false;

// Document upload handling
document.getElementById('document').addEventListener('change', function(e) {
    const file = e.target.files[0];
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            document.getElementById('documentImage').src = e.target.result;
            document.getElementById('documentDropZone').classList.add('hidden');
            document.getElementById('documentPreview').classList.remove('hidden');
        };
        reader.readAsDataURL(file);
    }
});

// Drag and drop functionality
const dropZone = document.getElementById('documentDropZone');

['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
    dropZone.addEventListener(eventName, preventDefaults, false);
});

function preventDefaults(e) {
    e.preventDefault();
    e.stopPropagation();
}

['dragenter', 'dragover'].forEach(eventName => {
    dropZone.addEventListener(eventName, highlight, false);
});

['dragleave', 'drop'].forEach(eventName => {
    dropZone.addEventListener(eventName, unhighlight, false);
});

function highlight(e) {
    dropZone.style.borderColor = 'rgb(168, 85, 247)';
}

function unhighlight(e) {
    dropZone.style.borderColor = '';
}

dropZone.addEventListener('drop', handleDrop, false);

function handleDrop(e) {
    const dt = e.dataTransfer;
    const files = dt.files;
    
    if (files.length > 0) {
        document.getElementById('document').files = files;
        document.getElementById('document').dispatchEvent(new Event('change'));
    }
}

function removeDocument() {
    document.getElementById('document').value = '';
    document.getElementById('documentImage').src = '';
    document.getElementById('documentDropZone').classList.remove('hidden');
    document.getElementById('documentPreview').classList.add('hidden');
}

// Camera functionality
async function startCamera() {
    try {
        mediaStream = await navigator.mediaDevices.getUserMedia({ 
            video: { 
                width: { ideal: 640 },
                height: { ideal: 480 },
                facingMode: 'user'
            } 
        });
        
        const video = document.getElementById('cameraPreview');
        video.srcObject = mediaStream;
        video.classList.remove('hidden');
        
        document.getElementById('startCameraBtn').classList.add('hidden');
        document.getElementById('captureBtn').classList.remove('hidden');
        
    } catch (err) {
        console.error('Error accessing camera:', err);
        showToast('Camera access denied. Please allow camera access and try again.', 'error');
    }
}

function capturePhoto() {
    const video = document.getElementById('cameraPreview');
    const canvas = document.getElementById('selfieCanvas');
    const ctx = canvas.getContext('2d');
    
    canvas.width = video.videoWidth;
    canvas.height = video.videoHeight;
    
    ctx.drawImage(video, 0, 0);
    
    // Convert to blob and store
    canvas.toBlob(function(blob) {
        const reader = new FileReader();
        reader.onload = function(e) {
            document.getElementById('selfieData').value = e.target.result;
        };
        reader.readAsDataURL(blob);
    }, 'image/jpeg', 0.8);
    
    video.classList.add('hidden');
    canvas.classList.remove('hidden');
    document.getElementById('captureBtn').classList.add('hidden');
    document.getElementById('retakeBtn').classList.remove('hidden');
    
    // Stop camera stream
    if (mediaStream) {
        mediaStream.getTracks().forEach(track => track.stop());
    }
    
    selfieCapture = true;
    showToast('Selfie captured successfully!', 'success');
}

function retakePhoto() {
    document.getElementById('selfieCanvas').classList.add('hidden');
    document.getElementById('retakeBtn').classList.add('hidden');
    document.getElementById('startCameraBtn').classList.remove('hidden');
    document.getElementById('selfieData').value = '';
    selfieCapture = false;
}

// Form validation
document.getElementById('documentForm').addEventListener('submit', function(e) {
    const documentFile = document.getElementById('document').files[0];
    const selfieData = document.getElementById('selfieData').value;
    const submitButton = this.querySelector('button[type="submit"]');
    const submitText = document.getElementById('submit-text');
    
    if (!documentFile) {
        e.preventDefault();
        showToast('Please upload a document image', 'error');
        return;
    }
    
    if (!selfieData) {
        e.preventDefault();
        showToast('Please take a selfie for verification', 'error');
        return;
    }
    
    // Show loading state
    submitButton.disabled = true;
    submitText.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Uploading...';
});

function showToast(message, type = 'info') {
    const toast = document.createElement('div');
    const iconClass = type === 'success' ? 'fa-check-circle' : type === 'error' ? 'fa-exclamation-circle' : 'fa-info-circle';
    const bgClass = type === 'success' ? 'bg-green-600' : type === 'error' ? 'bg-red-600' : 'bg-blue-600';
    
    toast.className = `fixed top-4 right-4 p-4 rounded-2xl text-white z-50 ${bgClass} flex items-center shadow-2xl max-w-sm`;
    toast.innerHTML = `
        <i class="fas ${iconClass} mr-3 text-lg"></i>
        <span class="font-medium">${message}</span>
    `;
    
    document.body.appendChild(toast);
    
    // Animate in
    toast.style.transform = 'translateX(100%)';
    setTimeout(() => {
        toast.style.transform = 'translateX(0)';
        toast.style.transition = 'transform 0.3s ease-out';
    }, 10);
    
    // Remove after delay
    setTimeout(() => {
        toast.style.transform = 'translateX(100%)';
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}

// Cleanup camera on page unload
window.addEventListener('beforeunload', function() {
    if (mediaStream) {
        mediaStream.getTracks().forEach(track => track.stop());
    }
});
</script>
@endpush
@endsection