@extends('layouts.responsive')

@section('title', 'Wallet')

@section('content')
<div class="space-y-8">
    <!-- Enhanced Wallet Balance Card -->
    <div class="balance-card slide-up">
        <div class="relative z-10">
            <div class="flex justify-between items-start mb-6">
                <div>
                    <p class="text-white/70 text-sm font-medium">Available Balance</p>
                    <p class="text-4xl font-bold text-white mt-1" id="main-balance">₦{{ number_format($wallet?->available_balance ?? 0, 2) }}</p>
                    @if($wallet && $wallet->is_frozen)
                        <div class="flex items-center mt-3">
                            <i class="fas fa-lock text-red-400 text-sm mr-2"></i>
                            <span class="text-red-400 text-sm font-medium">Wallet Frozen</span>
                        </div>
                    @else
                        <div class="flex items-center mt-3">
                            <i class="fas fa-check-circle text-green-400 text-sm mr-2"></i>
                            <span class="text-green-400 text-sm font-medium">Active</span>
                        </div>
                    @endif
                </div>
                <div class="text-right">
                    <button onclick="toggleBalanceVisibility()" class="p-3 bg-white/15 rounded-2xl hover:bg-white/25 transition-all duration-200 backdrop-blur-sm">
                        <i id="balance-eye" class="fas fa-eye text-white text-lg"></i>
                    </button>
                </div>
            </div>
            
            <div class="grid grid-cols-2 gap-6">
                <div>
                    <p class="text-white/60 text-xs uppercase tracking-wide font-medium">Total Balance</p>
                    <p class="text-white font-bold text-xl mt-1" id="total-balance">₦{{ number_format($wallet?->balance ?? 0, 2) }}</p>
                </div>
                <div class="text-right">
                    <p class="text-white/60 text-xs uppercase tracking-wide font-medium">Locked Balance</p>
                    <p class="text-white font-bold text-xl mt-1" id="locked-balance">₦{{ number_format($wallet?->locked_balance ?? 0, 2) }}</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Enhanced Quick Actions -->
    <div class="space-y-6">
        <h3 class="text-xl font-bold text-white">Quick Actions</h3>
        <div class="grid grid-cols-2 gap-4">
            <button onclick="openFundModal()" class="quick-action group">
                <div class="quick-action-icon bg-gradient-to-br from-green-500 to-green-700 text-white group-hover:scale-110 transition-transform duration-200">
                    <i class="fas fa-plus"></i>
                </div>
                <span class="text-xs text-gray-300 font-semibold">Fund Wallet</span>
            </button>
            
            <a href="{{ route('transfer.p2p') }}" class="quick-action group">
                <div class="quick-action-icon bg-gradient-to-br from-blue-500 to-blue-700 text-white group-hover:scale-110 transition-transform duration-200">
                    <i class="fas fa-paper-plane"></i>
                </div>
                <span class="text-xs text-gray-300 font-semibold">Send Money</span>
            </a>
            
            <a href="{{ route('transfer.bank') }}" class="quick-action group">
                <div class="quick-action-icon bg-gradient-to-br from-purple-500 to-purple-700 text-white group-hover:scale-110 transition-transform duration-200">
                    <i class="fas fa-university"></i>
                </div>
                <span class="text-xs text-gray-300 font-semibold">Withdraw</span>
            </a>
            
            <a href="{{ route('transactions.index') }}" class="quick-action group">
                <div class="quick-action-icon bg-gradient-to-br from-orange-500 to-orange-700 text-white group-hover:scale-110 transition-transform duration-200">
                    <i class="fas fa-history"></i>
                </div>
                <span class="text-xs text-gray-300 font-semibold">History</span>
            </a>
        </div>
    </div>

    <!-- Enhanced Daily Limits Card -->
    <div class="bg-dark-800/50 backdrop-blur-sm border border-dark-700/50 rounded-2xl p-6">
        <h3 class="text-xl font-bold text-white mb-6">Daily Limits</h3>
        <div class="space-y-6">
            <div>
                <div class="flex justify-between text-sm mb-3 text-gray-300">
                    <span class="font-medium">Daily Transfer Limit</span>
                    <span class="font-bold">₦{{ number_format($wallet?->daily_limit ?? 100000, 0) }}</span>
                </div>
                <div class="w-full bg-dark-700 rounded-full h-4">
                    <div class="bg-gradient-to-r from-primary-500 via-primary-600 to-primary-700 h-4 rounded-full transition-all duration-700 shadow-sm" 
                         style="width: {{ $wallet ? (($wallet->daily_limit - $wallet->remaining_daily_limit) / $wallet->daily_limit) * 100 : 0 }}%">
                    </div>
                </div>
                <p class="text-sm text-gray-400 mt-3 font-medium">₦{{ number_format($wallet?->remaining_daily_limit ?? 0, 0) }} remaining today</p>
            </div>
            
            @if(!auth()->user()->hasCompletedKyc())
            <div class="bg-gradient-to-r from-orange-500/20 to-yellow-500/20 border border-orange-500/30 rounded-2xl p-4">
                <div class="flex items-start">
                    <div class="w-10 h-10 bg-orange-500/20 rounded-full flex items-center justify-center mr-4">
                        <i class="fas fa-exclamation-triangle text-orange-400"></i>
                    </div>
                    <div class="flex-1">
                        <p class="text-orange-400 text-sm font-bold mb-1">Complete KYC for higher limits</p>
                        <p class="text-gray-300 text-xs">Increase your daily limit up to ₦5,000,000</p>
                        <a href="{{ route('kyc.index') }}" class="inline-flex items-center mt-3 px-3 py-1.5 bg-orange-500 hover:bg-orange-600 text-white text-xs font-semibold rounded-lg transition-colors">
                            <i class="fas fa-shield-check mr-1"></i>
                            Verify Now
                        </a>
                    </div>
                </div>
            </div>
            @endif
        </div>
    </div>

    <!-- Enhanced Virtual Account -->
    @if($virtualAccount)
    <div class="bg-dark-800/50 backdrop-blur-sm border border-dark-700/50 rounded-2xl p-6">
        <h3 class="text-xl font-bold text-white mb-6">Virtual Account</h3>
        <div class="bg-gradient-to-br from-dark-700/80 to-dark-600/80 rounded-2xl p-5 border border-dark-600/50">
            <div class="flex justify-between items-center mb-4">
                <span class="text-sm font-semibold text-gray-300 uppercase tracking-wider">Account Number</span>
                <button onclick="copyToClipboard('{{ $virtualAccount->account_number }}')" 
                        class="flex items-center px-3 py-1.5 bg-primary-600/20 hover:bg-primary-600/30 border border-primary-600/30 text-primary-400 text-sm font-semibold rounded-lg transition-colors">
                    <i class="fas fa-copy mr-2"></i>
                    Copy
                </button>
            </div>
            <p class="text-3xl font-mono font-bold mb-4 text-white tracking-wider">{{ $virtualAccount->account_number }}</p>
            <div class="flex justify-between items-end">
                <div>
                    <p class="text-sm text-gray-400 font-medium">{{ $virtualAccount->bank_name }}</p>
                    <p class="text-xs text-gray-500 mt-1">Use this account to fund your wallet</p>
                </div>
                <div class="px-3 py-1 bg-green-500/20 text-green-400 border border-green-500/30 rounded-full text-xs font-semibold">
                    <i class="fas fa-circle text-[8px] mr-1"></i>Active
                </div>
            </div>
        </div>
    </div>
    @else
    <div class="bg-dark-800/50 backdrop-blur-sm border border-dark-700/50 rounded-2xl p-6">
        <h3 class="text-xl font-bold text-white mb-6">Virtual Account</h3>
        <div class="text-center py-8">
            <div class="w-20 h-20 bg-dark-700/50 rounded-2xl flex items-center justify-center mx-auto mb-6">
                <i class="fas fa-university text-gray-500 text-3xl"></i>
            </div>
            <h4 class="text-white font-bold text-lg mb-2">No virtual account yet</h4>
            <p class="text-gray-400 mb-6">Create a dedicated account number for easy funding</p>
            <button onclick="createVirtualAccount()" class="inline-flex items-center px-6 py-3 bg-primary-600 hover:bg-primary-700 text-white font-semibold rounded-xl transition-colors duration-200">
                <i class="fas fa-plus mr-2"></i>Create Virtual Account
            </button>
        </div>
    </div>
    @endif

    <!-- Enhanced Recent Transactions -->
    <div class="space-y-6">
        <div class="flex justify-between items-center">
            <h3 class="text-xl font-bold text-white">Recent Transactions</h3>
            <a href="{{ route('transactions.index') }}" class="text-primary-400 text-sm font-semibold hover:text-primary-300 transition-colors">View All</a>
        </div>
        
        @if($recentTransactions && $recentTransactions->count() > 0)
            <div class="space-y-3">
                @foreach($recentTransactions as $transaction)
                <div class="transaction-item group">
                    <div class="flex items-center flex-1">
                        <div class="transaction-icon {{ $transaction->type === 'credit' ? 'bg-green-500/20 text-green-400 border border-green-500/30' : 'bg-red-500/20 text-red-400 border border-red-500/30' }}">
                            @if($transaction->category === 'wallet_funding')
                                <i class="fas fa-plus"></i>
                            @elseif($transaction->category === 'p2p_transfer')
                                <i class="fas fa-exchange-alt"></i>
                            @elseif($transaction->category === 'bank_transfer')
                                <i class="fas fa-university"></i>
                            @elseif($transaction->category === 'airtime_purchase')
                                <i class="fas fa-mobile-alt"></i>
                            @elseif($transaction->category === 'data_purchase')
                                <i class="fas fa-wifi"></i>
                            @else
                                <i class="fas fa-credit-card"></i>
                            @endif
                        </div>
                        <div class="ml-4 flex-1">
                            <p class="text-white font-semibold text-sm">{{ $transaction->description }}</p>
                            <div class="flex items-center mt-1">
                                <p class="text-gray-400 text-xs">{{ $transaction->created_at->format('M d, Y H:i') }}</p>
                                <span class="mx-2 text-gray-600">•</span>
                                <div class="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium {{ $transaction->status === 'completed' ? 'bg-green-500/20 text-green-400' : ($transaction->status === 'pending' ? 'bg-yellow-500/20 text-yellow-400' : 'bg-red-500/20 text-red-400') }}">
                                    <i class="fas fa-circle text-[8px] mr-1"></i>{{ ucfirst($transaction->status) }}
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="text-right">
                        <p class="font-bold text-base {{ $transaction->type === 'credit' ? 'text-green-400' : 'text-red-400' }}">
                            {{ $transaction->type === 'credit' ? '+' : '-' }}₦{{ number_format($transaction->amount, 2) }}
                        </p>
                    </div>
                </div>
                @endforeach
            </div>
        @else
            <div class="text-center py-16">
                <div class="w-20 h-20 bg-dark-800/50 rounded-2xl flex items-center justify-center mx-auto mb-6">
                    <i class="fas fa-receipt text-gray-500 text-3xl"></i>
                </div>
                <h4 class="text-white font-bold text-lg mb-2">No transactions yet</h4>
                <p class="text-gray-400 text-sm mb-6">Your transaction history will appear here</p>
                <button onclick="openFundModal()" class="inline-flex items-center px-6 py-3 bg-primary-600 hover:bg-primary-700 text-white font-semibold rounded-xl transition-colors duration-200">
                    <i class="fas fa-plus mr-2"></i>Fund Wallet
                </button>
            </div>
        @endif
    </div>
</div>

<!-- Enhanced Fund Wallet Modal -->
<div id="fundModal" class="fixed inset-0 bg-black/60 backdrop-blur-sm hidden z-50">
    <div class="absolute bottom-0 left-0 right-0 bg-dark-900/95 backdrop-blur-xl rounded-t-3xl p-6 slide-up border-t border-dark-700/50">
        <div class="w-12 h-1 bg-gray-600 rounded-full mx-auto mb-6"></div>
        <h3 class="text-2xl font-bold text-white mb-6">Fund Your Wallet</h3>
        
        <form id="fundForm" method="POST" action="{{ route('wallet.fund') }}">
            @csrf
            <div class="mb-6">
                <label for="amount" class="block text-sm font-semibold text-gray-300 mb-3">Amount (₦)</label>
                <input type="number" id="amount" name="amount" min="100" max="500000" step="0.01" 
                       class="w-full px-4 py-4 bg-dark-800 border border-dark-600 rounded-2xl text-white text-xl placeholder-gray-400 focus:outline-none focus:ring-2 focus:ring-primary-500 focus:border-transparent transition-colors" 
                       placeholder="Enter amount" required>
                <p class="text-xs text-gray-400 mt-3 flex items-center">
                    <i class="fas fa-info-circle mr-2"></i>
                    Minimum: ₦100, Maximum: ₦500,000
                </p>
            </div>
            
            <!-- Enhanced Quick Amount Buttons -->
            <div class="mb-8">
                <p class="text-sm font-semibold text-gray-300 mb-3">Quick amounts</p>
                <div class="grid grid-cols-4 gap-3">
                    <button type="button" onclick="setAmount(1000)" class="py-3 px-3 bg-dark-700/50 hover:bg-dark-600 text-white rounded-xl text-sm font-semibold transition-all duration-200 border border-dark-600 hover:border-dark-500">₦1K</button>
                    <button type="button" onclick="setAmount(5000)" class="py-3 px-3 bg-dark-700/50 hover:bg-dark-600 text-white rounded-xl text-sm font-semibold transition-all duration-200 border border-dark-600 hover:border-dark-500">₦5K</button>
                    <button type="button" onclick="setAmount(10000)" class="py-3 px-3 bg-dark-700/50 hover:bg-dark-600 text-white rounded-xl text-sm font-semibold transition-all duration-200 border border-dark-600 hover:border-dark-500">₦10K</button>
                    <button type="button" onclick="setAmount(20000)" class="py-3 px-3 bg-dark-700/50 hover:bg-dark-600 text-white rounded-xl text-sm font-semibold transition-all duration-200 border border-dark-600 hover:border-dark-500">₦20K</button>
                </div>
            </div>
            
            <!-- Payment Method Info -->
            <div class="mb-8 p-4 bg-primary-500/10 border border-primary-500/20 rounded-2xl">
                <div class="flex items-center mb-2">
                    <i class="fas fa-shield-check text-primary-400 mr-2"></i>
                    <span class="text-primary-400 font-semibold text-sm">Secure Payment via Paystack</span>
                </div>
                <p class="text-gray-300 text-xs">You'll be redirected to complete your payment securely</p>
            </div>
            
            <div class="flex space-x-4">
                <button type="button" onclick="closeFundModal()" class="flex-1 py-4 bg-dark-700 hover:bg-dark-600 text-white font-semibold rounded-2xl transition-colors duration-200">
                    Cancel
                </button>
                <button type="submit" class="flex-1 py-4 bg-primary-600 hover:bg-primary-700 text-white font-semibold rounded-2xl transition-colors duration-200 flex items-center justify-center">
                    <i class="fas fa-credit-card mr-2"></i>
                    Proceed to Payment
                </button>
            </div>
        </form>
    </div>
</div>

@push('scripts')
<script>
let balanceVisible = true;

function toggleBalanceVisibility() {
    const eye = document.getElementById('balance-eye');
    const mainBalance = document.getElementById('main-balance');
    const totalBalance = document.getElementById('total-balance');
    const lockedBalance = document.getElementById('locked-balance');
    
    if (balanceVisible) {
        eye.className = 'fas fa-eye-slash text-white text-lg';
        mainBalance.textContent = '₦••••••';
        totalBalance.textContent = '₦••••••';
        lockedBalance.textContent = '₦••••••';
        balanceVisible = false;
    } else {
        eye.className = 'fas fa-eye text-white text-lg';
        mainBalance.textContent = '₦{{ number_format($wallet?->available_balance ?? 0, 2) }}';
        totalBalance.textContent = '₦{{ number_format($wallet?->balance ?? 0, 2) }}';
        lockedBalance.textContent = '₦{{ number_format($wallet?->locked_balance ?? 0, 2) }}';
        balanceVisible = true;
    }
    
    // Add haptic feedback
    if (navigator.vibrate) {
        navigator.vibrate(50);
    }
}

function openFundModal() {
    document.getElementById('fundModal').classList.remove('hidden');
    document.getElementById('amount').focus();
}

function closeFundModal() {
    document.getElementById('fundModal').classList.add('hidden');
    document.getElementById('amount').value = '';
}

function setAmount(amount) {
    const input = document.getElementById('amount');
    input.value = amount;
    input.focus();
    
    // Add visual feedback
    input.style.transform = 'scale(1.02)';
    setTimeout(() => {
        input.style.transform = '';
    }, 150);
}

function copyToClipboard(text) {
    navigator.clipboard.writeText(text).then(function() {
        showToast('Account number copied to clipboard!', 'success');
    }).catch(function() {
        // Fallback for older browsers
        const textArea = document.createElement('textarea');
        textArea.value = text;
        textArea.style.position = 'fixed';
        textArea.style.opacity = '0';
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
        showToast('Account number copied!', 'success');
    });
    
    // Add haptic feedback
    if (navigator.vibrate) {
        navigator.vibrate([50, 100, 50]);
    }
}

function createVirtualAccount() {
    if (!confirm('Create a virtual account to fund your wallet via bank transfer?\n\nNote: KYC verification may be required.')) {
        return;
    }
    
    // Show loading state
    const button = event.target;
    const originalText = button.innerHTML;
    button.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Creating...';
    button.disabled = true;
    
    fetch('/virtual-account/create', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast('Virtual account created successfully!', 'success');
            setTimeout(() => window.location.reload(), 2000);
        } else {
            showToast(data.message || 'Failed to create virtual account', 'error');
            button.innerHTML = originalText;
            button.disabled = false;
        }
    })
    .catch(error => {
        showToast('An error occurred. Please try again.', 'error');
        button.innerHTML = originalText;
        button.disabled = false;
    });
}

function showToast(message, type = 'info') {
    const toast = document.createElement('div');
    const iconClass = type === 'success' ? 'fa-check-circle' : type === 'error' ? 'fa-exclamation-circle' : 'fa-info-circle';
    const bgClass = type === 'success' ? 'bg-green-600' : type === 'error' ? 'bg-red-600' : 'bg-blue-600';
    
    toast.className = `fixed top-4 right-4 p-4 rounded-2xl text-white z-50 ${bgClass} flex items-center shadow-2xl max-w-sm`;
    toast.innerHTML = `
        <i class="fas ${iconClass} mr-3 text-lg"></i>
        <span class="font-medium">${message}</span>
    `;
    
    document.body.appendChild(toast);
    
    // Animate in
    toast.style.transform = 'translateX(100%)';
    setTimeout(() => {
        toast.style.transform = 'translateX(0)';
        toast.style.transition = 'transform 0.3s ease-out';
    }, 10);
    
    // Remove after delay
    setTimeout(() => {
        toast.style.transform = 'translateX(100%)';
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}

// Enhanced form validation
document.getElementById('fundForm').addEventListener('submit', function(e) {
    const amount = parseFloat(document.getElementById('amount').value);
    
    if (isNaN(amount) || amount < 100) {
        e.preventDefault();
        showToast('Minimum funding amount is ₦100', 'error');
        return;
    }
    
    if (amount > 500000) {
        e.preventDefault();
        showToast('Maximum funding amount is ₦500,000', 'error');
        return;
    }
    
    // Show loading state
    const submitButton = this.querySelector('button[type="submit"]');
    submitButton.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Processing...';
    submitButton.disabled = true;
});

// Close modal when clicking outside
document.addEventListener('click', function(event) {
    const modal = document.getElementById('fundModal');
    if (event.target === modal) {
        closeFundModal();
    }
});

// Format amount input
document.getElementById('amount').addEventListener('input', function(e) {
    const value = parseFloat(e.target.value);
    if (!isNaN(value)) {
        // Add visual feedback for valid amounts
        if (value >= 100 && value <= 500000) {
            e.target.style.borderColor = 'rgb(34, 197, 94)';
        } else {
            e.target.style.borderColor = 'rgb(239, 68, 68)';
        }
    } else {
        e.target.style.borderColor = '';
    }
});

// Auto-refresh balance every 30 seconds
let refreshInterval = setInterval(function() {
    if (document.visibilityState === 'visible') {
        fetch('/wallet/balance', {
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success && balanceVisible) {
                document.getElementById('main-balance').textContent = '₦' + Number(data.available_balance).toLocaleString('en-NG', {minimumFractionDigits: 2});
                document.getElementById('total-balance').textContent = '₦' + Number(data.balance).toLocaleString('en-NG', {minimumFractionDigits: 2});
                document.getElementById('locked-balance').textContent = '₦' + Number(data.locked_balance).toLocaleString('en-NG', {minimumFractionDigits: 2});
            }
        })
        .catch(error => console.log('Balance refresh failed:', error));
    }
}, 30000);

// Pause refresh when page is hidden
document.addEventListener('visibilitychange', function() {
    if (document.visibilityState === 'hidden') {
        clearInterval(refreshInterval);
    } else {
        // Restart refresh when page becomes visible
        refreshInterval = setInterval(function() {
            fetch('/wallet/balance', {
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success && balanceVisible) {
                    document.getElementById('main-balance').textContent = '₦' + Number(data.available_balance).toLocaleString('en-NG', {minimumFractionDigits: 2});
                    document.getElementById('total-balance').textContent = '₦' + Number(data.balance).toLocaleString('en-NG', {minimumFractionDigits: 2});
                    document.getElementById('locked-balance').textContent = '₦' + Number(data.locked_balance).toLocaleString('en-NG', {minimumFractionDigits: 2});
                }
            })
            .catch(error => console.log('Balance refresh failed:', error));
        }, 30000);
    }
});
</script>
@endpush
@endsection