<?php

namespace Tests\Feature\Api;

use Tests\TestCase;
use App\Models\User;
use App\Models\Wallet;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Laravel\Sanctum\Sanctum;

class SecurityFeaturesTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Test rate limiting on API endpoints
     */
    public function test_api_rate_limiting(): void
    {
        // Make many requests quickly to trigger rate limiting
        for ($i = 0; $i < 65; $i++) {
            $response = $this->post('/api/auth/login', [
                'email' => 'test@example.com',
                'password' => 'wrongpassword'
            ]);
        }

        // Should get rate limited
        $this->assertEquals(429, $response->status());
        $response->assertJsonStructure([
            'success',
            'message',
            'retry_after'
        ]);
    }

    /**
     * Test transaction PIN verification requirement
     */
    public function test_transaction_pin_required_for_transfers(): void
    {
        $user = User::factory()->phoneVerified()->create();
        $wallet = Wallet::factory()->create(['user_id' => $user->id]);
        
        Sanctum::actingAs($user);

        // Test without transaction PIN
        $response = $this->post('/api/transactions/p2p', [
            'amount' => 1000,
            'recipient_identifier' => 'test2@example.com',
            'narration' => 'Test transfer'
        ]);

        $this->assertEquals(403, $response->status());
        $response->assertJson([
            'success' => false,
            'error_code' => 'PIN_NOT_SET'
        ]);
    }

    /**
     * Test audit logging for financial operations
     */
    public function test_audit_logging_for_transactions(): void
    {
        $user = User::factory()->phoneVerified()->create([
            'transaction_pin' => bcrypt('1234')
        ]);
        $wallet = Wallet::factory()->create(['user_id' => $user->id, 'balance' => 10000]);
        
        Sanctum::actingAs($user);

        // Perform a transaction that should be audited
        $response = $this->post('/api/transactions/p2p', [
            'amount' => 1000,
            'recipient_identifier' => 'test2@example.com',
            'transaction_pin' => '1234',
            'narration' => 'Test transfer'
        ]);

        // Check if audit log was created
        $this->assertDatabaseHas('audit_logs', [
            'user_id' => $user->id,
            'action' => 'transaction_created'
        ]);
    }

    /**
     * Test input validation on registration
     */
    public function test_comprehensive_input_validation(): void
    {
        // Test invalid phone number format
        $response = $this->post('/api/auth/register', [
            'first_name' => 'Test',
            'last_name' => 'User',
            'email' => 'test@example.com',
            'phone' => '08012345678', // Should be +234 format
            'password' => 'weak',      // Too weak
            'password_confirmation' => 'weak',
            'date_of_birth' => '2020-01-01', // Too young
            'gender' => 'invalid'
        ]);

        $this->assertEquals(422, $response->status());
        $response->assertJsonValidationErrors([
            'phone', 'password', 'date_of_birth', 'gender'
        ]);
    }

    /**
     * Test authentication required for protected endpoints
     */
    public function test_authentication_required(): void
    {
        $response = $this->get('/api/wallet');
        $this->assertEquals(401, $response->status());

        $response = $this->get('/api/transactions');
        $this->assertEquals(401, $response->status());

        $response = $this->post('/api/transactions/p2p');
        $this->assertEquals(401, $response->status());
    }

    /**
     * Test KYC verification requirement for financial operations
     */
    public function test_kyc_verification_required(): void
    {
        $user = User::factory()->create(); // User without KYC
        Sanctum::actingAs($user);

        $response = $this->get('/api/wallet');
        
        // Should be blocked by kyc.verified middleware
        $this->assertNotEquals(200, $response->status());
    }
}