<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Wallet;
use App\Models\Transaction;
use App\Models\KycVerification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    /**
     * Display a listing of users with search and filters.
     */
    public function index(Request $request)
    {
        $query = User::with(['wallet', 'kycVerification']);

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('first_name', 'LIKE', "%{$search}%")
                  ->orWhere('last_name', 'LIKE', "%{$search}%")
                  ->orWhere('email', 'LIKE', "%{$search}%")
                  ->orWhere('phone', 'LIKE', "%{$search}%");
            });
        }

        // Status filter
        if ($request->filled('status')) {
            if ($request->status === 'active') {
                $query->where('is_active', true);
            } elseif ($request->status === 'inactive') {
                $query->where('is_active', false);
            } elseif ($request->status === 'blocked') {
                $query->where('is_blocked', true);
            }
        }

        // KYC filter
        if ($request->filled('kyc_status')) {
            $query->whereHas('kycVerification', function ($q) use ($request) {
                $q->where('status', $request->kyc_status);
            });
        }

        // Sorting
        $sortField = $request->get('sort', 'created_at');
        $sortDirection = $request->get('direction', 'desc');
        
        $allowedSorts = ['created_at', 'first_name', 'last_name', 'email', 'phone'];
        if (in_array($sortField, $allowedSorts)) {
            $query->orderBy($sortField, $sortDirection);
        }

        $users = $query->paginate(20)->withQueryString();

        return view('admin.users.index', compact('users'));
    }

    /**
     * Display the specified user details.
     */
    public function show(User $user)
    {
        $user->load(['wallet', 'kycVerification', 'virtualCards', 'virtualAccounts']);
        
        // Get recent transactions
        $transactions = Transaction::where(function ($query) use ($user) {
            $query->where('user_id', $user->id)
                  ->orWhere('recipient_id', $user->id);
        })->latest()->limit(10)->get();

        return view('admin.users.show', compact('user', 'transactions'));
    }

    /**
     * Block or unblock a user.
     */
    public function toggleBlock(User $user)
    {
        $user->update([
            'is_blocked' => !$user->is_blocked,
            'blocked_at' => $user->is_blocked ? null : now(),
        ]);

        $action = $user->is_blocked ? 'blocked' : 'unblocked';
        
        return back()->with('success', "User has been {$action} successfully.");
    }

    /**
     * Activate or deactivate a user.
     */
    public function toggleActive(User $user)
    {
        $user->update([
            'is_active' => !$user->is_active,
        ]);

        $action = $user->is_active ? 'activated' : 'deactivated';
        
        return back()->with('success', "User has been {$action} successfully.");
    }

    /**
     * Manually verify a user.
     */
    public function verify(User $user)
    {
        $user->update([
            'email_verified_at' => now(),
            'phone_verified_at' => now(),
        ]);

        return back()->with('success', 'User has been manually verified successfully.');
    }

    /**
     * Reset user password.
     */
    public function resetPassword(Request $request, User $user)
    {
        $request->validate([
            'password' => 'required|string|min:8|confirmed',
        ]);

        $user->update([
            'password' => Hash::make($request->password),
        ]);

        return back()->with('success', 'User password has been reset successfully.');
    }

    /**
     * Reset user 2FA.
     */
    public function reset2FA(User $user)
    {
        $user->update([
            'two_factor_secret' => null,
            'two_factor_confirmed_at' => null,
        ]);

        return back()->with('success', 'User 2FA has been reset successfully.');
    }

    /**
     * Delete a user (soft delete).
     */
    public function destroy(User $user)
    {
        // Check if user has pending transactions
        $pendingTransactions = Transaction::where('user_id', $user->id)
            ->where('status', 'pending')
            ->count();

        if ($pendingTransactions > 0) {
            return back()->withErrors(['error' => 'Cannot delete user with pending transactions.']);
        }

        $user->delete();

        return redirect()->route('admin.users.index')
            ->with('success', 'User has been deleted successfully.');
    }

    /**
     * Show user wallet details.
     */
    public function wallet(User $user)
    {
        $wallet = $user->wallet;
        
        if (!$wallet) {
            return back()->withErrors(['error' => 'User does not have a wallet.']);
        }

        $transactions = Transaction::where('user_id', $user->id)
            ->orWhere('recipient_id', $user->id)
            ->latest()
            ->paginate(20);

        return view('admin.users.wallet', compact('user', 'wallet', 'transactions'));
    }

    /**
     * Adjust user wallet balance.
     */
    public function adjustBalance(Request $request, User $user)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'type' => 'required|in:credit,debit',
            'narration' => 'required|string|max:255',
        ]);

        $wallet = $user->wallet;
        
        if (!$wallet) {
            return back()->withErrors(['error' => 'User does not have a wallet.']);
        }

        $amount = $request->amount;
        
        if ($request->type === 'debit' && $wallet->balance < $amount) {
            return back()->withErrors(['error' => 'Insufficient wallet balance for debit.']);
        }

        // Create transaction record
        $transaction = Transaction::create([
            'user_id' => $user->id,
            'type' => $request->type === 'credit' ? 'credit' : 'debit',
            'amount' => $amount,
            'narration' => $request->narration . ' (Admin Adjustment)',
            'status' => 'completed',
            'reference' => 'ADJ-' . uniqid(),
            'metadata' => [
                'admin_id' => auth('admin')->id(),
                'admin_name' => auth('admin')->user()->name,
                'reason' => $request->narration,
            ],
        ]);

        // Update wallet balance
        if ($request->type === 'credit') {
            $wallet->increment('balance', $amount);
        } else {
            $wallet->decrement('balance', $amount);
        }

        return back()->with('success', "Wallet balance {$request->type}ed by ₦" . number_format($amount, 2));
    }

    /**
     * Freeze or unfreeze user wallet.
     */
    public function toggleWalletFreeze(User $user)
    {
        $wallet = $user->wallet;
        
        if (!$wallet) {
            return back()->withErrors(['error' => 'User does not have a wallet.']);
        }

        $wallet->update([
            'is_frozen' => !$wallet->is_frozen,
            'frozen_at' => $wallet->is_frozen ? null : now(),
        ]);

        $action = $wallet->is_frozen ? 'frozen' : 'unfrozen';
        
        return back()->with('success', "User wallet has been {$action} successfully.");
    }
}