<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rules\Password;

class SettingsController extends Controller
{
    /**
     * Show settings dashboard
     */
    public function index()
    {
        $user = Auth::user();
        
        return view('settings.index', compact('user'));
    }
    
    /**
     * Show profile settings
     */
    public function profile()
    {
        $user = Auth::user();
        
        return view('settings.profile', compact('user'));
    }
    
    /**
     * Update profile information
     */
    public function updateProfile(Request $request)
    {
        $user = Auth::user();
        
        $request->validate([
            'first_name' => 'required|string|max:255',
            'last_name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'required|string|unique:users,phone,' . $user->id,
            'date_of_birth' => 'nullable|date|before:today',
            'address' => 'nullable|string|max:500',
            'profile_photo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);
        
        $updateData = $request->only([
            'first_name', 'last_name', 'email', 'phone', 
            'date_of_birth', 'address'
        ]);
        
        // Handle profile photo upload
        if ($request->hasFile('profile_photo')) {
            // Delete old photo
            if ($user->profile_photo) {
                Storage::disk('public')->delete($user->profile_photo);
            }
            
            $photoPath = $request->file('profile_photo')->store('profile-photos', 'public');
            $updateData['profile_photo'] = $photoPath;
        }
        
        $user->update($updateData);
        
        return back()->with('success', 'Profile updated successfully');
    }
    
    /**
     * Show security settings
     */
    public function security()
    {
        $user = Auth::user();
        $sessions = $this->getActiveSessions($user);
        
        return view('settings.security', compact('user', 'sessions'));
    }
    
    /**
     * Update password
     */
    public function updatePassword(Request $request)
    {
        $user = Auth::user();
        
        $request->validate([
            'current_password' => 'required',
            'new_password' => ['required', 'confirmed', Password::defaults()],
        ]);
        
        if (!Hash::check($request->current_password, $user->password)) {
            return back()->withErrors(['current_password' => 'Current password is incorrect']);
        }
        
        $user->update([
            'password' => Hash::make($request->new_password)
        ]);
        
        return back()->with('success', 'Password updated successfully');
    }
    
    /**
     * Setup or update transaction PIN
     */
    public function updateTransactionPin(Request $request)
    {
        $user = Auth::user();
        
        $request->validate([
            'current_pin' => $user->transaction_pin ? 'required|digits:4' : 'nullable',
            'new_pin' => 'required|digits:4|confirmed',
        ]);
        
        if ($user->transaction_pin && !Hash::check($request->current_pin, $user->transaction_pin)) {
            return back()->withErrors(['current_pin' => 'Current PIN is incorrect']);
        }
        
        $user->update([
            'transaction_pin' => Hash::make($request->new_pin)
        ]);
        
        return back()->with('success', 'Transaction PIN updated successfully');
    }
    
    /**
     * Show notification settings
     */
    public function notifications()
    {
        $user = Auth::user();
        
        return view('settings.notifications', compact('user'));
    }
    
    /**
     * Update notification preferences
     */
    public function updateNotifications(Request $request)
    {
        $user = Auth::user();
        
        $preferences = [
            'email_notifications' => $request->boolean('email_notifications'),
            'sms_notifications' => $request->boolean('sms_notifications'),
            'push_notifications' => $request->boolean('push_notifications'),
            'transaction_alerts' => $request->boolean('transaction_alerts'),
            'kyc_updates' => $request->boolean('kyc_updates'),
            'promotional_emails' => $request->boolean('promotional_emails'),
        ];
        
        $user->update(['notification_preferences' => $preferences]);
        
        return back()->with('success', 'Notification preferences updated successfully');
    }
    
    /**
     * Show API settings
     */
    public function api()
    {
        $user = Auth::user();
        $apiKeys = $user->tokens ?? collect();
        
        return view('settings.api', compact('user', 'apiKeys'));
    }
    
    /**
     * Generate new API key
     */
    public function generateApiKey(Request $request)
    {
        $user = Auth::user();
        
        $request->validate([
            'name' => 'required|string|max:255',
            'permissions' => 'required|array',
        ]);
        
        $token = $user->createToken(
            $request->name,
            $request->permissions
        );
        
        return back()->with([
            'success' => 'API key generated successfully',
            'token' => $token->plainTextToken
        ]);
    }
    
    /**
     * Revoke API key
     */
    public function revokeApiKey(Request $request, $tokenId)
    {
        $user = Auth::user();
        
        $user->tokens()->where('id', $tokenId)->delete();
        
        return back()->with('success', 'API key revoked successfully');
    }
    
    /**
     * Get active sessions
     */
    private function getActiveSessions($user)
    {
        // This is a simplified implementation
        // In production, you'd track sessions in database
        return [
            [
                'id' => session()->getId(),
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
                'last_activity' => now(),
                'is_current' => true
            ]
        ];
    }
    
    /**
     * Logout from all devices
     */
    public function logoutAllDevices(Request $request)
    {
        $user = Auth::user();
        
        // Revoke all tokens
        $user->tokens()->delete();
        
        // In production, you'd also invalidate all sessions
        
        return back()->with('success', 'Logged out from all devices successfully');
    }
}