<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\VirtualCard;
use App\Services\AnchorService;

/**
 * Virtual Card Web Controller
 * 
 * TODO: Add comprehensive input validation
 * TODO: Add CSRF protection for all forms
 * TODO: Add card security features
 * TODO: Add spending analytics
 */
class VirtualCardController extends Controller
{
    protected $anchorService;

    public function __construct(AnchorService $anchorService)
    {
        $this->anchorService = $anchorService;
    }

    /**
     * Display virtual cards page
     * 
     * TODO: Add pagination for large card lists
     * TODO: Add card filtering options
     */
    public function index(Request $request)
    {
        try {
            $user = $request->user();
            
            $cards = VirtualCard::where('user_id', $user->id)
                ->orderBy('created_at', 'desc')
                ->paginate(10);

            // Mask sensitive card details
            foreach ($cards as $card) {
                $card->card_number = $this->maskCardNumber($card->card_number);
                $card->makeHidden(['cvv', 'expiry_month', 'expiry_year']);
            }

            return view('virtual-cards.index', compact('cards'));

        } catch (\Exception $e) {
            // TODO: Add proper error logging
            return back()->with('error', 'Failed to load virtual cards');
        }
    }

    /**
     * Show create virtual card form
     * 
     * TODO: Add card type selection
     * TODO: Add spending limit configuration
     */
    public function create()
    {
        try {
            $cardTypes = [
                'debit' => 'Debit Card',
                'prepaid' => 'Prepaid Card'
            ];

            $currencies = [
                'NGN' => 'Nigerian Naira',
                'USD' => 'US Dollar'
            ];

            return view('virtual-cards.create', compact('cardTypes', 'currencies'));

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to load create form');
        }
    }

    /**
     * Store new virtual card
     * 
     * TODO: Add comprehensive validation
     * TODO: Add card creation limits
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'card_type' => 'required|string|in:debit,prepaid',
                'currency' => 'required|string|in:NGN,USD',
                'spending_limit' => 'nullable|numeric|min:1000|max:1000000',
                'pin' => 'required|string|size:4|regex:/^[0-9]+$/',
                'confirm_pin' => 'required|same:pin',
            ]);

            if ($validator->fails()) {
                return back()
                    ->withErrors($validator)
                    ->withInput();
            }

            $user = $request->user();

            // TODO: Check user's card creation limits
            // TODO: Verify user's wallet balance

            // Create virtual card through service
            $cardData = [
                'user_id' => $user->id,
                'card_type' => $request->card_type,
                'currency' => $request->currency,
                'spending_limit' => $request->spending_limit ?? 100000,
                'email' => $user->email,
                'phone' => $user->phone,
                'name' => $user->first_name . ' ' . $user->last_name
            ];

            $response = $this->anchorService->createVirtualCard($cardData);

            if (!$response['success']) {
                return back()
                    ->with('error', 'Failed to create virtual card: ' . $response['message'])
                    ->withInput();
            }

            // Save card to database
            $virtualCard = VirtualCard::create([
                'user_id' => $user->id,
                'card_number' => $response['data']['card_number'],
                'card_name' => $response['data']['card_name'],
                'cvv' => $response['data']['cvv'],
                'expiry_month' => $response['data']['expiry_month'],
                'expiry_year' => $response['data']['expiry_year'],
                'card_type' => $request->card_type,
                'currency' => $request->currency,
                'spending_limit' => $request->spending_limit ?? 100000,
                'balance' => 0,
                'status' => 'active',
                'provider' => 'anchor',
                'provider_reference' => $response['data']['card_id']
            ]);

            // TODO: Hash and store PIN securely
            // TODO: Send card creation notification

            return redirect()
                ->route('cards.index')
                ->with('success', 'Virtual card created successfully');

        } catch (\Exception $e) {
            return back()
                ->with('error', 'Failed to create virtual card')
                ->withInput();
        }
    }

    /**
     * Show virtual card details
     * 
     * TODO: Add transaction history
     * TODO: Add spending analytics
     */
    public function show(Request $request, $cardId)
    {
        try {
            $user = $request->user();
            
            $card = VirtualCard::where('id', $cardId)
                ->where('user_id', $user->id)
                ->first();

            if (!$card) {
                return redirect()
                    ->route('cards.index')
                    ->with('error', 'Virtual card not found');
            }

            // Mask sensitive data for display
            $displayCard = $card->replicate();
            $displayCard->card_number = $this->maskCardNumber($card->card_number);
            $displayCard->makeHidden(['cvv', 'expiry_month', 'expiry_year']);

            // TODO: Get transaction history for this card
            $transactions = collect(); // Placeholder

            return view('virtual-cards.show', compact('displayCard', 'transactions'));

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to load card details');
        }
    }

    /**
     * Freeze virtual card
     * 
     * TODO: Add confirmation modal
     * TODO: Add freeze reason tracking
     */
    public function freeze(Request $request, $cardId)
    {
        try {
            $user = $request->user();
            
            $card = VirtualCard::where('id', $cardId)
                ->where('user_id', $user->id)
                ->first();

            if (!$card) {
                return back()->with('error', 'Virtual card not found');
            }

            if ($card->status === 'frozen') {
                return back()->with('error', 'Card is already frozen');
            }

            // Freeze card through provider
            $response = $this->anchorService->freezeCard($card->provider_reference);

            if (!$response['success']) {
                return back()->with('error', 'Failed to freeze card: ' . $response['message']);
            }

            $card->update(['status' => 'frozen']);

            // TODO: Log card freeze event
            // TODO: Send freeze notification

            return back()->with('success', 'Virtual card frozen successfully');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to freeze card');
        }
    }

    /**
     * Unfreeze virtual card
     */
    public function unfreeze(Request $request, $cardId)
    {
        try {
            $user = $request->user();
            
            $card = VirtualCard::where('id', $cardId)
                ->where('user_id', $user->id)
                ->first();

            if (!$card) {
                return back()->with('error', 'Virtual card not found');
            }

            if ($card->status !== 'frozen') {
                return back()->with('error', 'Card is not frozen');
            }

            // Unfreeze card through provider
            $response = $this->anchorService->unfreezeCard($card->provider_reference);

            if (!$response['success']) {
                return back()->with('error', 'Failed to unfreeze card: ' . $response['message']);
            }

            $card->update(['status' => 'active']);

            // TODO: Log card unfreeze event
            // TODO: Send unfreeze notification

            return back()->with('success', 'Virtual card unfrozen successfully');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to unfreeze card');
        }
    }

    /**
     * Show card settings form
     * 
     * TODO: Add spending limit configuration
     * TODO: Add security settings
     */
    public function settings(Request $request, $cardId)
    {
        try {
            $user = $request->user();
            
            $card = VirtualCard::where('id', $cardId)
                ->where('user_id', $user->id)
                ->first();

            if (!$card) {
                return redirect()
                    ->route('cards.index')
                    ->with('error', 'Virtual card not found');
            }

            $displayCard = $card->replicate();
            $displayCard->card_number = $this->maskCardNumber($card->card_number);
            $displayCard->makeHidden(['cvv', 'expiry_month', 'expiry_year']);

            return view('virtual-cards.settings', compact('displayCard'));

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to load card settings');
        }
    }

    /**
     * Update card settings
     * 
     * TODO: Add spending limit validation
     * TODO: Add security settings updates
     */
    public function updateSettings(Request $request, $cardId)
    {
        try {
            $validator = Validator::make($request->all(), [
                'spending_limit' => 'required|numeric|min:1000|max:1000000',
                'status' => 'required|string|in:active,frozen',
            ]);

            if ($validator->fails()) {
                return back()
                    ->withErrors($validator)
                    ->withInput();
            }

            $user = $request->user();
            
            $card = VirtualCard::where('id', $cardId)
                ->where('user_id', $user->id)
                ->first();

            if (!$card) {
                return back()->with('error', 'Virtual card not found');
            }

            // Update spending limit with provider if changed
            if ($card->spending_limit != $request->spending_limit) {
                $response = $this->anchorService->updateSpendingLimit(
                    $card->provider_reference,
                    $request->spending_limit
                );

                if (!$response['success']) {
                    return back()->with('error', 'Failed to update spending limit: ' . $response['message']);
                }
            }

            $card->update([
                'spending_limit' => $request->spending_limit,
                'status' => $request->status,
            ]);

            // TODO: Log settings update

            return back()->with('success', 'Card settings updated successfully');

        } catch (\Exception $e) {
            return back()
                ->with('error', 'Failed to update card settings')
                ->withInput();
        }
    }

    /**
     * Show card termination confirmation
     * 
     * TODO: Add termination warnings
     * TODO: Add balance refund information
     */
    public function confirmTerminate(Request $request, $cardId)
    {
        try {
            $user = $request->user();
            
            $card = VirtualCard::where('id', $cardId)
                ->where('user_id', $user->id)
                ->first();

            if (!$card) {
                return redirect()
                    ->route('cards.index')
                    ->with('error', 'Virtual card not found');
            }

            $displayCard = $card->replicate();
            $displayCard->card_number = $this->maskCardNumber($card->card_number);
            $displayCard->makeHidden(['cvv', 'expiry_month', 'expiry_year']);

            return view('virtual-cards.terminate', compact('displayCard'));

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to load termination form');
        }
    }

    /**
     * Terminate virtual card
     * 
     * TODO: Add confirmation requirement
     * TODO: Add balance refund process
     */
    public function terminate(Request $request, $cardId)
    {
        try {
            $user = $request->user();
            
            $card = VirtualCard::where('id', $cardId)
                ->where('user_id', $user->id)
                ->first();

            if (!$card) {
                return back()->with('error', 'Virtual card not found');
            }

            if ($card->status === 'terminated') {
                return back()->with('error', 'Card is already terminated');
            }

            // TODO: Check for pending transactions
            // TODO: Process balance refund

            // Terminate card through provider
            $response = $this->anchorService->terminateCard($card->provider_reference);

            if (!$response['success']) {
                return back()->with('error', 'Failed to terminate card: ' . $response['message']);
            }

            $card->update(['status' => 'terminated']);

            // TODO: Log card termination
            // TODO: Send termination notification

            return redirect()
                ->route('cards.index')
                ->with('success', 'Virtual card terminated successfully');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to terminate card');
        }
    }

    /**
     * Mask card number for security
     */
    private function maskCardNumber($cardNumber)
    {
        $cardNumber = str_replace(' ', '', $cardNumber);
        return substr($cardNumber, 0, 4) . ' **** **** ' . substr($cardNumber, -4);
    }
}