<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ApiConfig;
use App\Models\AuditLog;
use App\Models\Admin;
use App\Models\Notification;
use App\Models\Page;
use App\Models\Setting;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SettingsController extends Controller
{
    /**
     * Display fees and limits settings.
     */
    public function fees()
    {
        // Get settings from database with defaults
        $settings = [
            'fees' => [
                'transfer_fee_type' => setting('fees.transfer.type', 'percentage'),
                'transfer_fee_value' => setting('fees.transfer.value', 1.5),
                'transfer_flat_fee' => setting('fees.transfer.flat', 10.00),
                'withdrawal_fee_type' => setting('fees.withdrawal.type', 'flat'),
                'withdrawal_fee_value' => setting('fees.withdrawal.value', 25.00),
                'card_issuance_fee' => setting('fees.card.issuance', 500.00),
                'card_maintenance_fee' => setting('fees.card.maintenance', 50.00),
                'virtual_account_fee' => setting('fees.virtual_account', 0.00),
            ],
            'limits' => [
                'daily_transfer_limit' => setting('limits.transfer.daily', 500000),
                'single_transfer_limit' => setting('limits.transfer.single', 100000),
                'daily_withdrawal_limit' => setting('limits.withdrawal.daily', 200000),
                'single_withdrawal_limit' => setting('limits.withdrawal.single', 50000),
                'daily_airtime_limit' => setting('limits.airtime.daily', 10000),
                'single_airtime_limit' => setting('limits.airtime.single', 5000),
                'card_daily_limit' => setting('limits.card.daily', 100000),
                'card_monthly_limit' => setting('limits.card.monthly', 1000000),
                'minimum_wallet_balance' => setting('limits.wallet.minimum', 100),
            ],
            'requirements' => [
                'kyc_required_for_transfers' => setting('requirements.kyc.transfers', true),
                'kyc_required_for_cards' => setting('requirements.kyc.cards', true),
                'kyc_required_amount_threshold' => setting('requirements.kyc.threshold', 50000),
                'require_2fa_for_large_transfers' => setting('requirements.2fa.large_transfers', true),
                'large_transfer_threshold' => setting('requirements.2fa.threshold', 100000),
            ]
        ];

        return view('admin.settings.fees', compact('settings'));
    }

    /**
     * Display site settings.
     */
    public function site()
    {
        $settings = [
            'business' => [
                'name' => setting('business.name', config('app.name', 'AbokiPay')),
                'tagline' => setting('business.tagline', 'Your trusted fintech partner'),
                'description' => setting('business.description', 'Leading digital financial services platform in Nigeria'),
                'logo' => setting('business.logo', '/images/logo.png'),
                'favicon' => setting('business.favicon', '/images/favicon.png'),
                'support_email' => setting('business.support_email', 'support@abokipay.ng'),
                'support_phone' => setting('business.support_phone', '+234 800 000 0000'),
                'address' => setting('business.address', '123 Business District, Lagos, Nigeria'),
            ],
            'features' => [
                'enable_wallet_funding' => setting('features.wallet_funding', true),
                'enable_p2p_transfers' => setting('features.p2p_transfers', true),
                'enable_bank_transfers' => setting('features.bank_transfers', true),
                'enable_bill_payments' => setting('features.bill_payments', true),
                'enable_virtual_cards' => setting('features.virtual_cards', true),
                'enable_airtime_data' => setting('features.airtime_data', true),
                'enable_user_registration' => setting('features.user_registration', true),
                'maintenance_mode' => setting('features.maintenance_mode', false),
            ],
            'integrations' => [
                'sms_provider' => setting('integrations.sms_provider', 'termii'),
                'email_provider' => setting('integrations.email_provider', 'mailgun'),
                'payment_gateway' => setting('integrations.payment_gateway', 'paystack'),
                'identity_provider' => setting('integrations.identity_provider', 'dojah'),
                'card_provider' => setting('integrations.card_provider', 'anchor'),
            ],
            'security' => [
                'session_lifetime' => setting('security.session_lifetime', 120),
                'password_expiry_days' => setting('security.password_expiry_days', 90),
                'max_login_attempts' => setting('security.max_login_attempts', 5),
                'lockout_duration' => setting('security.lockout_duration', 30),
                'require_email_verification' => setting('security.require_email_verification', true),
                'require_phone_verification' => setting('security.require_phone_verification', true),
            ]
        ];

        // Get available providers from API Config instead of static options
        $categories = [
            'messaging' => 'sms_providers',
            'email' => 'email_providers',
            'payment' => 'payment_gateways',
            'identity' => 'identity_providers',
            'cards' => 'card_providers',
        ];

        $apiConfigs = ApiConfig::active()
            ->whereIn('category', array_keys($categories))
            ->select('id', 'name', 'slug', 'description', 'last_test_status', 'category')
            ->get()
            ->groupBy('category');

        $availableProviders = [];
        foreach ($categories as $category => $key) {
            $availableProviders[$key] = $apiConfigs->get($category, collect());
        }
        return view('admin.settings.site', compact('settings', 'availableProviders'));
    }

    /**
     * Display notification settings.
     */
    public function notifications()
    {
        try {
            $settings = [
                'email_notifications' => [
                    'user_registration' => Setting::get('email.user_registration', true),
                    'kyc_approval' => Setting::get('email.kyc_approval', true),
                    'kyc_rejection' => Setting::get('email.kyc_rejection', true),
                    'transaction_success' => Setting::get('email.transaction_success', true),
                    'transaction_failure' => Setting::get('email.transaction_failure', true),
                    'card_issuance' => Setting::get('email.card_issuance', true),
                    'suspicious_activity' => Setting::get('email.suspicious_activity', true),
                    'password_reset' => Setting::get('email.password_reset', true),
                    'low_balance_warning' => Setting::get('email.low_balance_warning', true),
                ],
                'sms_notifications' => [
                    'transaction_alerts' => Setting::get('sms.transaction_alerts', true),
                    'otp_verification' => Setting::get('sms.otp_verification', true),
                    'security_alerts' => Setting::get('sms.security_alerts', true),
                    'balance_alerts' => Setting::get('sms.balance_alerts', false),
                    'promotional_messages' => Setting::get('sms.promotional_messages', false),
                ],
                'push_notifications' => [
                    'transaction_updates' => Setting::get('push.transaction_updates', true),
                    'system_maintenance' => Setting::get('push.system_maintenance', true),
                    'new_features' => Setting::get('push.new_features', false),
                    'promotional_offers' => Setting::get('push.promotional_offers', false),
                ],
                'admin_notifications' => [
                    'new_user_registration' => Setting::get('admin.new_user_registration', true),
                    'kyc_submissions' => Setting::get('admin.kyc_submissions', true),
                    'high_value_transactions' => Setting::get('admin.high_value_transactions', true),
                    'failed_transactions' => Setting::get('admin.failed_transactions', true),
                    'system_errors' => Setting::get('admin.system_errors', true),
                    'security_incidents' => Setting::get('admin.security_incidents', true),
                ]
            ];

            // Get recent broadcast messages from database
            $broadcasts = Notification::broadcast()
                ->latest()
                ->limit(10)
                ->get()
                ->map(function ($notification) {
                    return [
                        'id' => $notification->id,
                        'title' => $notification->title,
                        'message' => $notification->message,
                        'type' => $notification->type,
                        'sent_at' => $notification->sent_at?->diffForHumans() ?? 'Not sent',
                        'recipients' => $notification->recipients,
                        'status' => $notification->sent_at ? 'sent' : 'draft'
                    ];
                })->toArray();

            return view('admin.settings.notifications', compact('settings', 'broadcasts'));
        } catch (\Exception $e) {
            \Log::error('Failed to load notification settings', [
                'admin_id' => auth('admin')->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return back()->with('error', 'Failed to load notification settings: ' . $e->getMessage());
        }
    }

    /**
     * Update notification settings.
     */
    public function updateNotificationSettings(Request $request)
    {
        $request->validate([
            'type' => 'required|in:email,sms,push,admin',
            'setting' => 'required|string',
            'enabled' => 'required|boolean',
        ]);

        try {
            $key = $request->type . '.' . $request->setting;
            
            // Get previous value before updating
            $previousValue = Setting::get($key, false);
            
            Setting::set($key, $request->enabled, 'boolean', "Notification setting for {$key}");

            // Log admin action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'update_notification_setting',
                'target_type' => Setting::class,
                'target_id' => null,
                'old_values' => json_encode([
                    'setting' => $key,
                    'enabled' => $previousValue,
                ]),
                'new_values' => json_encode([
                    'setting' => $key,
                    'enabled' => $request->enabled,
                ]),
                'metadata' => json_encode([
                    'setting_key' => $key,
                    'previous_value' => $previousValue,
                    'new_value' => $request->enabled,
                    'admin_name' => auth('admin')->user()->name,
                ]),
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Notification setting updated successfully.'
            ]);
        } catch (\Exception $e) {
            \Log::error('Failed to update notification settings', [
                'admin_id' => auth('admin')->id(),
                'setting' => $request->type . '.' . $request->setting,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to update notification setting: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display security logs.
     */
    public function securityLogs(Request $request)
    {
        $query = AuditLog::with(['admin', 'user']);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('action', 'like', "%{$search}%")
                  ->orWhere('target_type', 'like', "%{$search}%")
                  ->orWhere('ip_address', 'like', "%{$search}%")
                  ->orWhereHas('admin', function($adminQuery) use ($search) {
                      $adminQuery->where('name', 'like', "%{$search}%")
                               ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('action')) {
            $query->where('action', $request->action);
        }

        if ($request->filled('admin_id')) {
            $query->where('admin_id', $request->admin_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $logs = $query->latest()->paginate(20)->withQueryString();

        // Real-time Statistics
        $stats = [
            'total_logs' => AuditLog::count(),
            'today_actions' => AuditLog::whereDate('created_at', today())->count(),
            'unique_admins' => AuditLog::whereNotNull('admin_id')->distinct('admin_id')->count(),
            'login_attempts' => AuditLog::where('action', 'like', '%login%')->count(),
        ];

        // Common actions for filter (get actual actions from database)
        $actions = AuditLog::distinct('action')->pluck('action')->sort();

        // Get all admins for filter dropdown
        $admins = Admin::where('is_active', true)
                      ->select('id', 'name', 'email', 'role')
                      ->orderBy('name')
                      ->get();

        // Calculate real activity analytics
        $activityAnalytics = $this->calculateActivityAnalytics();

        // Generate real security alerts
        $securityAlerts = $this->generateSecurityAlerts();

        return view('admin.settings.security-logs', compact(
            'logs', 
            'stats', 
            'actions', 
            'admins', 
            'activityAnalytics', 
            'securityAlerts'
        ));
    }

    /**
     * Calculate real activity analytics from audit logs.
     */
    private function calculateActivityAnalytics(): array
    {
        // Most common actions in the last 30 days
        $commonActions = AuditLog::where('created_at', '>=', now()->subDays(30))
                                ->select('action', DB::raw('count(*) as count'))
                                ->groupBy('action')
                                ->orderBy('count', 'desc')
                                ->limit(5)
                                ->get()
                                ->mapWithKeys(function ($item) {
                                    return [$item->action => $item->count];
                                });

        // Activity by time of day (last 7 days)
        $timeDistribution = AuditLog::where('created_at', '>=', now()->subDays(7))
                                   ->select(DB::raw('HOUR(created_at) as hour'), DB::raw('count(*) as count'))
                                   ->groupBy(DB::raw('HOUR(created_at)'))
                                   ->get()
                                   ->mapWithKeys(function ($item) {
                                       return [$item->hour => $item->count];
                                   });

        // Convert to time ranges
        $timeRanges = [
            'Morning (6-12)' => 0,
            'Afternoon (12-18)' => 0,
            'Evening (18-24)' => 0,
            'Night (0-6)' => 0,
        ];

        foreach ($timeDistribution as $hour => $count) {
            if ($hour >= 6 && $hour < 12) {
                $timeRanges['Morning (6-12)'] += $count;
            } elseif ($hour >= 12 && $hour < 18) {
                $timeRanges['Afternoon (12-18)'] += $count;
            } elseif ($hour >= 18 && $hour < 24) {
                $timeRanges['Evening (18-24)'] += $count;
            } else {
                $timeRanges['Night (0-6)'] += $count;
            }
        }

        // Convert to percentages
        $total = array_sum($timeRanges);
        if ($total > 0) {
            $timeRanges = array_map(function ($count) use ($total) {
                return round(($count / $total) * 100, 1);
            }, $timeRanges);
        }

        return [
            'commonActions' => $commonActions,
            'timeDistribution' => $timeRanges,
        ];
    }

    /**
     * Generate real security alerts based on audit log analysis.
     */
    private function generateSecurityAlerts(): array
    {
        $alerts = [];

        // Check for suspicious login attempts (multiple failed logins from same IP)
        $suspiciousIPs = AuditLog::where('action', 'like', '%login%')
                                ->where('created_at', '>=', now()->subHours(24))
                                ->whereJsonContains('data', ['success' => false])
                                ->select('ip_address', DB::raw('count(*) as attempts'))
                                ->groupBy('ip_address')
                                ->having('attempts', '>=', 3)
                                ->get();

        if ($suspiciousIPs->count() > 0) {
            $alerts[] = [
                'type' => 'warning',
                'title' => 'Suspicious login attempts detected',
                'message' => "Multiple failed login attempts from {$suspiciousIPs->count()} IP address(es) in the last 24 hours",
                'icon' => 'exclamation-triangle',
            ];
        } else {
            $alerts[] = [
                'type' => 'success',
                'title' => 'No suspicious activity',
                'message' => 'All login attempts from known IPs in the last 24 hours',
                'icon' => 'check-circle',
            ];
        }

        // Check for high-privilege actions
        $highPrivilegeActions = AuditLog::whereIn('action', [
                'delete_user', 'update_fees_and_limits', 'update_site_settings', 
                'send_broadcast_notification', 'clear_old_logs'
            ])
            ->where('created_at', '>=', now()->subHours(24))
            ->count();

        if ($highPrivilegeActions > 0) {
            $alerts[] = [
                'type' => 'info',
                'title' => 'High-privilege actions performed',
                'message' => "{$highPrivilegeActions} administrative action(s) in the last 24 hours",
                'icon' => 'shield-check',
            ];
        }

        // Check recent backup/audit activities
        $recentBackups = AuditLog::where('action', 'like', '%backup%')
                                ->orWhere('action', 'like', '%export%')
                                ->where('created_at', '>=', now()->subDays(7))
                                ->count();

        if ($recentBackups > 0) {
            $alerts[] = [
                'type' => 'info',
                'title' => 'Recent backup activity',
                'message' => "Audit logs backed up successfully ({$recentBackups} backup(s) this week)",
                'icon' => 'database',
            ];
        }

        return $alerts;
    }

    /**
     * Export security logs as CSV or Excel.
     */
    public function exportLogs(Request $request)
    {
        $request->validate([
            'format' => 'required|in:csv,excel',
            'date_from' => 'nullable|date',
            'date_to' => 'nullable|date|after_or_equal:date_from',
            'action' => 'nullable|string',
            'admin_id' => 'nullable|exists:admins,id',
        ]);

        try {
            // Build query with same filters as the main view
            $query = AuditLog::with(['admin', 'user']);

            if ($request->filled('action')) {
                $query->where('action', $request->action);
            }

            if ($request->filled('admin_id')) {
                $query->where('admin_id', $request->admin_id);
            }

            if ($request->filled('date_from')) {
                $query->whereDate('created_at', '>=', $request->date_from);
            }

            if ($request->filled('date_to')) {
                $query->whereDate('created_at', '<=', $request->date_to);
            }

            $logs = $query->orderBy('created_at', 'desc')->get();

            if ($logs->isEmpty()) {
                return back()->with('warning', 'No logs found for the selected criteria.');
            }

            // Prepare data for export
            $exportData = $logs->map(function ($log) {
                return [
                    'ID' => $log->id,
                    'Admin' => $log->admin ? $log->admin->name : 'System',
                    'Admin Email' => $log->admin ? $log->admin->email : 'N/A',
                    'User' => $log->user ? "{$log->user->first_name} {$log->user->last_name}" : 'N/A',
                    'Action' => $log->action,
                    'Target Type' => $log->target_type ?: 'N/A',
                    'Target ID' => $log->target_id ?: 'N/A',
                    'IP Address' => $log->ip_address,
                    'User Agent' => $log->user_agent ?: 'N/A',
                    'Old Values' => $log->old_values ? json_encode($log->old_values) : 'N/A',
                    'New Values' => $log->new_values ? json_encode($log->new_values) : 'N/A',
                    'Metadata' => $log->metadata ? json_encode($log->metadata) : 'N/A',
                    'Data' => $log->data ? json_encode($log->data) : 'N/A',
                    'Created At' => $log->created_at->format('Y-m-d H:i:s'),
                ];
            })->toArray();

            // Generate filename
            $timestamp = now()->format('Y-m-d_H-i-s');
            $filename = "security_logs_{$timestamp}";

            if ($request->format === 'csv') {
                return $this->exportAsCSV($exportData, $filename);
            } else {
                return $this->exportAsExcel($exportData, $filename);
            }

        } catch (\Exception $e) {
            Log::error('Failed to export security logs', [
                'admin_id' => auth('admin')->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'filters' => $request->all()
            ]);

            return back()->with('error', 'Failed to export logs: ' . $e->getMessage());
        }
    }

    /**
     * Export data as CSV.
     */
    private function exportAsCSV(array $data, string $filename): \Symfony\Component\HttpFoundation\StreamedResponse
    {
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}.csv\"",
        ];

        return response()->stream(function () use ($data) {
            $handle = fopen('php://output', 'w');
            
            if (!empty($data)) {
                // Add CSV headers
                fputcsv($handle, array_keys($data[0]));
                
                // Add data rows
                foreach ($data as $row) {
                    fputcsv($handle, $row);
                }
            }
            
            fclose($handle);
        }, 200, $headers);
    }

    /**
     * Export data as Excel (simple CSV with .xlsx extension for now).
     * Export data as a CSV file with Excel-friendly formatting.
     * Note: This does not generate a true Excel (.xlsx) file, but rather a CSV file with a .xlsx extension and Excel MIME type for compatibility.
     * In a production environment, you might want to use PhpSpreadsheet or similar to generate real Excel files.
     */
    private function exportAsExcel(array $data, string $filename): \Symfony\Component\HttpFoundation\StreamedResponse
    {
        // For now, export as CSV with Excel-friendly formatting
        $headers = [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}.csv\"",
        ];

        return response()->stream(function () use ($data) {
            $handle = fopen('php://output', 'w');
            
            if (!empty($data)) {
                // Add BOM for Excel UTF-8 recognition
                fputs($handle, "\xEF\xBB\xBF");
                
                // Add headers
                fputcsv($handle, array_keys($data[0]));
                
                // Add data rows
                foreach ($data as $row) {
                    fputcsv($handle, $row);
                }
            }
            
            fclose($handle);
        }, 200, $headers);
    }

    /**
     * Clear old logs (admin-only action).
     */
    public function clearOldLogs(Request $request)
    {
        // Ensure only super admins can clear logs
        if (!$this->isSuperAdmin()) {
            return response()->json([
                'success' => false,
                'message' => 'Only super administrators can clear old logs.'
            ], 403);
        }

        $request->validate([
            'date_before' => 'required|date|before:today',
            'confirm' => 'required|boolean|accepted',
        ]);

        try {
            DB::beginTransaction();

            $dateBefore = $request->date_before;
            
            // Count logs to be deleted
            $logsToDelete = AuditLog::whereDate('created_at', '<', $dateBefore)->count();
            
            if ($logsToDelete === 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'No logs found before the specified date.'
                ]);
            }

            // Delete old logs
            $deletedCount = AuditLog::whereDate('created_at', '<', $dateBefore)->delete();

            DB::commit();

            // Log this action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'clear_old_logs',
                'target_type' => 'AuditLog',
                'target_id' => null,
                'data' => [
                    'date_before' => $dateBefore,
                    'deleted_count' => $deletedCount,
                    'admin_name' => auth('admin')->user()->name,
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            return response()->json([
                'success' => true,
                'message' => "Successfully deleted {$deletedCount} log records before {$dateBefore}."
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Failed to clear old logs', [
                'admin_id' => auth('admin')->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'date_before' => $request->date_before
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to clear old logs: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get log details for modal view.
     */
    public function getLogDetails(Request $request, $logId)
    {
        try {
            $log = AuditLog::with(['admin', 'user'])->findOrFail($logId);

            return response()->json([
                'success' => true,
                'log' => [
                    'id' => $log->id,
                    'admin' => $log->admin ? [
                        'name' => $log->admin->name,
                        'email' => $log->admin->email,
                        'role' => $log->admin->role,
                    ] : null,
                    'user' => $log->user ? [
                        'name' => "{$log->user->first_name} {$log->user->last_name}",
                        'email' => $log->user->email,
                    ] : null,
                    'action' => $log->action,
                    'target_type' => $log->target_type,
                    'target_id' => $log->target_id,
                    'ip_address' => $log->ip_address,
                    'user_agent' => $log->user_agent,
                    'old_values' => $log->old_values,
                    'new_values' => $log->new_values,
                    'metadata' => $log->metadata,
                    'data' => $log->data,
                    'created_at' => $log->created_at->format('Y-m-d H:i:s'),
                    'formatted_date' => $log->created_at->format('M j, Y g:i A'),
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Log not found or access denied.'
            ], 404);
        }
    }

    /**
     * Block IP address (if implemented).
     */
    public function blockIP(Request $request)
    {
        $request->validate([
            'ip_address' => 'required|ip',
        ]);

        try {
            // This would typically add the IP to a blocked IPs table or firewall rules
            // For now, we'll just log the action
            
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'block_ip_address',
                'target_type' => 'Security',
                'target_id' => null,
                'data' => [
                    'blocked_ip' => $request->ip_address,
                    'admin_name' => auth('admin')->user()->name,
                    'reason' => 'Suspicious activity',
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            return response()->json([
                'success' => true,
                'message' => "IP address {$request->ip_address} has been blocked."
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to block IP address', [
                'admin_id' => auth('admin')->id(),
                'ip_to_block' => $request->ip_address,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to block IP address: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display pages management.
     */
    public function pages()
    {
        $pages = Page::with(['creator', 'updater'])
            ->latest()
            ->paginate(10);

        return view('admin.settings.pages', compact('pages'));
    }

    /**
     * Store a new page.
     */
    public function storePage(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:pages,slug',
            'content' => 'required|string',
            'excerpt' => 'nullable|string|max:500',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'status' => 'required|in:draft,published,archived',
            'is_featured' => 'boolean',
        ]);

        try {
            DB::beginTransaction();

            $page = Page::create([
                'title' => $request->title,
                'slug' => $request->slug,
                'content' => $request->content,
                'excerpt' => $request->excerpt,
                'meta_title' => $request->meta_title,
                'meta_description' => $request->meta_description,
                'status' => $request->status,
                'published' => ($request->status === 'published'),
                'is_featured' => $request->boolean('is_featured'),
                'created_by' => auth('admin')->id(),
                'updated_by' => auth('admin')->id(),
                'published_at' => $request->status === 'published' ? now() : null,
            ]);

            DB::commit();

            // Log admin action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'create_page',
                'target_type' => 'Page',
                'target_id' => $page->id,
                'new_values' => $page->toArray(),
                'metadata' => [
                    'page_title' => $page->title,
                    'page_status' => $page->status,
                    'admin_name' => auth('admin')->user()->name,
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            return back()->with('success', 'Page created successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Failed to create page', [
                'admin_id' => auth('admin')->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);

            return back()->with('error', 'Failed to create page: ' . $e->getMessage());
        }
    }

    /**
     * Update an existing page.
     */
    public function updatePage(Request $request, Page $page)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:pages,slug,' . $page->id,
            'content' => 'required|string',
            'excerpt' => 'nullable|string|max:500',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'status' => 'required|in:draft,published,archived',
            'is_featured' => 'boolean',
        ]);

        try {
            DB::beginTransaction();

            $oldValues = $page->toArray();

            $page->update([
                'title' => $request->title,
                'slug' => $request->slug,
                'content' => $request->content,
                'excerpt' => $request->excerpt,
                'meta_title' => $request->meta_title,
                'meta_description' => $request->meta_description,
                'status' => $request->status,
                'published' => ($request->status === 'published'),
                'is_featured' => $request->boolean('is_featured'),
                'updated_by' => auth('admin')->id(),
                'published_at' => $request->status === 'published' && !$page->published_at ? now() : $page->published_at,
            ]);

            DB::commit();

            // Log admin action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'update_page',
                'target_type' => 'Page',
                'target_id' => $page->id,
                'old_values' => $oldValues,
                'new_values' => $page->fresh()->toArray(),
                'metadata' => [
                    'page_title' => $page->title,
                    'page_status' => $page->status,
                    'admin_name' => auth('admin')->user()->name,
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            return back()->with('success', 'Page updated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Failed to update page', [
                'admin_id' => auth('admin')->id(),
                'page_id' => $page->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);

            return back()->with('error', 'Failed to update page: ' . $e->getMessage());
        }
    }

    /**
     * Delete a page.
     */
    public function destroyPage(Page $page)
    {
        try {
            DB::beginTransaction();

            $pageData = $page->toArray();
            $page->delete();

            DB::commit();

            // Log admin action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'delete_page',
                'target_type' => 'Page',
                'target_id' => $pageData['id'],
                'old_values' => $pageData,
                'metadata' => [
                    'page_title' => $pageData['title'],
                    'admin_name' => auth('admin')->user()->name,
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            return response()->json(['success' => true, 'message' => 'Page deleted successfully.']);

        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Failed to delete page', [
                'admin_id' => auth('admin')->id(),
                'page_id' => $page->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json(['success' => false, 'message' => 'Failed to delete page: ' . $e->getMessage()]);
        }
    }

    /**
     * Show a specific page for editing.
     */
    public function showPage(Page $page)
    {
        return view('admin.settings.pages-view', compact('page'));
    }

    /**
     * Get page data for editing (AJAX).
     */
    public function editPage(Page $page)
    {
        try {
            return response()->json([
                'success' => true,
                'page' => $page->toArray()
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading page data: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Toggle page status (publish/unpublish).
     */
    public function togglePageStatus(Request $request, Page $page)
    {
        $request->validate([
            'status' => 'required|in:draft,published,archived'
        ]);

        try {
            DB::beginTransaction();

            $oldStatus = $page->status;
            $newStatus = $request->status;

            $page->update([
                'status' => $newStatus,
                'published' => ($newStatus === 'published'),
                'published_at' => ($newStatus === 'published') ? now() : null,
                'updated_by' => auth('admin')->id(),
            ]);

            DB::commit();

            // Log admin action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'toggle_page_status',
                'target_type' => 'Page',
                'target_id' => $page->id,
                'old_values' => ['status' => $oldStatus],
                'new_values' => ['status' => $newStatus],
                'metadata' => [
                    'page_title' => $page->title,
                    'old_status' => $oldStatus,
                    'new_status' => $newStatus,
                    'admin_name' => auth('admin')->user()->name,
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            return response()->json([
                'success' => true,
                'message' => "Page {$newStatus} successfully.",
                'status' => $newStatus
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Failed to toggle page status', [
                'admin_id' => auth('admin')->id(),
                'page_id' => $page->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to update page status: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Duplicate a page.
     */
    public function duplicatePage(Page $page)
    {
        try {
            DB::beginTransaction();

            $duplicatedPage = Page::create([
                'title' => $page->title . ' (Copy)',
                'slug' => null, // Will be auto-generated
                'content' => $page->content,
                'excerpt' => $page->excerpt,
                'meta_title' => $page->meta_title,
                'meta_description' => $page->meta_description,
                'status' => 'draft', // Always create as draft
                'published' => false,
                'is_featured' => false,
                'metadata' => $page->metadata,
                'created_by' => auth('admin')->id(),
                'updated_by' => auth('admin')->id(),
                'published_at' => null,
            ]);

            DB::commit();

            // Log admin action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'duplicate_page',
                'target_type' => 'Page',
                'target_id' => $duplicatedPage->id,
                'new_values' => $duplicatedPage->toArray(),
                'metadata' => [
                    'original_page_id' => $page->id,
                    'original_page_title' => $page->title,
                    'new_page_title' => $duplicatedPage->title,
                    'admin_name' => auth('admin')->user()->name,
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Page duplicated successfully.',
                'page' => $duplicatedPage
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Failed to duplicate page', [
                'admin_id' => auth('admin')->id(),
                'page_id' => $page->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to duplicate page: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Export pages to CSV.
     */
    public function exportPages()
    {
        try {
            $pages = Page::with(['creator', 'updater'])->get();

            $filename = 'pages_export_' . now()->format('Y_m_d_H_i_s') . '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => 'attachment; filename="' . $filename . '"',
            ];

            $callback = function() use ($pages) {
                $file = fopen('php://output', 'w');
                
                // CSV headers
                fputcsv($file, [
                    'ID',
                    'Title',
                    'Slug',
                    'Status',
                    'Published',
                    'Featured',
                    'Created By',
                    'Updated By',
                    'Created At',
                    'Updated At',
                    'Published At',
                    'Meta Title',
                    'Meta Description',
                    'Excerpt'
                ]);

                // CSV data
                foreach ($pages as $page) {
                    fputcsv($file, [
                        $page->id,
                        $page->title,
                        $page->slug,
                        $page->status,
                        $page->published ? 'Yes' : 'No',
                        $page->is_featured ? 'Yes' : 'No',
                        $page->creator ? $page->creator->name : '',
                        $page->updater ? $page->updater->name : '',
                        $page->created_at ? $page->created_at->format('Y-m-d H:i:s') : '',
                        $page->updated_at ? $page->updated_at->format('Y-m-d H:i:s') : '',
                        $page->published_at ? $page->published_at->format('Y-m-d H:i:s') : '',
                        $page->meta_title,
                        $page->meta_description,
                        $page->excerpt
                    ]);
                }

                fclose($file);
            };

            // Log admin action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'export_pages',
                'target_type' => 'Page',
                'metadata' => [
                    'pages_count' => $pages->count(),
                    'admin_name' => auth('admin')->user()->name,
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            return response()->stream($callback, 200, $headers);

        } catch (\Exception $e) {
            Log::error('Failed to export pages', [
                'admin_id' => auth('admin')->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return back()->with('error', 'Failed to export pages: ' . $e->getMessage());
        }
    }

    /**
     * Update site settings.
     */
    public function updateSiteSettings(Request $request)
    {
        $request->validate([
            // Business info validation
            'business_name' => 'required|string|max:255',
            'business_tagline' => 'nullable|string|max:255',
            'business_description' => 'nullable|string|max:1000',
            'logo_url' => 'nullable|string|max:500',
            'favicon_url' => 'nullable|string|max:500',
            'support_email' => 'required|email|max:255',
            'support_phone' => 'required|string|max:20',
            'business_address' => 'nullable|string|max:500',
            
            // Feature toggles validation (checkboxes)
            'features.enable_wallet_funding' => 'nullable|boolean',
            'features.enable_p2p_transfers' => 'nullable|boolean',
            'features.enable_bank_transfers' => 'nullable|boolean',
            'features.enable_bill_payments' => 'nullable|boolean',
            'features.enable_virtual_cards' => 'nullable|boolean',
            'features.enable_airtime_data' => 'nullable|boolean',
            'features.enable_user_registration' => 'nullable|boolean',
            'features.maintenance_mode' => 'nullable|boolean',
            
            // Integration settings validation
            'integrations.sms_provider' => 'required|string|max:50',
            'integrations.email_provider' => 'required|string|max:50',
            'integrations.payment_gateway' => 'required|string|max:50',
            'integrations.identity_provider' => 'required|string|max:50',
            'integrations.card_provider' => 'required|string|max:50',
            
            // Security settings validation
            'security.session_lifetime' => 'required|integer|min:30|max:1440',
            'security.password_expiry_days' => 'required|integer|min:0|max:365',
            'security.max_login_attempts' => 'required|integer|min:3|max:10',
            'security.lockout_duration' => 'required|integer|min:5|max:1440',
            'security.require_email_verification' => 'nullable|boolean',
            'security.require_phone_verification' => 'nullable|boolean',
        ]);

        try {
            DB::beginTransaction();

            // Store old values for audit
            $oldValues = [
                'business.name' => Setting::get('business.name'),
                'business.logo' => Setting::get('business.logo'),
                'business.favicon' => Setting::get('business.favicon'),
            ];

            // Save business info settings
            Setting::set('business.name', $request->business_name, 'string', 'Business name');
            Setting::set('business.tagline', $request->business_tagline, 'string', 'Business tagline');
            Setting::set('business.description', $request->business_description, 'string', 'Business description');
            Setting::set('business.logo', $request->logo_url, 'string', 'Business logo path');
            Setting::set('business.favicon', $request->favicon_url, 'string', 'Business favicon path');
            Setting::set('business.support_email', $request->support_email, 'string', 'Support email address');
            Setting::set('business.support_phone', $request->support_phone, 'string', 'Support phone number');
            Setting::set('business.address', $request->business_address, 'string', 'Business address');

            // Save feature settings (handle checkboxes properly)
            $features = $request->input('features', []);
            Setting::set('features.wallet_funding', isset($features['enable_wallet_funding']), 'boolean', 'Enable wallet funding');
            Setting::set('features.p2p_transfers', isset($features['enable_p2p_transfers']), 'boolean', 'Enable P2P transfers');
            Setting::set('features.bank_transfers', isset($features['enable_bank_transfers']), 'boolean', 'Enable bank transfers');
            Setting::set('features.bill_payments', isset($features['enable_bill_payments']), 'boolean', 'Enable bill payments');
            Setting::set('features.virtual_cards', isset($features['enable_virtual_cards']), 'boolean', 'Enable virtual cards');
            Setting::set('features.airtime_data', isset($features['enable_airtime_data']), 'boolean', 'Enable airtime/data');
            Setting::set('features.user_registration', isset($features['enable_user_registration']), 'boolean', 'Enable user registration');
            Setting::set('features.maintenance_mode', isset($features['maintenance_mode']), 'boolean', 'Maintenance mode');

            // Save integration settings
            $integrations = $request->input('integrations', []);
            Setting::set('integrations.sms_provider', $integrations['sms_provider'] ?? '', 'string', 'SMS provider');
            Setting::set('integrations.email_provider', $integrations['email_provider'] ?? '', 'string', 'Email provider');
            Setting::set('integrations.payment_gateway', $integrations['payment_gateway'] ?? '', 'string', 'Payment gateway');
            Setting::set('integrations.identity_provider', $integrations['identity_provider'] ?? '', 'string', 'Identity provider');
            Setting::set('integrations.card_provider', $integrations['card_provider'] ?? '', 'string', 'Card provider');

            // Save security settings
            $security = $request->input('security', []);
            Setting::set('security.session_lifetime', $security['session_lifetime'] ?? 120, 'integer', 'Session lifetime (minutes)');
            Setting::set('security.password_expiry_days', $security['password_expiry_days'] ?? 90, 'integer', 'Password expiry days');
            Setting::set('security.max_login_attempts', $security['max_login_attempts'] ?? 5, 'integer', 'Max login attempts');
            Setting::set('security.lockout_duration', $security['lockout_duration'] ?? 30, 'integer', 'Lockout duration (minutes)');
            Setting::set('security.require_email_verification', isset($security['require_email_verification']), 'boolean', 'Require email verification');
            Setting::set('security.require_phone_verification', isset($security['require_phone_verification']), 'boolean', 'Require phone verification');

            // Clear settings cache to ensure instant updates across the site
            Setting::clearCache();
            
            // Clear Laravel's config cache as well to ensure app.name changes take effect
            \Artisan::call('config:clear');

            DB::commit();

            // Log admin action with detailed changes
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'update_site_settings',
                'target_type' => 'Settings',
                'target_id' => null,
                'old_values' => $oldValues,
                'new_values' => $request->all(),
                'metadata' => [
                    'section' => 'site_settings',
                    'admin_name' => auth('admin')->user()->name,
                    'logo_changed' => $oldValues['business.logo'] !== $request->logo_url,
                    'name_changed' => $oldValues['business.name'] !== $request->business_name,
                    'favicon_changed' => $oldValues['business.favicon'] !== $request->favicon_url,
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            // Return JSON response for AJAX requests
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Site settings updated successfully. Changes are now active across the platform.'
                ]);
            }

            return back()->with('success', 'Site settings updated successfully. Changes are now active across the platform.');

        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Failed to update site settings', [
                'admin_id' => auth('admin')->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to update site settings: ' . $e->getMessage()
                ], 500);
            }

            return back()->with('error', 'Failed to update site settings: ' . $e->getMessage());
        }
    }
    /**
     * Update the various fee and limit settings for the platform.
     *
     * Validates and updates transfer, withdrawal, card, and account fees, as well as transaction limits and KYC requirements.
     *
     * @param  \Illuminate\Http\Request  $request  The HTTP request containing the fee and limit data.
     * @return \Illuminate\Http\RedirectResponse    Redirects back with a success or error message.
     */
    public function updateFees(Request $request)
    {
        $request->validate([
            // Fee validation rules
            'transfer_fee_type' => 'required|in:percentage,flat',
            'transfer_fee_value' => 'required|numeric|min:0|max:100',
            'transfer_flat_fee' => 'nullable|numeric|min:0',
            'withdrawal_fee_type' => 'required|in:percentage,flat',
            'withdrawal_fee_value' => 'required|numeric|min:0',
            'card_issuance_fee' => 'required|numeric|min:0',
            'card_maintenance_fee' => 'required|numeric|min:0',
            'virtual_account_fee' => 'required|numeric|min:0',
            
            // Limit validation rules
            'daily_transfer_limit' => 'required|numeric|min:1000',
            'single_transfer_limit' => 'required|numeric|min:100',
            'daily_withdrawal_limit' => 'required|numeric|min:1000',
            'single_withdrawal_limit' => 'required|numeric|min:100',
            'daily_airtime_limit' => 'required|numeric|min:100',
            'single_airtime_limit' => 'required|numeric|min:10',
            'card_daily_limit' => 'required|numeric|min:1000',
            'card_monthly_limit' => 'required|numeric|min:10000',
            'minimum_wallet_balance' => 'required|numeric|min:0',
            
            // Requirements validation rules
            'kyc_required_for_transfers' => 'in:0,1',
            'kyc_required_for_cards' => 'in:0,1',
            'kyc_required_amount_threshold' => 'required|numeric|min:0',
            'require_2fa_for_large_transfers' => 'in:0,1',
            'large_transfer_threshold' => 'required|numeric|min:0',
        ]);

        try {
            DB::beginTransaction();

            // Convert checkbox values to booleans
            $kycForTransfers = $request->input('kyc_required_for_transfers') === '1';
            $kycForCards = $request->input('kyc_required_for_cards') === '1';
            $require2FA = $request->input('require_2fa_for_large_transfers') === '1';

            // Save fee settings
            Setting::set('fees.transfer.type', $request->transfer_fee_type, 'string', 'Transfer fee calculation type');
            Setting::set('fees.transfer.value', $request->transfer_fee_value, 'float', 'Transfer fee value');
            Setting::set('fees.transfer.flat', $request->transfer_flat_fee ?? 0, 'float', 'Transfer flat fee amount');
            Setting::set('fees.withdrawal.type', $request->withdrawal_fee_type, 'string', 'Withdrawal fee calculation type');
            Setting::set('fees.withdrawal.value', $request->withdrawal_fee_value, 'float', 'Withdrawal fee value');
            Setting::set('fees.card.issuance', $request->card_issuance_fee, 'float', 'Card issuance fee');
            Setting::set('fees.card.maintenance', $request->card_maintenance_fee, 'float', 'Card maintenance fee');
            Setting::set('fees.virtual_account', $request->virtual_account_fee, 'float', 'Virtual account fee');

            // Save limit settings
            Setting::set('limits.transfer.daily', $request->daily_transfer_limit, 'integer', 'Daily transfer limit');
            Setting::set('limits.transfer.single', $request->single_transfer_limit, 'integer', 'Single transfer limit');
            Setting::set('limits.withdrawal.daily', $request->daily_withdrawal_limit, 'integer', 'Daily withdrawal limit');
            Setting::set('limits.withdrawal.single', $request->single_withdrawal_limit, 'integer', 'Single withdrawal limit');
            Setting::set('limits.airtime.daily', $request->daily_airtime_limit, 'integer', 'Daily airtime limit');
            Setting::set('limits.airtime.single', $request->single_airtime_limit, 'integer', 'Single airtime limit');
            Setting::set('limits.card.daily', $request->card_daily_limit, 'integer', 'Card daily limit');
            Setting::set('limits.card.monthly', $request->card_monthly_limit, 'integer', 'Card monthly limit');
            Setting::set('limits.wallet.minimum', $request->minimum_wallet_balance, 'float', 'Minimum wallet balance');

            // Save requirement settings with proper boolean conversion
            Setting::set('requirements.kyc.transfers', $kycForTransfers, 'boolean', 'KYC required for transfers');
            Setting::set('requirements.kyc.cards', $kycForCards, 'boolean', 'KYC required for cards');
            Setting::set('requirements.kyc.threshold', $request->kyc_required_amount_threshold, 'float', 'KYC requirement threshold amount');
            Setting::set('requirements.2fa.large_transfers', $require2FA, 'boolean', '2FA required for large transfers');
            Setting::set('requirements.2fa.threshold', $request->large_transfer_threshold, 'float', '2FA requirement threshold amount');

            // Clear settings cache to ensure instant updates
            Setting::clearCache();

            DB::commit();

            // Log admin action with detailed changes
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'update_fees_and_limits',
                'target_type' => 'Settings',
                'target_id' => null,
                'new_values' => $request->all(),
                'metadata' => [
                    'section' => 'fees_and_limits',
                    'admin_name' => auth('admin')->user()->name,
                    'changed_settings' => [
                        'fees' => [
                            'transfer_fee_type' => $request->transfer_fee_type,
                            'transfer_fee_value' => $request->transfer_fee_value,
                            'withdrawal_fee_value' => $request->withdrawal_fee_value,
                            'card_issuance_fee' => $request->card_issuance_fee,
                        ],
                        'limits' => [
                            'daily_transfer_limit' => $request->daily_transfer_limit,
                            'single_transfer_limit' => $request->single_transfer_limit,
                            'daily_withdrawal_limit' => $request->daily_withdrawal_limit,
                        ],
                        'requirements' => [
                            'kyc_required_for_transfers' => $kycForTransfers,
                            'kyc_required_for_cards' => $kycForCards,
                            'require_2fa_for_large_transfers' => $require2FA,
                        ]
                    ]
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            // Return JSON response for AJAX requests
            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Fees and limits updated successfully. Changes are now active across the platform.'
                ]);
            }

            return back()->with('success', 'Fees and limits updated successfully. Changes are now active across the platform.');

        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Failed to update fees and limits', [
                'admin_id' => auth('admin')->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);

            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to update fees and limits: ' . $e->getMessage()
                ], 500);
            }

            return back()->with('error', 'Failed to update fees and limits: ' . $e->getMessage());
        }
    }

    /**
     * Send broadcast notification.
     */
    public function sendBroadcast(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string|max:1000',
            'type' => 'required|in:info,warning,success,error',
            'recipients' => 'required|in:all_users,verified_users,unverified_users,active_users',
            'channels' => 'required|array|min:1',
            'channels.*' => 'in:in_app,email,sms',
        ]);

        try {
            DB::beginTransaction();

            // Get target users based on recipients filter
            $users = $this->getTargetUsers($request->recipients);
            $userCount = $users->count();
            
            if ($userCount === 0) {
                return back()->with('warning', 'No users found matching the selected criteria.');
            }

            // Create master broadcast notification
            $broadcastNotification = Notification::create([
                'admin_id' => auth('admin')->id(),
                'title' => $request->title,
                'message' => $request->message,
                'type' => $request->type,
                'channel' => implode(',', $request->channels),
                'recipients' => $request->recipients,
                'is_broadcast' => true,
                'sent_at' => now(),
                'metadata' => [
                    'admin_name' => auth('admin')->user()->name,
                    'sent_via' => 'admin_panel',
                    'channels' => $request->channels,
                    'target_user_count' => $userCount,
                ],
            ]);

            $results = [
                'in_app' => ['sent' => 0, 'failed' => 0],
                'email' => ['sent' => 0, 'failed' => 0],
                'sms' => ['sent' => 0, 'failed' => 0],
            ];

            // Process users in chunks to avoid memory issues
            $users->chunk(100, function ($userChunk) use ($request, $broadcastNotification, &$results) {
                foreach ($userChunk as $user) {
                    // Send in-app notification if requested
                    if (in_array('in_app', $request->channels)) {
                        try {
                            Notification::create([
                                'user_id' => $user->id,
                                'admin_id' => auth('admin')->id(),
                                'title' => $request->title,
                                'message' => $request->message,
                                'type' => $request->type,
                                'channel' => 'in_app',
                                'recipients' => $request->recipients,
                                'is_broadcast' => false,
                                'sent_at' => now(),
                                'metadata' => [
                                    'broadcast_id' => $broadcastNotification->id,
                                    'admin_name' => auth('admin')->user()->name,
                                ],
                            ]);
                            $results['in_app']['sent']++;
                        } catch (\Exception $e) {
                            Log::error('Failed to send in-app notification', [
                                'user_id' => $user->id,
                                'broadcast_id' => $broadcastNotification->id,
                                'error' => $e->getMessage()
                            ]);
                            $results['in_app']['failed']++;
                        }
                    }

                    // Send email notification if requested and user has email
                    if (in_array('email', $request->channels) && $user->email) {
                        try {
                            // Check if email notifications are enabled
                            if (Setting::get('email.system_notifications', true)) {
                                $this->sendEmailNotification($user, $request->title, $request->message);
                                $results['email']['sent']++;
                            }
                        } catch (\Exception $e) {
                            Log::error('Failed to send email notification', [
                                'user_id' => $user->id,
                                'email' => $user->email,
                                'broadcast_id' => $broadcastNotification->id,
                                'error' => $e->getMessage()
                            ]);
                            $results['email']['failed']++;
                        }
                    }

                    // Send SMS notification if requested and user has phone
                    if (in_array('sms', $request->channels) && $user->phone) {
                        try {
                            // Check if SMS notifications are enabled and provider is configured
                            if (Setting::get('sms.system_notifications', true)) {
                                $this->sendSmsNotification($user, $request->title, $request->message);
                                $results['sms']['sent']++;
                            }
                        } catch (\Exception $e) {
                            Log::error('Failed to send SMS notification', [
                                'user_id' => $user->id,
                                'phone' => $user->phone,
                                'broadcast_id' => $broadcastNotification->id,
                                'error' => $e->getMessage()
                            ]);
                            $results['sms']['failed']++;
                        }
                    }
                }
            });

            // Update broadcast notification with results
            $broadcastNotification->update([
                'metadata' => array_merge($broadcastNotification->metadata, [
                    'delivery_results' => $results,
                    'completed_at' => now(),
                ])
            ]);

            DB::commit();

            // Log admin action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'send_broadcast_notification',
                'target_type' => 'Notification',
                'target_id' => $broadcastNotification->id,
                'new_values' => [
                    'title' => $request->title,
                    'recipients' => $request->recipients,
                    'channels' => $request->channels,
                    'type' => $request->type,
                    'user_count' => $userCount,
                    'delivery_results' => $results,
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            $totalSent = array_sum(array_column($results, 'sent'));
            $totalFailed = array_sum(array_column($results, 'failed'));

            $message = "Broadcast notification sent successfully! ";
            $message .= "Delivered: {$totalSent}, Failed: {$totalFailed}";
            
            if ($totalFailed > 0) {
                $message .= " (Check logs for failure details)";
            }

            return back()->with('success', $message);

        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Failed to send broadcast notification', [
                'admin_id' => auth('admin')->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);

            return back()->with('error', 'Failed to send broadcast notification: ' . $e->getMessage());
        }
    }

    /**
     * Send email notification to user.
     */
    private function sendEmailNotification($user, $title, $message)
    {
        try {
            // Update mail config with current settings
            $this->updateMailConfig();
            
            $fromAddress = Setting::get('mail.from.address');
            $fromName = Setting::get('mail.from.name', Setting::get('business.name', 'SwiftPay'));
            
            if (!$fromAddress) {
                throw new \Exception('From email address not configured');
            }

            \Mail::raw(
                "Dear {$user->first_name},\n\n{$message}\n\nBest regards,\n{$fromName}",
                function ($mail) use ($user, $title, $fromAddress, $fromName) {
                    $mail->to($user->email, $user->first_name . ' ' . $user->last_name)
                         ->subject($title)
                         ->from($fromAddress, $fromName);
                }
            );
            
            Log::info('Email notification sent successfully', [
                'user_id' => $user->id,
                'email' => $user->email,
                'title' => $title,
                'timestamp' => now(),
            ]);
            
        } catch (\Exception $e) {
            Log::error('Failed to send email notification', [
                'user_id' => $user->id,
                'email' => $user->email,
                'title' => $title,
                'error' => $e->getMessage(),
                'smtp_configured' => Setting::get('mail.smtp.host') ? 'yes' : 'no',
            ]);
            
            throw $e;
        }
    }

    /**
     * Send SMS notification to user.
     */
    private function sendSmsNotification($user, $title, $message)
    {
        try {
            // Get SMS provider configuration
            $smsProvider = Setting::get('integrations.sms_provider', 'termii');

            // Check if SMS provider is configured
            $smsConfig = ApiConfig::where('category', 'messaging')
                ->where('slug', $smsProvider)
                ->where('is_active', true)
                ->first();

            if (!$smsConfig) {
                throw new \Exception('SMS provider not configured or inactive');
            }

            // For now, we'll simulate SMS sending and log it
            // In a real implementation, you would integrate with the actual SMS service
            $smsMessage = "{$title}: {$message}";
            
            // Log the SMS as sent (in real implementation, use actual SMS service)
            Log::info('SMS notification sent', [
                'user_id' => $user->id,
                'phone' => $user->phone,
                'message' => $smsMessage,
                'provider' => $smsProvider,
                'timestamp' => now(),
            ]);

            // In a real implementation, you would call the SMS service:
            // $smsService = app(SmsServiceFactory::class)->get($smsProvider);
            // $result = $smsService->sendSms($user->phone, $smsMessage);
            
        } catch (\Exception $e) {
            Log::error('Failed to send SMS notification', [
                'user_id' => $user->id,
                'phone' => $user->phone,
                'title' => $title,
                'error' => $e->getMessage(),
            ]);
            
            throw $e;
        }
    }

    /**
     * Get target users based on recipients filter.
     */
    private function getTargetUsers(string $recipients)
    {
        $query = User::query();

        return match($recipients) {
            'verified_users' => $query->whereNotNull('email_verified_at')->whereHas('kycVerification', function($q) {
                $q->where('status', 'approved');
            }),
            'unverified_users' => $query->whereNull('email_verified_at')->orWhereDoesntHave('kycVerification')->orWhereHas('kycVerification', function($q) {
                $q->where('status', '!=', 'approved');
            }),
            'active_users' => $query->where('is_active', true)->where('is_blocked', false),
            default => $query, // all_users
        };
    }

    /**
     * Test notification systems.
     */
    public function testNotifications()
    {
        $results = [
            'email' => $this->testEmailNotification(),
            'sms' => $this->testSmsNotification(),
            'push' => $this->testPushNotification(),
        ];

        // Log admin action
        AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'test_notification_systems',
            'target_type' => 'System',
            'target_id' => null,
            'data' => $results,
            'ip_address' => request()->ip(),
        ]);

        return response()->json([
            'success' => true,
            'results' => $results,
            'message' => 'Notification systems tested successfully.'
        ]);
    }

    /**
     * Test email notification system.
     */
    private function testEmailNotification(): array
    {
        try {
            // In a real implementation, you'd send a test email
            // For now, we'll just check configuration
            $emailConfig = config('mail');
            
            return [
                'status' => 'success',
                'message' => 'Email system is configured and ready',
                'driver' => $emailConfig['default'] ?? 'null',
                'tested_at' => now(),
            ];
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'message' => 'Email system error: ' . $e->getMessage(),
                'tested_at' => now(),
            ];
        }
    }

    /**
     * Test SMS notification system.
     */
    private function testSmsNotification(): array
    {
        try {
            // Check if SMS service is configured (Termii, etc.)
            $smsConfig = ApiConfig::where('category', 'messaging')
                ->where('is_active', true)
                ->first();
            
            if (!$smsConfig) {
                return [
                    'status' => 'warning',
                    'message' => 'No SMS service configured',
                    'tested_at' => now(),
                ];
            }

            return [
                'status' => 'success',
                'message' => 'SMS system is configured and ready',
                'provider' => $smsConfig->name,
                'tested_at' => now(),
            ];
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'message' => 'SMS system error: ' . $e->getMessage(),
                'tested_at' => now(),
            ];
        }
    }

    /**
     * Display plugin/module selection.
     */
    public function modules()
    {
        // Get enabled API configurations grouped by category
        $modules = ApiConfig::active()
            ->select('id', 'name', 'slug', 'category', 'description', 'is_live_mode', 'last_test_status')
            ->get()
            ->groupBy('category');

        // Get module configuration settings
        $moduleSettings = [
            'enabled_modules' => Setting::get('modules.enabled', []),
            'default_payment_gateway' => Setting::get('modules.default.payment_gateway'),
            'default_sms_provider' => Setting::get('modules.default.sms_provider'),
            'default_identity_provider' => Setting::get('modules.default.identity_provider'),
            'default_card_provider' => Setting::get('modules.default.card_provider'),
        ];

        return view('admin.settings.modules', compact('modules', 'moduleSettings'));
    }

    /**
     * Update module settings.
     */
    public function updateModules(Request $request)
    {
        $request->validate([
            'enabled_modules' => 'array',
            'enabled_modules.*' => 'string|exists:api_configs,slug',
            'default_payment_gateway' => 'nullable|string|exists:api_configs,slug',
            'default_sms_provider' => 'nullable|string|exists:api_configs,slug',
            'default_identity_provider' => 'nullable|string|exists:api_configs,slug',
            'default_card_provider' => 'nullable|string|exists:api_configs,slug',
        ]);

        try {
            DB::beginTransaction();

            // Save enabled modules
            Setting::set('modules.enabled', $request->enabled_modules ?? [], 'array', 'Enabled modules/providers');
            
            // Save default providers
            Setting::set('modules.default.payment_gateway', $request->default_payment_gateway, 'string', 'Default payment gateway');
            Setting::set('modules.default.sms_provider', $request->default_sms_provider, 'string', 'Default SMS provider');
            Setting::set('modules.default.identity_provider', $request->default_identity_provider, 'string', 'Default identity provider');
            Setting::set('modules.default.card_provider', $request->default_card_provider, 'string', 'Default card provider');

            // Update integration settings to match
            if ($request->default_payment_gateway) {
                Setting::set('integrations.payment_gateway', $request->default_payment_gateway, 'string', 'Payment gateway');
            }
            if ($request->default_sms_provider) {
                Setting::set('integrations.sms_provider', $request->default_sms_provider, 'string', 'SMS provider');
            }
            if ($request->default_identity_provider) {
                Setting::set('integrations.identity_provider', $request->default_identity_provider, 'string', 'Identity provider');
            }
            if ($request->default_card_provider) {
                Setting::set('integrations.card_provider', $request->default_card_provider, 'string', 'Card provider');
            }

            // Clear settings cache
            Setting::clearCache();

            DB::commit();

            // Log admin action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'update_module_settings',
                'target_type' => 'Settings',
                'target_id' => null,
                'new_values' => [
                    'enabled_modules' => $request->enabled_modules ?? [],
                    'default_providers' => [
                        'payment_gateway' => $request->default_payment_gateway,
                        'sms_provider' => $request->default_sms_provider,
                        'identity_provider' => $request->default_identity_provider,
                        'card_provider' => $request->default_card_provider,
                    ],
                ],
                'metadata' => [
                    'section' => 'modules',
                    'admin_name' => auth('admin')->user()->name,
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            return back()->with('success', 'Module settings updated successfully. Changes are now active across the platform.');

        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Failed to update module settings', [
                'admin_id' => auth('admin')->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);

            return back()->with('error', 'Failed to update module settings: ' . $e->getMessage());
        }
    }

    /**
     * Get available providers for a category.
     */
    public function getAvailableProviders(Request $request)
    {
        // Define allowed categories. Update this list as needed.
        $allowedCategories = ['sms', 'email', 'push', 'payment']; // Example categories

        $request->validate([
            'category' => ['required', 'string', Rule::in($allowedCategories)],
        ]);

        $providers = ApiConfig::active()
            ->where('category', $request->category)
            ->select('id', 'name', 'slug', 'description', 'is_live_mode', 'last_test_status')
            ->get();

        return response()->json([
            'success' => true,
            'providers' => $providers,
        ]);
    }
    /**
     * Update SMTP configuration.
     */
    public function updateSmtpConfig(Request $request)
    {
        $request->validate([
            'smtp_host' => 'required|string|max:255',
            'smtp_port' => 'required|integer|min:1|max:65535',
            'smtp_username' => 'required|string|max:255',
            'smtp_password' => 'required|string|max:255',
            'smtp_encryption' => 'nullable|in:tls,ssl',
            'mail_from_address' => 'required|email|max:255',
            'mail_from_name' => 'required|string|max:255',
        ]);

        try {
            DB::beginTransaction();

            // Store SMTP settings
            Setting::set('mail.smtp.host', $request->smtp_host, 'string', 'SMTP server host');
            Setting::set('mail.smtp.port', $request->smtp_port, 'integer', 'SMTP server port');
            Setting::set('mail.smtp.username', $request->smtp_username, 'string', 'SMTP username');
            Setting::set('mail.smtp.password', $request->smtp_password, 'string', 'SMTP password');
            Setting::set('mail.smtp.encryption', $request->smtp_encryption ?: null, 'string', 'SMTP encryption method');
            Setting::set('mail.from.address', $request->mail_from_address, 'string', 'Default from email address');
            Setting::set('mail.from.name', $request->mail_from_name, 'string', 'Default from name');

            // Update Laravel config dynamically
            config([
                'mail.mailers.smtp.host' => $request->smtp_host,
                'mail.mailers.smtp.port' => $request->smtp_port,
                'mail.mailers.smtp.username' => $request->smtp_username,
                'mail.mailers.smtp.password' => $request->smtp_password,
                'mail.mailers.smtp.encryption' => $request->smtp_encryption,
                'mail.from.address' => $request->mail_from_address,
                'mail.from.name' => $request->mail_from_name,
            ]);

            // Clear settings cache
            Setting::clearCache();

            DB::commit();

            // Log admin action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'update_smtp_configuration',
                'target_type' => 'Settings',
                'target_id' => null,
                'new_values' => array_merge($request->except('smtp_password'), [
                    'smtp_password' => '[HIDDEN]'
                ]),
                'metadata' => [
                    'section' => 'smtp_config',
                    'admin_name' => auth('admin')->user()->name,
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'SMTP configuration updated successfully.'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            
            Log::error('Failed to update SMTP configuration', [
                'admin_id' => auth('admin')->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->except('smtp_password')
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to update SMTP configuration: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Test SMTP connection.
     */
    public function testSmtpConnection()
    {
        try {
            // Get current SMTP settings
            $host = Setting::get('mail.smtp.host');
            $port = Setting::get('mail.smtp.port', 587);
            $username = Setting::get('mail.smtp.username');
            $password = Setting::get('mail.smtp.password');
            $encryption = Setting::get('mail.smtp.encryption', 'tls');

            // Validate all required fields are present and not empty
            if ($host === null || $host === '' || $username === null || $username === '' || $password === null || $password === '') {
                return response()->json([
                    'success' => false,
                    'message' => 'SMTP configuration is incomplete. Please fill all required fields.'
                ], 400);
            }

            // Construct DSN string with properly encoded credentials
            $dsn = sprintf(
                'smtp://%s:%s@%s:%d?encryption=%s',
                rawurlencode($username),
                rawurlencode($password),
                $host,
                $port,
                $encryption ?: 'tls'
            );

            // Test connection using Symfony Mailer EsmtpTransport
            $transport = \Symfony\Component\Mailer\Transport::fromDsn($dsn);

            // Start and stop transport to test connectivity
            $transport->start();
            $transport->stop();

            return response()->json([
                'success' => true,
                'message' => 'SMTP connection test successful.'
            ]);

        } catch (\Exception $e) {
            Log::error('SMTP connection test failed', [
                'admin_id' => auth('admin')->id(),
                'error' => $e->getMessage(),
                'smtp_host' => Setting::get('mail.smtp.host'),
                'smtp_port' => Setting::get('mail.smtp.port', 587),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'SMTP connection failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Send test email.
     */
    public function sendTestEmail(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
        ]);

        try {
            // Update Laravel config with current settings
            $this->updateMailConfig();

            $testEmail = $request->email;
            $fromName = Setting::get('mail.from.name', Setting::get('business.name', 'SwiftPay'));
            
            \Mail::raw(
                "This is a test email from {$fromName}.\n\nIf you received this email, your SMTP configuration is working correctly!\n\nSent at: " . now()->format('Y-m-d H:i:s T'),
                function ($message) use ($testEmail, $fromName) {
                    $message->to($testEmail)
                           ->subject('Test Email from ' . $fromName)
                           ->from(
                               Setting::get('mail.from.address', 'noreply@example.com'),
                               $fromName
                           );
                }
            );

            // Log admin action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'send_test_email',
                'target_type' => 'Email',
                'target_id' => null,
                'new_values' => [
                    'recipient' => $testEmail,
                    'admin_name' => auth('admin')->user()->name,
                ],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Test email sent successfully to ' . $testEmail
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to send test email', [
                'admin_id' => auth('admin')->id(),
                'recipient' => $request->email,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to send test email: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update Laravel mail configuration with current settings.
     */
    private function updateMailConfig()
    {
        config([
            'mail.default' => 'smtp',
            'mail.mailers.smtp.transport' => 'smtp',
            'mail.mailers.smtp.host' => Setting::get('mail.smtp.host'),
            'mail.mailers.smtp.port' => Setting::get('mail.smtp.port', 587),
            'mail.mailers.smtp.encryption' => Setting::get('mail.smtp.encryption', 'tls'),
            'mail.mailers.smtp.username' => Setting::get('mail.smtp.username'),
            'mail.mailers.smtp.password' => Setting::get('mail.smtp.password'),
            'mail.from.address' => Setting::get('mail.from.address'),
            'mail.from.name' => Setting::get('mail.from.name'),
        ]);
    }

    private function testPushNotification(): array
    {
        try {
            // In a real implementation, you'd test Firebase/FCM or similar
            return [
                'status' => 'warning',
                'message' => 'Push notification system not yet implemented',
                'tested_at' => now(),
            ];
        } catch (\Exception $e) {
            return [
                'status' => 'error',
                'message' => 'Push notification error: ' . $e->getMessage(),
                'tested_at' => now(),
            ];
        }
    }

    /**
     * Check if the current admin is a super admin.
     */
    private function isSuperAdmin(): bool
    {
        $admin = auth('admin')->user();
        if (!$admin) {
            return false;
        }
        // Check for common super admin indicators
        if (property_exists($admin, 'is_super_admin') && $admin->is_super_admin) {
            return true;
        }
        if (property_exists($admin, 'role') && $admin->role === 'super_admin') {
            return true;
        }
        // Add more checks as needed
        return false;
    }
}