<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\Wallet;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class WalletController extends Controller
{
    /**
     * Display a listing of wallets.
     */
    public function index(Request $request)
    {
        $query = Wallet::with('user');

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('user', function($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->filled('status')) {
            if ($request->status === 'frozen') {
                $query->where('is_frozen', true);
            } elseif ($request->status === 'active') {
                $query->where('is_frozen', false);
            }
        }

        if ($request->filled('min_balance')) {
            $query->where('balance', '>=', $request->min_balance);
        }

        if ($request->filled('max_balance')) {
            $query->where('balance', '<=', $request->max_balance);
        }

        $wallets = $query->latest()->paginate(20);

        // Statistics
        $stats = [
            'total_wallets' => Wallet::count(),
            'active_wallets' => Wallet::where('is_frozen', false)->count(),
            'frozen_wallets' => Wallet::where('is_frozen', true)->count(),
            'total_balance' => Wallet::sum('balance'),
            'average_balance' => Wallet::avg('balance'),
        ];

        return view('admin.wallets.index', compact('wallets', 'stats'));
    }

    /**
     * Display wallet transactions.
     */
    public function transactions(Request $request)
    {
        $query = Transaction::with(['user', 'wallet']);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('reference', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhereHas('user', function($subq) use ($search) {
                      $subq->where('first_name', 'like', "%{$search}%")
                           ->orWhere('last_name', 'like', "%{$search}%")
                           ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        if ($request->filled('min_amount')) {
            $query->where('amount', '>=', $request->min_amount);
        }

        if ($request->filled('max_amount')) {
            $query->where('amount', '<=', $request->max_amount);
        }

        $transactions = $query->latest()->paginate(20)->withQueryString();

        // Statistics
        $totalTransactions = Transaction::count();
        $completedTransactions = Transaction::where('status', 'completed')->count();
        $pendingTransactions = Transaction::where('status', 'pending')->count();
        $failedTransactions = Transaction::where('status', 'failed')->count();
        $todayTransactions = Transaction::whereDate('created_at', today())->count();
        $todayVolume = Transaction::whereDate('created_at', today())->where('status', 'completed')->sum('amount');
        $totalVolume = Transaction::where('status', 'completed')->sum('amount');
        $pendingVolume = Transaction::where('status', 'pending')->sum('amount');
        $failedRate = $totalTransactions > 0 ? ($failedTransactions / $totalTransactions) * 100 : 0;

        $stats = [
            'total_transactions' => $totalTransactions,
            'today_transactions' => $todayTransactions,
            'total_volume' => $totalVolume,
            'today_volume' => $todayVolume,
            'pending_transactions' => $pendingTransactions,
            'pending_volume' => $pendingVolume,
            'failed_transactions' => $failedTransactions,
            'failed_rate' => $failedRate,
        ];

        return view('admin.wallets.transactions', compact('transactions', 'stats'));
    }

    /**
     * Display pending withdrawals.
     */
    public function pendingWithdrawals(Request $request)
    {
        $query = Transaction::with(['user', 'wallet'])
                          ->where('category', 'withdrawal')
                          ->where('type', 'debit')
                          ->where('status', 'pending');

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('reference', 'like', "%{$search}%")
                  ->orWhereHas('user', function($subq) use ($search) {
                      $subq->where('first_name', 'like', "%{$search}%")
                           ->orWhere('last_name', 'like', "%{$search}%")
                           ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('min_amount')) {
            $query->where('amount', '>=', $request->min_amount);
        }

        if ($request->filled('max_amount')) {
            $query->where('amount', '<=', $request->max_amount);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        $withdrawals = $query->latest()->paginate(20)->withQueryString();

        // Add bank details to withdrawals from metadata
        $withdrawals->getCollection()->transform(function ($withdrawal) {
            $metadata = json_decode($withdrawal->metadata, true) ?? [];
            $withdrawal->bank_name = $metadata['bank_name'] ?? 'N/A';
            $withdrawal->account_number = $metadata['account_number'] ?? 'N/A';
            $withdrawal->account_name = $metadata['account_name'] ?? 'N/A';
            return $withdrawal;
        });

        // Statistics
        $pendingCount = $query->count();
        $pendingAmount = $query->sum('amount');
        $urgentCount = Transaction::where('category', 'withdrawal')
                                 ->where('type', 'debit')
                                 ->where('status', 'pending')
                                 ->where('created_at', '<=', now()->subHours(24))
                                 ->count();
        
        $approvedToday = Transaction::where('category', 'withdrawal')
                                   ->where('type', 'debit')
                                   ->where('status', 'completed')
                                   ->whereDate('updated_at', today())
                                   ->count();
        
        $approvedAmountToday = Transaction::where('category', 'withdrawal')
                                         ->where('type', 'debit')
                                         ->where('status', 'completed')
                                         ->whereDate('updated_at', today())
                                         ->sum('amount');
        
        $declinedToday = Transaction::where('category', 'withdrawal')
                                   ->where('type', 'debit')
                                   ->where('status', 'failed')
                                   ->whereDate('updated_at', today())
                                   ->count();
        
        $declinedAmountToday = Transaction::where('category', 'withdrawal')
                                         ->where('type', 'debit')
                                         ->where('status', 'failed')
                                         ->whereDate('updated_at', today())
                                         ->sum('amount');

        $stats = [
            'pending_count' => $pendingCount,
            'pending_amount' => $pendingAmount,
            'urgent_count' => $urgentCount,
            'approved_today' => $approvedToday,
            'approved_amount_today' => $approvedAmountToday,
            'declined_today' => $declinedToday,
            'declined_amount_today' => $declinedAmountToday,
        ];

        return view('admin.wallets.pending-withdrawals', compact('withdrawals', 'stats'));
    }

    /**
     * Approve a withdrawal.
     */
    public function approveWithdrawal(Transaction $transaction)
    {
        if ($transaction->category !== 'bank_transfer' || $transaction->type !== 'debit' || $transaction->status !== 'pending') {
            return back()->withErrors(['error' => 'Invalid transaction for approval.']);
        }

        DB::transaction(function () use ($transaction) {
            $transaction->update([
                'status' => 'completed',
                'processed_at' => now(),
            ]);

            // Log admin action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'approve_withdrawal',
                'target_type' => 'Transaction',
                'target_id' => $transaction->id,
                'old_values' => ['status' => 'pending'],
                'new_values' => ['status' => 'completed'],
                'ip_address' => request()->ip(),
            ]);
        });

        return back()->with('success', 'Withdrawal approved successfully.');
    }

    /**
     * Decline a withdrawal.
     */
    public function declineWithdrawal(Request $request, Transaction $transaction)
    {
        $request->validate([
            'reason' => 'required|string|max:500',
        ]);

        if ($transaction->category !== 'bank_transfer' || $transaction->type !== 'debit' || $transaction->status !== 'pending') {
            return back()->withErrors(['error' => 'Invalid transaction for decline.']);
        }

        DB::transaction(function () use ($transaction, $request) {
            // Reverse the transaction - credit back to wallet
            $transaction->wallet->increment('balance', $transaction->total_amount);

            $transaction->update([
                'status' => 'failed',
                'processed_at' => now(),
                'metadata' => array_merge(
                    json_decode($transaction->metadata, true) ?? [],
                    ['decline_reason' => $request->reason, 'declined_by' => auth('admin')->id()]
                ),
            ]);

            // Log admin action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'decline_withdrawal',
                'target_type' => 'Transaction',
                'target_id' => $transaction->id,
                'old_values' => ['status' => 'pending'],
                'new_values' => ['status' => 'failed', 'reason' => $request->reason],
                'ip_address' => request()->ip(),
            ]);
        });

        return back()->with('success', 'Withdrawal declined and funds returned to user wallet.');
    }

    /**
     * Manually adjust wallet balance.
     */
    public function adjustBalance(Request $request, Wallet $wallet)
    {
        $request->validate([
            'type' => 'required|in:credit,debit',
            'amount' => 'required|numeric|min:0.01|max:1000000',
            'reason' => 'required|string|max:500',
        ]);

        try {
            DB::transaction(function () use ($request, $wallet) {
                $oldBalance = $wallet->balance;
                
                if ($request->type === 'credit') {
                    $wallet->increment('balance', $request->amount);
                    $transactionType = 'credit';
                    $description = "Admin credit: {$request->reason}";
                } else {
                    if ($wallet->balance < $request->amount) {
                        throw new \Exception('Insufficient wallet balance for debit.');
                    }
                    $wallet->decrement('balance', $request->amount);
                    $transactionType = 'debit';
                    $description = "Admin debit: {$request->reason}";
                }

                // Create transaction record
                Transaction::create([
                    'user_id' => $wallet->user_id,
                    'wallet_id' => $wallet->id,
                    'type' => $transactionType,
                    'category' => 'admin_adjustment',
                    'amount' => $request->amount,
                    'fee' => 0,
                    'total_amount' => $request->amount,
                    'status' => 'completed',
                    'description' => $description,
                    'reference' => 'ADM-' . strtoupper(uniqid()),
                    'processed_at' => now(),
                    'metadata' => json_encode([
                        'admin_id' => auth('admin')->id(),
                        'reason' => $request->reason,
                        'old_balance' => $oldBalance,
                        'new_balance' => $wallet->fresh()->balance,
                    ]),
                ]);

                // Log admin action
                AuditLog::create([
                    'admin_id' => auth('admin')->id(),
                    'action' => 'adjust_wallet_balance',
                    'model_type' => 'App\Models\Wallet',
                    'model_id' => $wallet->id,
                    'old_values' => json_encode(['balance' => $oldBalance]),
                    'new_values' => json_encode(['balance' => $wallet->fresh()->balance]),
                    'metadata' => json_encode([
                        'type' => $request->type,
                        'amount' => $request->amount,
                        'reason' => $request->reason,
                    ]),
                    'ip_address' => request()->ip(),
                    'user_agent' => request()->userAgent(),
                ]);
            });

            return back()->with('success', 'Wallet balance adjusted successfully.');
        } catch (\Exception $e) {
            \Log::error('Failed to adjust wallet balance', [
                'wallet_id' => $wallet->id,
                'admin_id' => auth('admin')->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return back()->with('error', 'Failed to adjust wallet balance: ' . $e->getMessage());
        }
    }

    /**
     * Freeze or unfreeze a wallet.
     */
    public function toggleFreeze(Request $request, Wallet $wallet)
    {
        $request->validate([
            'reason' => 'required_if:action,freeze|string|max:500',
        ]);

        $action = $wallet->is_frozen ? 'unfreeze' : 'freeze';
        
        DB::transaction(function () use ($wallet, $request, $action) {
            $wallet->update([
                'is_frozen' => !$wallet->is_frozen,
                'frozen_at' => $wallet->is_frozen ? null : now(),
                'freeze_reason' => $wallet->is_frozen ? null : $request->reason,
            ]);

            // Log admin action
            AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => $action . '_wallet',
                'target_type' => 'Wallet',
                'target_id' => $wallet->id,
                'old_values' => ['is_frozen' => !$wallet->is_frozen],
                'new_values' => ['is_frozen' => $wallet->is_frozen, 'reason' => $request->reason ?? null],
                'ip_address' => request()->ip(),
            ]);
        });

        $message = $action === 'freeze' ? 'Wallet frozen successfully.' : 'Wallet unfrozen successfully.';
        return back()->with('success', $message);
    }
}