<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Symfony\Component\HttpFoundation\Response;

class VerifyTransactionPin
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $user = $request->user();
        
        // Check if user has set a transaction PIN
        if (empty($user->transaction_pin)) {
            return response()->json([
                'success' => false,
                'message' => 'Transaction PIN is required. Please set up your transaction PIN first.',
                'error_code' => 'PIN_NOT_SET'
            ], 403);
        }

        // Get transaction PIN from request
        $providedPin = $request->input('transaction_pin');
        
        if (empty($providedPin)) {
            return response()->json([
                'success' => false,
                'message' => 'Transaction PIN is required for this operation.',
                'error_code' => 'PIN_REQUIRED'
            ], 400);
        }

        // Verify the PIN
        if (!Hash::check($providedPin, $user->transaction_pin)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid transaction PIN.',
                'error_code' => 'INVALID_PIN'
            ], 401);
        }

        return $next($request);
    }
}