<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ApiConfig;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;

class ApiConfigController extends Controller
{
    /**
     * Display API configuration dashboard.
     */
    public function index()
    {
        $apiConfigs = ApiConfig::all();
        
        // Group services by category
        $serviceCategories = [
            'payment' => $apiConfigs->where('category', 'payment'),
            'identity' => $apiConfigs->where('category', 'identity'),
            'messaging' => $apiConfigs->where('category', 'messaging'),
            'utilities' => $apiConfigs->where('category', 'utilities'),
            'banking' => $apiConfigs->where('category', 'banking'),
        ];

        // Get health stats from cache or set defaults
        $healthStats = [];
        foreach ($apiConfigs as $config) {
            $healthStats[$config->name] = Cache::get("api_health_{$config->slug}", [
                'status' => $config->last_test_status ?? 'unknown',
                'last_check' => $config->last_tested_at,
                'response_time' => null,
            ]);
        }

        return view('admin.api-config.index', compact('serviceCategories', 'healthStats'));
    }

    /**
     * Show API service configuration.
     */
    public function show(ApiConfig $apiConfig)
    {
        return view('admin.api-config.show', compact('apiConfig'));
    }

    /**
     * Update API service configuration.
     */
    public function update(Request $request, ApiConfig $apiConfig)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'base_url' => 'required|url',
            'test_url' => 'nullable|url',
            'live_url' => 'nullable|url',
            'api_key' => 'nullable|string',
            'secret_key' => 'nullable|string',
            'public_key' => 'nullable|string',
            'private_key' => 'nullable|string',
            'merchant_id' => 'nullable|string',
            'client_id' => 'nullable|string',
            'client_secret' => 'nullable|string',
            'is_active' => 'boolean',
            'is_live_mode' => 'boolean',
            'webhook_url' => 'nullable|url',
            'callback_url' => 'nullable|url',
            'return_url' => 'nullable|url',
            'config.timeout' => 'nullable|integer|min:5|max:120',
            'config.retry_attempts' => 'nullable|integer|min:0|max:5',
        ]);

        DB::transaction(function () use ($request, $apiConfig) {
            $oldValues = $apiConfig->toArray();
            
            // Prepare config data
            $config = $apiConfig->config ?? [];
            if ($request->has('config.timeout')) {
                $config['timeout'] = $request->input('config.timeout');
            }
            if ($request->has('config.retry_attempts')) {
                $config['retry_attempts'] = $request->input('config.retry_attempts');
            }

            $apiConfig->update([
                'name' => $request->name,
                'description' => $request->description,
                'base_url' => $request->base_url,
                'test_url' => $request->test_url,
                'live_url' => $request->live_url,
                'api_key' => $request->api_key,
                'secret_key' => $request->secret_key,
                'public_key' => $request->public_key,
                'private_key' => $request->private_key,
                'merchant_id' => $request->merchant_id,
                'client_id' => $request->client_id,
                'client_secret' => $request->client_secret,
                'is_active' => $request->boolean('is_active'),
                'is_live_mode' => $request->boolean('is_live_mode'),
                'webhook_url' => $request->webhook_url,
                'callback_url' => $request->callback_url,
                'return_url' => $request->return_url,
                'config' => $config,
                'updated_at' => now(),
            ]);

            // Log admin action
            AuditLog::create([
                'user_id' => auth('admin')->id(),
                'action' => 'update_api_config',
                'model_type' => 'ApiConfig',
                'model_id' => $apiConfig->id,
                'old_values' => $oldValues,
                'new_values' => $apiConfig->fresh()->toArray(),
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);
        });

        return back()->with('success', 'API configuration updated successfully.');
    }

    /**
     * Test API connection.
     */
    public function testConnection(ApiConfig $apiConfig)
    {
        try {
            $startTime = microtime(true);
            
            // Update status to pending
            $apiConfig->update([
                'last_test_status' => 'pending',
                'last_tested_at' => now(),
            ]);
            
            // Build test request based on service type
            $testEndpoint = $this->getTestEndpoint($apiConfig);
            $headers = $this->buildHeaders($apiConfig);
            
            $response = Http::withHeaders($headers)
                           ->timeout($apiConfig->config['timeout'] ?? 30)
                           ->get($apiConfig->effective_base_url . $testEndpoint);
            
            $responseTime = round((microtime(true) - $startTime) * 1000, 2); // in milliseconds
            
            $success = $response->successful();
            $status = $success ? 'success' : 'failed';
            
            // Update the API config with test results
            $apiConfig->update([
                'last_test_status' => $status,
                'last_tested_at' => now(),
                'last_test_response' => json_encode([
                    'status_code' => $response->status(),
                    'response_time' => $responseTime,
                    'response_body' => $response->json() ?? $response->body(),
                ]),
            ]);
            
            $result = [
                'success' => $success,
                'status_code' => $response->status(),
                'response_time' => $responseTime,
                'response_body' => $response->json() ?? $response->body(),
                'headers' => $response->headers(),
                'tested_at' => now(),
            ];

            // Cache the health status
            Cache::put("api_health_{$apiConfig->slug}", [
                'status' => $success ? 'success' : 'failed',
                'last_check' => now(),
                'response_time' => $responseTime,
            ], 300); // Cache for 5 minutes

            // Log the test
            AuditLog::create([
                'user_id' => auth('admin')->id(),
                'action' => 'test_api_connection',
                'model_type' => 'ApiConfig',
                'model_id' => $apiConfig->id,
                'new_values' => $result,
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);

            return response()->json($result);

        } catch (\Exception $e) {
            // Update status to failed
            $apiConfig->update([
                'last_test_status' => 'failed',
                'last_tested_at' => now(),
                'last_test_response' => json_encode([
                    'error' => $e->getMessage(),
                ]),
            ]);

            Cache::put("api_health_{$apiConfig->slug}", [
                'status' => 'error',
                'last_check' => now(),
                'response_time' => null,
                'error' => $e->getMessage(),
            ], 300);

            return response()->json([
                'success' => false,
                'error' => $e->getMessage(),
                'tested_at' => now(),
            ], 500);
        }
    }

    /**
     * Toggle service active status.
     */
    public function toggleStatus(ApiConfig $apiConfig)
    {
        DB::transaction(function () use ($apiConfig) {
            $oldStatus = $apiConfig->is_active;
            $apiConfig->update(['is_active' => !$apiConfig->is_active]);

            // Log admin action
            AuditLog::create([
                'user_id' => auth('admin')->id(),
                'action' => $apiConfig->is_active ? 'activate_service' : 'deactivate_service',
                'model_type' => 'ApiConfig',
                'model_id' => $apiConfig->id,
                'old_values' => ['is_active' => $oldStatus],
                'new_values' => ['is_active' => $apiConfig->is_active],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);
        });

        $message = $apiConfig->is_active ? 'Service activated successfully.' : 'Service deactivated successfully.';
        
        if (request()->expectsJson()) {
            return response()->json(['success' => true, 'message' => $message]);
        }
        
        return back()->with('success', $message);
    }

    /**
     * Toggle live/test mode.
     */
    public function toggleMode(ApiConfig $apiConfig)
    {
        DB::transaction(function () use ($apiConfig) {
            $oldMode = $apiConfig->is_live_mode;
            $apiConfig->update(['is_live_mode' => !$apiConfig->is_live_mode]);

            // Log admin action
            AuditLog::create([
                'user_id' => auth('admin')->id(),
                'action' => 'toggle_service_mode',
                'model_type' => 'ApiConfig',
                'model_id' => $apiConfig->id,
                'old_values' => ['is_live_mode' => $oldMode],
                'new_values' => ['is_live_mode' => $apiConfig->is_live_mode],
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ]);
        });

        $mode = $apiConfig->is_live_mode ? 'live' : 'test';
        $message = "Service switched to {$mode} mode successfully.";
        
        if (request()->expectsJson()) {
            return response()->json(['success' => true, 'message' => $message]);
        }
        
        return back()->with('success', $message);
    }

    /**
     * View API logs.
     */
    public function logs(Request $request)
    {
        $query = AuditLog::with('user')
                         ->whereIn('action', [
                             'update_api_config',
                             'activate_service',
                             'deactivate_service',
                             'toggle_service_mode',
                             'test_api_connection'
                         ]);

        if ($request->filled('service')) {
            $query->where('model_type', 'ApiConfig')
                  ->where('model_id', $request->service);
        }

        if ($request->filled('action')) {
            $query->where('action', $request->action);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $logs = $query->latest()->paginate(20);
        $apiConfigs = ApiConfig::all();

        return view('admin.api-config.logs', compact('logs', 'apiConfigs'));
    }

    /**
     * Get appropriate test endpoint for service.
     */
    private function getTestEndpoint(ApiConfig $apiConfig): string
    {
        $testEndpoints = [
            'paystack' => '/bank',
            'flutterwave' => '/banks/NG',
            'monnify' => '/api/v1/banks',
            'dojah' => '/health',
            'verifyme' => '/health',
            'termii' => '/api/get-balance',
            'vtpass' => '/services',
            'anchor' => '/health',
        ];

        return $testEndpoints[$apiConfig->slug] ?? '/health';
    }

    /**
     * Build headers for API request.
     */
    private function buildHeaders(ApiConfig $apiConfig): array
    {
        $headers = [
            'Accept' => 'application/json',
            'Content-Type' => 'application/json',
        ];

        if ($apiConfig->api_key) {
            // Different services use different authentication methods
            switch ($apiConfig->slug) {
                case 'paystack':
                    $headers['Authorization'] = 'Bearer ' . $apiConfig->api_key;
                    break;
                case 'flutterwave':
                    $headers['Authorization'] = 'Bearer ' . $apiConfig->secret_key;
                    break;
                case 'monnify':
                    // Monnify uses basic auth with client_id:api_key
                    if ($apiConfig->client_id && $apiConfig->api_key) {
                        $headers['Authorization'] = 'Basic ' . base64_encode($apiConfig->client_id . ':' . $apiConfig->api_key);
                    }
                    break;
                case 'termii':
                    // Termii uses API key in URL or headers
                    $headers['Authorization'] = 'Bearer ' . $apiConfig->api_key;
                    break;
                case 'dojah':
                case 'verifyme':
                    $headers['Authorization'] = $apiConfig->api_key;
                    break;
                case 'vtpass':
                    $headers['api-key'] = $apiConfig->api_key;
                    $headers['secret-key'] = $apiConfig->secret_key;
                    break;
                default:
                    $headers['Authorization'] = 'Bearer ' . $apiConfig->api_key;
            }
        }

        // Add custom headers from config
        $customHeaders = $apiConfig->config['headers'] ?? [];
        return array_merge($headers, $customHeaders);
    }
}