<?php

namespace App\Services;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class WebhookVerificationService
{
    /**
     * Verify Paystack webhook signature.
     */
    public static function verifyPaystackSignature(Request $request): bool
    {
        $signature = $request->header('x-paystack-signature');
        
        if (empty($signature)) {
            Log::warning('Paystack webhook received without signature');
            return false;
        }

        $secret = config('services.paystack.secret_key');
        $body = $request->getContent();
        
        $expectedSignature = hash_hmac('sha512', $body, $secret);
        
        if (!hash_equals($expectedSignature, $signature)) {
            Log::warning('Invalid Paystack webhook signature', [
                'provided' => $signature,
                'expected' => $expectedSignature
            ]);
            return false;
        }

        return true;
    }

    /**
     * Verify Monnify webhook signature.
     */
    public static function verifyMonnifySignature(Request $request): bool
    {
        $signature = $request->header('monnify-signature');
        
        if (empty($signature)) {
            Log::warning('Monnify webhook received without signature');
            return false;
        }

        $secret = config('services.monnify.secret_key');
        $body = $request->getContent();
        
        $expectedSignature = hash_hmac('sha512', $body, $secret);
        
        if (!hash_equals($expectedSignature, $signature)) {
            Log::warning('Invalid Monnify webhook signature', [
                'provided' => $signature,
                'expected' => $expectedSignature
            ]);
            return false;
        }

        return true;
    }

    /**
     * Verify VTPass webhook signature.
     */
    public static function verifyVTPassSignature(Request $request): bool
    {
        // VTPass may use different signature verification
        // TODO: Implement based on VTPass documentation
        
        $signature = $request->header('vtpass-signature');
        
        if (empty($signature)) {
            Log::warning('VTPass webhook received without signature');
            return false;
        }

        // Implement VTPass-specific verification logic here
        return true;
    }

    /**
     * Verify Anchor webhook signature.
     */
    public static function verifyAnchorSignature(Request $request): bool
    {
        $signature = $request->header('x-anchor-signature');
        
        if (empty($signature)) {
            Log::warning('Anchor webhook received without signature');
            return false;
        }

        $secret = config('services.anchor.secret_key');
        $body = $request->getContent();
        
        $expectedSignature = hash_hmac('sha256', $body, $secret);
        
        if (!hash_equals('sha256=' . $expectedSignature, $signature)) {
            Log::warning('Invalid Anchor webhook signature', [
                'provided' => $signature,
                'expected' => 'sha256=' . $expectedSignature
            ]);
            return false;
        }

        return true;
    }

    /**
     * Generic webhook signature verification.
     */
    public static function verifyGenericSignature(
        Request $request,
        string $secret,
        string $algorithm = 'sha256',
        string $headerName = 'x-signature'
    ): bool {
        $signature = $request->header($headerName);
        
        if (empty($signature)) {
            Log::warning("Webhook received without signature header: {$headerName}");
            return false;
        }

        $body = $request->getContent();
        $expectedSignature = hash_hmac($algorithm, $body, $secret);
        
        // Handle different signature formats
        $signatureToCheck = $signature;
        if (strpos($signature, $algorithm . '=') === 0) {
            $signatureToCheck = substr($signature, strlen($algorithm) + 1);
        }
        
        if (!hash_equals($expectedSignature, $signatureToCheck)) {
            Log::warning('Invalid webhook signature', [
                'provider' => $headerName,
                'provided' => $signature,
                'expected' => $algorithm . '=' . $expectedSignature
            ]);
            return false;
        }

        return true;
    }

    /**
     * Log webhook verification attempt.
     */
    public static function logVerificationAttempt(
        string $provider,
        bool $success,
        Request $request,
        ?array $metadata = null
    ): void {
        Log::info('Webhook signature verification', [
            'provider' => $provider,
            'success' => $success,
            'ip' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'metadata' => $metadata
        ]);
    }
}