<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Transaction;
use App\Models\CronLog;
use App\Services\VTPassService;

class SyncBillPaymentsCommand extends Command
{
    protected $signature = 'fintech:sync-bill-payments';
    protected $description = 'Update data/airtime order status from VTPass';

    protected $vtpassService;

    public function __construct(VTPassService $vtpassService)
    {
        parent::__construct();
        $this->vtpassService = $vtpassService;
    }

    public function handle()
    {
        $startTime = microtime(true);
        $startMemory = memory_get_usage(true);
        
        try {
            $this->info('Starting bill payment status synchronization...');
            
            $syncedCount = 0;
            $errorCount = 0;
            
            // Get pending bill payment transactions from last 24 hours
            $pendingTransactions = Transaction::whereIn('category', [
                    'airtime_purchase', 
                    'data_purchase', 
                    'bill_payment'
                ])
                ->whereIn('status', ['processing', 'pending'])
                ->where('created_at', '>=', now()->subHours(24))
                ->whereNotNull('external_reference')
                ->get();
            
            foreach ($pendingTransactions as $transaction) {
                try {
                    if ($this->syncTransactionStatus($transaction)) {
                        $syncedCount++;
                    }
                } catch (\Exception $e) {
                    $this->error("Error syncing transaction {$transaction->id}: " . $e->getMessage());
                    $errorCount++;
                }
            }
            
            $executionTime = microtime(true) - $startTime;
            $memoryUsage = memory_get_usage(true) - $startMemory;
            
            $message = "Synced {$syncedCount} bill payments, {$errorCount} errors";
            $this->info($message);
            
            CronLog::create([
                'task_name' => 'sync_bill_payments',
                'status' => $errorCount > 0 ? 'warning' : 'success',
                'message' => $message,
                'executed_at' => now(),
                'execution_time' => $executionTime,
                'memory_usage' => $memoryUsage
            ]);
            
        } catch (\Exception $e) {
            $executionTime = microtime(true) - $startTime;
            $memoryUsage = memory_get_usage(true) - $startMemory;
            
            $this->error('Bill payment sync failed: ' . $e->getMessage());
            
            CronLog::create([
                'task_name' => 'sync_bill_payments',
                'status' => 'failed',
                'message' => 'Bill payment sync failed',
                'executed_at' => now(),
                'execution_time' => $executionTime,
                'memory_usage' => $memoryUsage,
                'error_details' => [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]
            ]);
        }
    }
    
    private function syncTransactionStatus(Transaction $transaction)
    {
        $response = $this->vtpassService->checkTransactionStatus($transaction->external_reference);
        
        if (!$response['success']) {
            $this->warn("Could not check status for transaction {$transaction->id}");
            return false;
        }
        
        $currentStatus = $transaction->status;
        $newStatus = $this->mapVTPassStatus($response['status']);
        
        if ($newStatus && $newStatus !== $currentStatus) {
            $metadata = $transaction->metadata ?? [];
            $metadata['vtpass_status'] = $response['status'];
            $metadata['status_updated_at'] = now()->toISOString();
            
            if (isset($response['token'])) {
                $metadata['recharge_token'] = $response['token'];
            }
            
            $transaction->update([
                'status' => $newStatus,
                'metadata' => $metadata
            ]);
            
            $this->info("Updated transaction {$transaction->id} status from {$currentStatus} to {$newStatus}");
            
            // Send notification if completed or failed
            if ($newStatus === 'completed') {
                $this->sendSuccessNotification($transaction, $response);
            } elseif ($newStatus === 'failed') {
                $this->sendFailureNotification($transaction, $response);
            }
            
            return true;
        }
        
        return false;
    }
    
    private function mapVTPassStatus($vtpassStatus)
    {
        return match(strtolower($vtpassStatus)) {
            'initiated', 'pending' => 'processing',
            'delivered', 'successful' => 'completed',
            'failed', 'rejected' => 'failed',
            default => null
        };
    }
    
    private function sendSuccessNotification(Transaction $transaction, $response)
    {
        $user = $transaction->user;
        $message = $this->generateSuccessMessage($transaction, $response);
        
        // Create in-app notification
        $user->notifications()->create([
            'type' => 'bill_payment_success',
            'title' => 'Payment Successful',
            'message' => $message,
            'data' => [
                'transaction_id' => $transaction->id,
                'amount' => $transaction->amount,
                'category' => $transaction->category,
                'token' => $response['token'] ?? null
            ]
        ]);
        
        $this->info("Sent success notification for transaction {$transaction->id}");
    }
    
    private function sendFailureNotification(Transaction $transaction, $response)
    {
        $user = $transaction->user;
        $message = $this->generateFailureMessage($transaction, $response);
        
        // Create in-app notification
        $user->notifications()->create([
            'type' => 'bill_payment_failed',
            'title' => 'Payment Failed',
            'message' => $message,
            'data' => [
                'transaction_id' => $transaction->id,
                'amount' => $transaction->amount,
                'category' => $transaction->category,
                'reason' => $response['message'] ?? 'Unknown error'
            ]
        ]);
        
        // If failed, refund the amount to wallet
        if ($transaction->type === 'debit') {
            $wallet = $transaction->wallet;
            $wallet->credit($transaction->total_amount);
            
            // Create refund transaction
            Transaction::create([
                'user_id' => $transaction->user_id,
                'wallet_id' => $wallet->id,
                'type' => 'credit',
                'category' => 'refund',
                'amount' => $transaction->total_amount,
                'fee' => 0,
                'total_amount' => $transaction->total_amount,
                'currency' => 'NGN',
                'status' => 'completed',
                'description' => "Refund for failed {$transaction->category}",
                'metadata' => [
                    'original_transaction_id' => $transaction->id,
                    'refund_reason' => 'Service delivery failed'
                ]
            ]);
            
            $this->info("Refunded ₦{$transaction->total_amount} for failed transaction {$transaction->id}");
        }
        
        $this->info("Sent failure notification for transaction {$transaction->id}");
    }
    
    private function generateSuccessMessage(Transaction $transaction, $response)
    {
        $metadata = $transaction->metadata ?? [];
        
        return match($transaction->category) {
            'airtime_purchase' => "Your ₦{$transaction->amount} {$metadata['provider']} airtime has been successfully delivered to {$metadata['recipient']}",
            'data_purchase' => "Your ₦{$transaction->amount} {$metadata['provider']} data bundle has been successfully delivered to {$metadata['recipient']}",
            'bill_payment' => "Your ₦{$transaction->amount} electricity payment has been successfully processed for meter {$metadata['recipient']}",
            default => "Your ₦{$transaction->amount} payment has been successfully processed"
        };
    }
    
    private function generateFailureMessage(Transaction $transaction, $response)
    {
        $metadata = $transaction->metadata ?? [];
        
        $baseMessage = match($transaction->category) {
            'airtime_purchase' => "Your ₦{$transaction->amount} {$metadata['provider']} airtime purchase failed",
            'data_purchase' => "Your ₦{$transaction->amount} {$metadata['provider']} data purchase failed",
            'bill_payment' => "Your ₦{$transaction->amount} electricity payment failed",
            default => "Your ₦{$transaction->amount} payment failed"
        };
        
        return $baseMessage . ". Your wallet has been refunded.";
    }
}