<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\VirtualCard;
use App\Models\User;
use App\Services\CardProviderManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class VirtualCardController extends Controller
{
    /**
     * Display a listing of virtual cards.
     */
    public function index(Request $request)
    {
        $query = VirtualCard::with(['user', 'wallet']);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('user', function($q) use ($search) {
                $q->where('first_name', 'like', "%{$search}%")
                  ->orWhere('last_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            })->orWhere('card_number', 'like', "%{$search}%");
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('type')) {
            $query->where('card_type', $request->type);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $cards = $query->latest()->paginate(20);

        // Statistics
        $stats = [
            'total_cards' => VirtualCard::count(),
            'active_cards' => VirtualCard::where('status', 'active')->count(),
            'frozen_cards' => VirtualCard::where('status', 'frozen')->count(),
            'terminated_cards' => VirtualCard::where('status', 'terminated')->count(),
            'total_spending' => 0, // Will implement when virtual card transactions are linked
        ];

        // Get available card providers with detailed status
        $cardProviderManager = new CardProviderManager();
        $cardProviders = $cardProviderManager->getAvailableProviders();
        
        // Separate active from problematic providers
        $activeProviders = collect($cardProviders)->filter(function ($provider) {
            return $provider['configured'] && $provider['is_active'] && $provider['last_test_status'] !== 'failed';
        });
        
        $problematicProviders = collect($cardProviders)->filter(function ($provider) {
            return !$provider['configured'] || !$provider['is_active'] || $provider['last_test_status'] === 'failed';
        });

        // Get users for assignment dropdown
        $users = User::select('id', 'first_name', 'last_name', 'email')
                    ->where('is_active', true)
                    ->orderBy('first_name')
                    ->get();

        return view('admin.virtual-cards.index', compact(
            'cards', 
            'stats', 
            'cardProviders', 
            'activeProviders',
            'problematicProviders',
            'users'
        ));
    }

    /**
     * Display the specified virtual card.
     */
    public function show(VirtualCard $virtualCard)
    {
        $virtualCard->load(['user', 'wallet', 'transactions' => function($query) {
            $query->latest()->limit(20);
        }]);

        return view('admin.virtual-cards.show', compact('virtualCard'));
    }

    /**
     * Activate a virtual card.
     */
    public function activate(VirtualCard $virtualCard)
    {
        if ($virtualCard->status === 'terminated') {
            return back()->withErrors(['error' => 'Cannot activate a terminated card.']);
        }

        $virtualCard->update([
            'status' => 'active',
            'activated_at' => now(),
        ]);

        return back()->with('success', 'Virtual card activated successfully.');
    }

    /**
     * Deactivate a virtual card.
     */
    public function deactivate(VirtualCard $virtualCard)
    {
        $virtualCard->update([
            'status' => 'inactive',
        ]);

        return back()->with('success', 'Virtual card deactivated successfully.');
    }

    /**
     * Freeze a virtual card.
     */
    public function freeze(VirtualCard $virtualCard)
    {
        if ($virtualCard->status === 'terminated') {
            return back()->withErrors(['error' => 'Cannot freeze a terminated card.']);
        }

        $virtualCard->update([
            'status' => 'frozen',
            'frozen_at' => now(),
        ]);

        return back()->with('success', 'Virtual card frozen successfully.');
    }

    /**
     * Unfreeze a virtual card.
     */
    public function unfreeze(VirtualCard $virtualCard)
    {
        $virtualCard->update([
            'status' => 'active',
            'frozen_at' => null,
        ]);

        return back()->with('success', 'Virtual card unfrozen successfully.');
    }

    /**
     * Terminate a virtual card.
     */
    public function terminate(VirtualCard $virtualCard)
    {
        $virtualCard->update([
            'status' => 'terminated',
            'terminated_at' => now(),
        ]);

        return back()->with('success', 'Virtual card terminated successfully.');
    }

    /**
     * Reissue a virtual card.
     */
    public function reissue(VirtualCard $virtualCard)
    {
        if ($virtualCard->status !== 'active' && $virtualCard->status !== 'frozen') {
            return back()->withErrors(['error' => 'Can only reissue active or frozen cards.']);
        }

        // Generate new card details
        $newCardNumber = $this->generateCardNumber();
        $newCvv = $this->generateCvv();
        $newExpiryDate = now()->addYears(3);

        $virtualCard->update([
            'card_number' => $newCardNumber,
            'cvv' => $newCvv,
            'expiry_date' => $newExpiryDate,
            'reissued_at' => now(),
        ]);

        return back()->with('success', 'Virtual card reissued successfully with new details.');
    }

    /**
     * Update card limits.
     */
    public function updateLimits(Request $request, VirtualCard $virtualCard)
    {
        $request->validate([
            'daily_limit' => 'required|numeric|min:0|max:1000000',
            'monthly_limit' => 'required|numeric|min:0|max:10000000',
            'per_transaction_limit' => 'required|numeric|min:0|max:500000',
        ]);

        $virtualCard->update([
            'daily_limit' => $request->daily_limit,
            'monthly_limit' => $request->monthly_limit,
            'per_transaction_limit' => $request->per_transaction_limit,
        ]);

        return back()->with('success', 'Card limits updated successfully.');
    }

    /**
     * Update card fees.
     */
    public function updateFees(Request $request, VirtualCard $virtualCard)
    {
        $request->validate([
            'creation_fee' => 'required|numeric|min:0|max:10000',
            'maintenance_fee' => 'required|numeric|min:0|max:1000',
            'transaction_fee' => 'required|numeric|min:0|max:1000',
        ]);

        $virtualCard->update([
            'creation_fee' => $request->creation_fee,
            'maintenance_fee' => $request->maintenance_fee,
            'transaction_fee' => $request->transaction_fee,
        ]);

        return back()->with('success', 'Card fees updated successfully.');
    }

    /**
     * Credit a virtual card.
     */
    public function credit(Request $request, VirtualCard $virtualCard)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1|max:1000000',
            'description' => 'nullable|string|max:255',
        ]);

        if ($virtualCard->status === 'terminated') {
            return back()->withErrors(['error' => 'Cannot credit a terminated card.']);
        }

        // Add to card balance
        $virtualCard->increment('balance', $request->amount);

        // Create audit log
        AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'virtual_card_credit',
            'model' => 'VirtualCard',
            'model_id' => $virtualCard->id,
            'old_values' => ['balance' => $virtualCard->balance - $request->amount],
            'new_values' => ['balance' => $virtualCard->balance],
            'description' => "Credited ₦{$request->amount} to virtual card {$virtualCard->masked_pan}. " . ($request->description ?? ''),
            'ip_address' => request()->ip(),
        ]);

        return back()->with('success', "Virtual card credited with ₦{$request->amount} successfully.");
    }

    /**
     * Debit a virtual card.
     */
    public function debit(Request $request, VirtualCard $virtualCard)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1|max:' . $virtualCard->balance,
            'description' => 'nullable|string|max:255',
        ]);

        if ($virtualCard->status === 'terminated') {
            return back()->withErrors(['error' => 'Cannot debit a terminated card.']);
        }

        if ($virtualCard->balance < $request->amount) {
            return back()->withErrors(['error' => 'Insufficient balance on the card.']);
        }

        // Deduct from card balance
        $virtualCard->decrement('balance', $request->amount);

        // Create audit log
        \App\Models\AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'virtual_card_debit',
            'model' => 'VirtualCard',
            'model_id' => $virtualCard->id,
            'old_values' => ['balance' => $virtualCard->balance + $request->amount],
            'new_values' => ['balance' => $virtualCard->balance],
            'description' => "Debited ₦{$request->amount} from virtual card {$virtualCard->masked_pan}. " . ($request->description ?? ''),
            'ip_address' => request()->ip(),
        ]);

        return back()->with('success', "Virtual card debited with ₦{$request->amount} successfully.");
    }

    /**
     * Assign a virtual card to a user.
     */
    public function assign(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'provider' => 'required|string',
            'card_type' => 'required|in:virtual,physical',
            'brand' => 'required|in:visa,mastercard',
            'currency' => 'required|in:USD,NGN',
            'funding_amount' => 'required|numeric|min:0|max:100000',
            'daily_limit' => 'required|numeric|min:0|max:100000',
            'monthly_limit' => 'required|numeric|min:0|max:1000000',
        ]);

        $user = User::findOrFail($request->user_id);

        // Check if user already has too many cards
        if ($user->virtualCards()->count() >= 5) {
            return back()->withErrors(['error' => 'User already has maximum number of virtual cards (5).']);
        }

        try {
            $cardProviderManager = new CardProviderManager();

            // Check if provider is supported and active
            if (!$cardProviderManager->isProviderSupported($request->provider)) {
                return back()->withErrors(['error' => 'Selected card provider is not supported.']);
            }

            // Get detailed provider information
            $providers = $cardProviderManager->getAvailableProviders();
            $selectedProvider = collect($providers)->firstWhere('slug', $request->provider);

            if (!$selectedProvider) {
                return back()->withErrors(['error' => 'Card provider configuration not found.']);
            }

            // Check if provider configuration exists
            if (!$selectedProvider['config_exists']) {
                return back()->withErrors([
                    'error' => "Card provider '{$selectedProvider['name']}' is not configured. Please configure the API settings first.",
                    'config_url' => route('admin.api-config.create') . "?template={$request->provider}"
                ]);
            }

            // Check if provider is properly configured
            if (!$selectedProvider['configured']) {
                $configErrors = $selectedProvider['config_errors'];
                $apiConfig = ApiConfig::where('slug', $request->provider)->first();
                if (!$apiConfig) {
                    return back()->withErrors([
                        'error' => "API configuration for provider '{$selectedProvider['name']}' not found."
                    ]);
                }
                return back()->withErrors([
                    'error' => "Card provider '{$selectedProvider['name']}' configuration is incomplete:",
                    'config_errors' => $configErrors,
                    'config_url' => route('admin.api-config.show', ['apiConfig' => $apiConfig])
                ]);
            }

            // Check if provider is active
            if (!$selectedProvider['is_active']) {
                return back()->withErrors([
                    'error' => "Card provider '{$selectedProvider['name']}' is not active. Please activate it first.",
                    'config_url' => route('admin.api-config.show', ['apiConfig' => ApiConfig::where('slug', $request->provider)->first()])
                ]);
            }

            // Check last test status
            if ($selectedProvider['last_test_status'] === 'failed') {
                return back()->withErrors([
                    'error' => "Card provider '{$selectedProvider['name']}' last connection test failed. Please test the connection first.",
                    'config_url' => route('admin.api-config.show', ['apiConfig' => ApiConfig::where('slug', $request->provider)->first()])
                ]);
            }

            // Prepare card creation data
            $cardData = [
                'card_type' => $request->card_type,
                'brand' => $request->brand,
                'currency' => $request->currency,
                'amount' => $request->funding_amount,
                'daily_limit' => $request->daily_limit,
                'monthly_limit' => $request->monthly_limit,
                'user' => [
                    'first_name' => $user->first_name,
                    'last_name' => $user->last_name,
                    'email' => $user->email,
                    'phone' => $user->phone,
                ],
            ];

            // Create card via provider API
            $response = $cardProviderManager->createCard($request->provider, $cardData);

            if (!$response['success']) {
                return back()->withErrors([
                    'error' => 'Card creation failed: ' . $response['message'],
                    'provider_error' => true
                ]);
            }

            $providerData = $response['data'];

            // Store card in database
            $virtualCard = VirtualCard::create([
                'user_id' => $user->id,
                'card_id' => $providerData['card_id'],
                'masked_pan' => $providerData['masked_pan'],
                'card_type' => $providerData['card_type'],
                'brand' => $providerData['brand'],
                'currency' => $providerData['currency'],
                'balance' => $providerData['balance'],
                'daily_limit' => $request->daily_limit,
                'monthly_limit' => $request->monthly_limit,
                'status' => $providerData['status'],
                'provider' => $request->provider,
                'provider_reference' => $providerData['provider_reference'],
                'metadata' => $providerData['metadata'],
                'expires_at' => $providerData['expires_at'],
            ]);

            // Create audit log
            \App\Models\AuditLog::create([
                'admin_id' => auth('admin')->id(),
                'action' => 'virtual_card_assigned',
                'model' => 'VirtualCard',
                'model_id' => $virtualCard->id,
                'description' => "Assigned new {$request->card_type} {$request->brand} card via {$request->provider} to user {$user->full_name}",
                'ip_address' => request()->ip(),
            ]);

            return redirect()->route('admin.virtual-cards.show', $virtualCard)
                ->with('success', "Virtual card successfully assigned to {$user->full_name} via {$request->provider}.");

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Card assignment failed', [
                'user_id' => $request->user_id,
                'provider' => $request->provider,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return back()->withErrors([
                'error' => 'Card assignment failed: ' . $e->getMessage(),
                'technical_error' => true
            ]);
        }
    }

    /**
     * Get card analytics.
     */
    public function analytics()
    {
        try {
            // Get month format expression based on database connection
            $monthFormat = $this->getMonthFormatExpression();

            $analytics = [
                'cards_by_status' => [],
                'cards_by_type' => [],
                'monthly_creations' => [],
                'transaction_volume' => []
            ];

            // Get cards by status with error handling
            try {
                $analytics['cards_by_status'] = VirtualCard::selectRaw('status, COUNT(*) as count')
                                              ->groupBy('status')
                                              ->pluck('count', 'status')
                                              ->toArray();
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::warning('Virtual card analytics cards_by_status query failed', [
                    'error' => $e->getMessage()
                ]);
            }

            // Get cards by type with error handling
            try {
                $analytics['cards_by_type'] = VirtualCard::selectRaw('card_type, COUNT(*) as count')
                                            ->groupBy('card_type')
                                            ->pluck('count', 'card_type')
                                            ->toArray();
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::warning('Virtual card analytics cards_by_type query failed', [
                    'error' => $e->getMessage()
                ]);
            }

            // Get monthly creations with error handling
            try {
                $analytics['monthly_creations'] = VirtualCard::select([
                    DB::raw($monthFormat . ' as month'),
                    DB::raw('COUNT(*) as count')
                ])
                                                ->where('created_at', '>=', now()->subMonths(12))
                                                ->groupBy(DB::raw($monthFormat))
                                                ->orderBy(DB::raw($monthFormat))
                                                ->pluck('count', 'month')
                                                ->toArray();
            } catch (\Exception $e) {
                \Illuminate\Support\Facades\Log::warning('Virtual card analytics monthly_creations query failed', [
                    'error' => $e->getMessage()
                ]);
            }

            // Only attempt transaction volume query if virtual_card_id column exists
            try {
                if (Schema::hasColumn('transactions', 'virtual_card_id')) {
                    // Get month format for transactions table
                    $transactionMonthFormat = $this->getTransactionMonthFormatExpression();

                    $analytics['transaction_volume'] = VirtualCard::join('transactions', 'virtual_cards.id', '=', 'transactions.virtual_card_id')
                                                     ->selectRaw($transactionMonthFormat . ' as month, SUM(transactions.amount) as volume')
                                                     ->where('transactions.created_at', '>=', now()->subMonths(12))
                                                     ->where('transactions.status', 'completed')
                                                     ->groupBy(DB::raw($transactionMonthFormat))
                                                     ->orderBy(DB::raw($transactionMonthFormat))
                                                     ->pluck('volume', 'month')
                                                     ->toArray();
                }
            } catch (\Exception $e) {
                // Log the error but continue with empty transaction volume
                \Illuminate\Support\Facades\Log::warning('Virtual card analytics transaction volume query failed', [
                    'error' => $e->getMessage()
                ]);
            }

            // Ensure all arrays have default empty values and are properly formatted
            $analytics = array_map(function($data) {
                return is_array($data) ? $data : [];
            }, $analytics);

            return view('admin.virtual-cards.analytics', compact('analytics'));
            
        } catch (\Exception $e) {
            // If everything fails, log error and return empty analytics
            \Illuminate\Support\Facades\Log::error('Virtual card analytics failed completely', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            $analytics = [
                'cards_by_status' => [],
                'cards_by_type' => [],
                'monthly_creations' => [],
                'transaction_volume' => []
            ];

            return view('admin.virtual-cards.analytics', compact('analytics'));
        }
    }

    /**
     * Generate a new card number.
     */
    private function generateCardNumber()
    {
        do {
            $cardNumber = '5399' . str_pad(mt_rand(0, 999999999999), 12, '0', STR_PAD_LEFT);
        } while (VirtualCard::where('card_number', $cardNumber)->exists());

        return $cardNumber;
    }

    /**
     * Generate a new CVV.
     */
    private function generateCvv()
    {
        return str_pad(mt_rand(0, 999), 3, '0', STR_PAD_LEFT);
    }

    /**
     * Get month format expression based on database connection.
     * Returns SQL expression for formatting dates as YYYY-MM.
     */
    private function getMonthFormatExpression(): string
    {
        $driver = config('database.default');
        $connection = config("database.connections.{$driver}.driver");
        
        switch ($connection) {
            case 'sqlite':
                return "strftime('%Y-%m', created_at)";
            case 'pgsql':
                return "to_char(created_at, 'YYYY-MM')";
            default: // mysql, mariadb
                return "DATE_FORMAT(created_at, '%Y-%m')";
        }
    }

    /**
     * Get month format expression for transactions table.
     * Returns SQL expression for formatting transaction dates as YYYY-MM.
     */
    private function getTransactionMonthFormatExpression(): string
    {
        $driver = config('database.default');
        $connection = config("database.connections.{$driver}.driver");
        
        switch ($connection) {
            case 'sqlite':
                return "strftime('%Y-%m', transactions.created_at)";
            case 'pgsql':
                return "to_char(transactions.created_at, 'YYYY-MM')";
            default: // mysql, mariadb
                return "DATE_FORMAT(transactions.created_at, '%Y-%m')";
        }
    }
}