<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use App\Models\User;
use App\Models\KycDocument;
use App\Services\BvnService;
use App\Services\NinService;

/**
 * KYC API Controller
 * 
 * TODO: Add comprehensive document validation
 * TODO: Add automated identity verification
 * TODO: Add KYC level management (tier 1, 2, 3)
 * TODO: Add compliance audit trails
 * TODO: Add document expiry tracking
 */
class KycController extends Controller
{
    protected $bvnService;
    protected $ninService;

    public function __construct(BvnService $bvnService, NinService $ninService)
    {
        $this->bvnService = $bvnService;
        $this->ninService = $ninService;
    }

    /**
     * Get user's KYC status and information
     * 
     * TODO: Add KYC score calculation
     * TODO: Add next required steps guidance
     */
    public function getStatus(Request $request)
    {
        try {
            $user = $request->user();
            
            $kycDocuments = KycDocument::where('user_id', $user->id)
                ->get()
                ->keyBy('type');

            $kycStatus = [
                'kyc_verified' => $user->kyc_verified,
                'kyc_level' => $user->kyc_level ?? 'tier_0',
                'bvn_verified' => $user->bvn_verified,
                'nin_verified' => $user->nin_verified,
                'documents' => [
                    'bvn' => $kycDocuments->get('bvn')?->status ?? 'not_submitted',
                    'nin' => $kycDocuments->get('nin')?->status ?? 'not_submitted',
                    'government_id' => $kycDocuments->get('government_id')?->status ?? 'not_submitted',
                    'proof_of_address' => $kycDocuments->get('proof_of_address')?->status ?? 'not_submitted',
                    'selfie' => $kycDocuments->get('selfie')?->status ?? 'not_submitted',
                ],
                'required_for_tier_1' => ['bvn', 'government_id', 'selfie'],
                'required_for_tier_2' => ['bvn', 'nin', 'government_id', 'proof_of_address', 'selfie'],
                'required_for_tier_3' => ['bvn', 'nin', 'government_id', 'proof_of_address', 'selfie', 'business_documents']
            ];

            return response()->json([
                'status' => 'success',
                'message' => 'KYC status retrieved successfully',
                'data' => $kycStatus
            ]);

        } catch (\Exception $e) {
            // TODO: Add proper error logging
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to retrieve KYC status',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Verify BVN
     * 
     * TODO: Add BVN format validation
     * TODO: Add fraud detection for BVN verification
     * TODO: Add BVN data cross-referencing
     */
    public function verifyBvn(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'bvn' => 'required|string|size:11|regex:/^[0-9]+$/',
                'date_of_birth' => 'required|date|before:today',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $user = $request->user();

            // Check if BVN is already verified for this user
            if ($user->bvn_verified) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'BVN already verified for this account'
                ], 400);
            }

            // TODO: Check if BVN is already used by another user

            // Verify BVN with service provider
            $bvnData = [
                'bvn' => $request->bvn,
                'date_of_birth' => $request->date_of_birth,
                'first_name' => $user->first_name,
                'last_name' => $user->last_name,
                'phone' => $user->phone
            ];

            $verificationResponse = $this->bvnService->verifyBvn($bvnData);

            if (!$verificationResponse['success']) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'BVN verification failed',
                    'error' => $verificationResponse['message']
                ], 400);
            }

            $bvnInfo = $verificationResponse['data'];

            // Create or update KYC document record
            KycDocument::updateOrCreate([
                'user_id' => $user->id,
                'type' => 'bvn'
            ], [
                'document_number' => $request->bvn,
                'status' => 'verified',
                'verified_at' => now(),
                'verification_data' => json_encode($bvnInfo)
            ]);

            // Update user BVN info
            $user->update([
                'bvn' => $request->bvn,
                'bvn_verified' => true,
                'bvn_verified_at' => now(),
                // TODO: Update other user info from BVN if needed
            ]);

            // TODO: Check if user can be upgraded to higher KYC tier
            // TODO: Send BVN verification notification
            // TODO: Log verification event

            return response()->json([
                'status' => 'success',
                'message' => 'BVN verified successfully',
                'data' => [
                    'bvn_verified' => true,
                    'verification_date' => now(),
                    'kyc_level' => $user->fresh()->kyc_level
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'BVN verification failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Verify NIN
     * 
     * TODO: Add NIN format validation
     * TODO: Add fraud detection for NIN verification
     */
    public function verifyNin(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'nin' => 'required|string|size:11|regex:/^[0-9]+$/',
                'date_of_birth' => 'required|date|before:today',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $user = $request->user();

            // Check if NIN is already verified for this user
            if ($user->nin_verified) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'NIN already verified for this account'
                ], 400);
            }

            // TODO: Check if NIN is already used by another user

            // Verify NIN with service provider
            $ninData = [
                'nin' => $request->nin,
                'date_of_birth' => $request->date_of_birth,
                'first_name' => $user->first_name,
                'last_name' => $user->last_name,
                'phone' => $user->phone
            ];

            $verificationResponse = $this->ninService->verifyNin($ninData);

            if (!$verificationResponse['success']) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'NIN verification failed',
                    'error' => $verificationResponse['message']
                ], 400);
            }

            $ninInfo = $verificationResponse['data'];

            // Create or update KYC document record
            KycDocument::updateOrCreate([
                'user_id' => $user->id,
                'type' => 'nin'
            ], [
                'document_number' => $request->nin,
                'status' => 'verified',
                'verified_at' => now(),
                'verification_data' => json_encode($ninInfo)
            ]);

            // Update user NIN info
            $user->update([
                'nin' => $request->nin,
                'nin_verified' => true,
                'nin_verified_at' => now(),
            ]);

            // TODO: Check if user can be upgraded to higher KYC tier
            // TODO: Send NIN verification notification

            return response()->json([
                'status' => 'success',
                'message' => 'NIN verified successfully',
                'data' => [
                    'nin_verified' => true,
                    'verification_date' => now(),
                    'kyc_level' => $user->fresh()->kyc_level
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'NIN verification failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Upload KYC document
     * 
     * TODO: Add document type validation
     * TODO: Add file format and size validation
     * TODO: Add automated document verification
     * TODO: Add document quality checks
     */
    public function uploadDocument(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'document_type' => 'required|string|in:government_id,proof_of_address,selfie,business_document',
                'document' => 'required|file|mimes:jpg,jpeg,png,pdf|max:5120', // 5MB max
                'document_number' => 'nullable|string|max:50',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $user = $request->user();

            // TODO: Add document quality validation
            // TODO: Add OCR for document text extraction
            // TODO: Add facial recognition for selfie verification

            // Store the uploaded file
            $file = $request->file('document');
            $fileName = time() . '_' . $user->id . '_' . $request->document_type . '.' . $file->getClientOriginalExtension();
            $filePath = $file->storeAs('kyc_documents', $fileName, 'private');

            // Create or update KYC document record
            $kycDocument = KycDocument::updateOrCreate([
                'user_id' => $user->id,
                'type' => $request->document_type
            ], [
                'document_number' => $request->document_number,
                'file_path' => $filePath,
                'file_name' => $file->getClientOriginalName(),
                'file_size' => $file->getSize(),
                'mime_type' => $file->getMimeType(),
                'status' => 'pending',
                'uploaded_at' => now(),
                'metadata' => json_encode([
                    'original_name' => $file->getClientOriginalName(),
                    'uploaded_from' => request()->ip()
                ])
            ]);

            // TODO: Queue document for automated verification
            // TODO: Send upload confirmation notification
            // TODO: Log document upload event

            return response()->json([
                'status' => 'success',
                'message' => 'Document uploaded successfully',
                'data' => [
                    'document_id' => $kycDocument->id,
                    'document_type' => $kycDocument->type,
                    'status' => $kycDocument->status,
                    'uploaded_at' => $kycDocument->uploaded_at
                ]
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Document upload failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get uploaded documents
     * 
     * TODO: Add document download functionality
     * TODO: Add document history tracking
     */
    public function getDocuments(Request $request)
    {
        try {
            $user = $request->user();
            
            $documents = KycDocument::where('user_id', $user->id)
                ->select(['id', 'type', 'status', 'uploaded_at', 'verified_at', 'rejection_reason'])
                ->orderBy('uploaded_at', 'desc')
                ->get();

            return response()->json([
                'status' => 'success',
                'message' => 'Documents retrieved successfully',
                'data' => $documents
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to retrieve documents',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Resubmit rejected document
     * 
     * TODO: Add resubmission limits
     * TODO: Add rejection reason tracking
     */
    public function resubmitDocument(Request $request, $documentId)
    {
        try {
            $validator = Validator::make($request->all(), [
                'document' => 'required|file|mimes:jpg,jpeg,png,pdf|max:5120',
                'document_number' => 'nullable|string|max:50',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Validation failed',
                    'errors' => $validator->errors()
                ], 400);
            }

            $user = $request->user();
            
            $kycDocument = KycDocument::where('id', $documentId)
                ->where('user_id', $user->id)
                ->where('status', 'rejected')
                ->first();

            if (!$kycDocument) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Document not found or cannot be resubmitted'
                ], 404);
            }

            // Delete old file
            if ($kycDocument->file_path) {
                Storage::disk('private')->delete($kycDocument->file_path);
            }

            // Store new file
            $file = $request->file('document');
            $fileName = time() . '_' . $user->id . '_' . $kycDocument->type . '.' . $file->getClientOriginalExtension();
            $filePath = $file->storeAs('kyc_documents', $fileName, 'private');

            // Update document record
            $kycDocument->update([
                'document_number' => $request->document_number ?? $kycDocument->document_number,
                'file_path' => $filePath,
                'file_name' => $file->getClientOriginalName(),
                'file_size' => $file->getSize(),
                'mime_type' => $file->getMimeType(),
                'status' => 'pending',
                'uploaded_at' => now(),
                'rejection_reason' => null,
                'resubmission_count' => $kycDocument->resubmission_count + 1
            ]);

            // TODO: Queue for verification
            // TODO: Send resubmission notification

            return response()->json([
                'status' => 'success',
                'message' => 'Document resubmitted successfully',
                'data' => [
                    'document_id' => $kycDocument->id,
                    'status' => $kycDocument->status,
                    'uploaded_at' => $kycDocument->uploaded_at
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Document resubmission failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}