<?php

namespace App\Services\Airtime\Providers;

use App\Services\Airtime\BaseAirtimeProvider;

class ClubkonnectProvider extends BaseAirtimeProvider
{
    protected string $username;
    protected string $password;
    protected string $apiKey;

    /**
     * Configure Clubkonnect provider.
     */
    public function configure(array $credentials): void
    {
        parent::configure($credentials);
        
        $this->username = $credentials['username'] ?? '';
        $this->password = $credentials['password'] ?? '';
        $this->apiKey = $credentials['api_key'] ?? '';
        
        if (empty($this->username) || empty($this->password)) {
            throw new \InvalidArgumentException('Clubkonnect username and password are required');
        }
    }

    /**
     * Purchase airtime via Clubkonnect.
     */
    public function purchaseAirtime(array $params): array
    {
        $this->validateParameters($params, ['phone_number', 'amount', 'network']);

        $networkCode = $this->getNetworkCode($params['network']);
        $requestId = $this->generateReference('CLUB_AIR');

        $data = [
            'username' => $this->username,
            'password' => $this->password,
            'phone' => $this->formatPhoneNumber($params['phone_number']),
            'network_id' => $networkCode,
            'amount' => $params['amount'],
            'client_reference' => $requestId,
        ];

        $response = $this->makeRequest('POST', '/api/topup/', $data);

        if (!$response['success']) {
            return [
                'success' => false,
                'message' => 'Network error: ' . ($response['error'] ?? 'Unknown error'),
                'reference' => $requestId,
                'provider_response' => $response,
            ];
        }

        $responseData = $response['data'];
        $success = isset($responseData['status']) && $responseData['status'] === 'success';

        return [
            'success' => $success,
            'message' => $responseData['message'] ?? ($success ? 'Transaction successful' : 'Transaction failed'),
            'reference' => $requestId,
            'provider_reference' => $responseData['reference'] ?? $requestId,
            'amount' => $params['amount'],
            'phone_number' => $params['phone_number'],
            'network' => $params['network'],
            'provider_response' => $response,
        ];
    }

    /**
     * Purchase data bundle via Clubkonnect.
     */
    public function purchaseData(array $params): array
    {
        $this->validateParameters($params, ['phone_number', 'data_plan', 'network']);

        $networkCode = $this->getNetworkCode($params['network']);
        $requestId = $this->generateReference('CLUB_DATA');

        $data = [
            'username' => $this->username,
            'password' => $this->password,
            'phone' => $this->formatPhoneNumber($params['phone_number']),
            'network_id' => $networkCode,
            'plan_id' => $params['data_plan'],
            'client_reference' => $requestId,
        ];

        $response = $this->makeRequest('POST', '/api/data/', $data);

        if (!$response['success']) {
            return [
                'success' => false,
                'message' => 'Network error: ' . ($response['error'] ?? 'Unknown error'),
                'reference' => $requestId,
                'provider_response' => $response,
            ];
        }

        $responseData = $response['data'];
        $success = isset($responseData['status']) && $responseData['status'] === 'success';

        return [
            'success' => $success,
            'message' => $responseData['message'] ?? ($success ? 'Transaction successful' : 'Transaction failed'),
            'reference' => $requestId,
            'provider_reference' => $responseData['reference'] ?? $requestId,
            'amount' => $responseData['amount'] ?? $params['amount'],
            'phone_number' => $params['phone_number'],
            'network' => $params['network'],
            'data_plan' => $params['data_plan'],
            'provider_response' => $response,
        ];
    }

    /**
     * Get available data plans for a network.
     */
    public function getDataPlans(string $network): array
    {
        $networkCode = $this->getNetworkCode($network);
        
        $data = [
            'username' => $this->username,
            'password' => $this->password,
            'network_id' => $networkCode,
        ];

        $response = $this->makeRequest('POST', '/api/dataplans/', $data);

        if (!$response['success']) {
            return [];
        }

        $responseData = $response['data'];
        if (!isset($responseData['plans']) || !is_array($responseData['plans'])) {
            return [];
        }

        $plans = [];
        foreach ($responseData['plans'] as $plan) {
            $plans[] = [
                'id' => $plan['plan_id'],
                'name' => $plan['plan_name'],
                'amount' => $plan['amount'],
                'validity' => $plan['validity'] ?? null,
                'data_volume' => $plan['data_volume'] ?? null,
            ];
        }

        return $plans;
    }

    /**
     * Check transaction status.
     */
    public function checkTransactionStatus(string $reference): array
    {
        $data = [
            'username' => $this->username,
            'password' => $this->password,
            'reference' => $reference,
        ];

        $response = $this->makeRequest('POST', '/api/status/', $data);

        if (!$response['success']) {
            return [
                'success' => false,
                'status' => 'unknown',
                'message' => 'Failed to query status',
            ];
        }

        $responseData = $response['data'];
        $status = 'pending';

        if (isset($responseData['status'])) {
            $clubStatus = strtolower($responseData['status']);
            $status = match($clubStatus) {
                'success', 'successful', 'delivered' => 'completed',
                'failed', 'error' => 'failed',
                default => 'pending',
            };
        }

        return [
            'success' => true,
            'status' => $status,
            'message' => $responseData['message'] ?? 'Status checked',
            'provider_response' => $response,
        ];
    }

    /**
     * Get account balance.
     */
    public function getBalance(): array
    {
        $data = [
            'username' => $this->username,
            'password' => $this->password,
        ];

        $response = $this->makeRequest('POST', '/api/balance/', $data);

        if (!$response['success']) {
            return [
                'success' => false,
                'balance' => 0,
                'message' => 'Failed to get balance',
            ];
        }

        $responseData = $response['data'];
        
        return [
            'success' => true,
            'balance' => $responseData['balance'] ?? 0,
            'currency' => 'NGN',
            'message' => 'Balance retrieved successfully',
        ];
    }

    /**
     * Test connection to Clubkonnect.
     */
    public function testConnection(): array
    {
        try {
            $balanceResponse = $this->getBalance();
            
            return [
                'success' => $balanceResponse['success'],
                'message' => $balanceResponse['success'] ? 'Connection successful' : 'Connection failed',
                'response_time' => 0,
                'provider_response' => $balanceResponse,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Connection test failed: ' . $e->getMessage(),
                'response_time' => 0,
            ];
        }
    }

    /**
     * Verify phone number and detect network.
     */
    public function verifyPhoneNumber(string $phoneNumber): array
    {
        $network = $this->detectNetwork($phoneNumber);
        
        return [
            'success' => !is_null($network),
            'phone_number' => $this->formatPhoneNumber($phoneNumber),
            'network' => $network,
            'message' => $network ? 'Phone number verified' : 'Network not detected',
        ];
    }

    /**
     * Get Clubkonnect network code for network.
     */
    protected function getNetworkCode(string $network): int
    {
        $networkCodes = [
            'mtn' => 1,
            'glo' => 2,
            'airtel' => 3,
            '9mobile' => 4,
        ];

        return $networkCodes[$network] ?? 1;
    }

    /**
     * Get default headers for Clubkonnect API.
     */
    protected function getDefaultHeaders(): array
    {
        $headers = parent::getDefaultHeaders();
        
        if (!empty($this->apiKey)) {
            $headers['Authorization'] = 'Bearer ' . $this->apiKey;
        }
        
        return $headers;
    }
}