<?php

namespace App\Services\CardProviders;

interface CardProviderInterface
{
    /**
     * Create a new virtual card for a user.
     *
     * @param array $data Card creation data
     * @return array Card creation response with provider-specific data
     */
    public function createCard(array $data): array;

    /**
     * Get card details from provider.
     *
     * @param string $cardId Provider card ID
     * @return array Card details
     */
    public function getCard(string $cardId): array;

    /**
     * Fund a virtual card.
     *
     * @param string $cardId Provider card ID
     * @param float $amount Amount to fund
     * @return array Funding response
     */
    public function fundCard(string $cardId, float $amount): array;

    /**
     * Freeze a virtual card.
     *
     * @param string $cardId Provider card ID
     * @return array Freeze response
     */
    public function freezeCard(string $cardId): array;

    /**
     * Unfreeze a virtual card.
     *
     * @param string $cardId Provider card ID
     * @return array Unfreeze response
     */
    public function unfreezeCard(string $cardId): array;

    /**
     * Terminate a virtual card.
     *
     * @param string $cardId Provider card ID
     * @return array Termination response
     */
    public function terminateCard(string $cardId): array;

    /**
     * Get card transaction history.
     *
     * @param string $cardId Provider card ID
     * @param array $filters Optional filters
     * @return array Transaction history
     */
    public function getTransactions(string $cardId, array $filters = []): array;

    /**
     * Check if the provider is properly configured.
     *
     * @return bool
     */
    public function isConfigured(): bool;

    /**
     * Get provider name.
     *
     * @return string
     */
    public function getProviderName(): string;

    /**
     * Get provider slug.
     *
     * @return string
     */
    public function getProviderSlug(): string;
}