<?php

namespace App\Services;

use App\Models\ApiConfig;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Exception;

class FlutterwaveService
{
    protected $config;
    protected $apiConfig;

    public function __construct()
    {
        // Don't load config in constructor to avoid errors during application bootstrap
        // Config will be loaded when needed in other methods
    }

    /**
     * Get API configuration.
     */
    protected function getApiConfig(): ?ApiConfig
    {
        if (!$this->apiConfig) {
            $this->apiConfig = ApiConfig::where('slug', 'flutterwave')->where('is_active', true)->first();
            if ($this->apiConfig) {
                $this->config = $this->apiConfig->config ?? [];
            }
        }
        return $this->apiConfig;
    }

    /**
     * Ensure service is configured before API calls.
     */
    protected function ensureConfigured(): void
    {
        if (!$this->getApiConfig()) {
            throw new Exception('Flutterwave API configuration not found or not active');
        }
    }

    /**
     * Initialize payment transaction.
     */
    public function initializePayment(array $data): array
    {
        try {
            $this->ensureConfigured();
            $payload = [
                'tx_ref' => $data['reference'],
                'amount' => $data['amount'],
                'currency' => $data['currency'] ?? 'NGN',
                'redirect_url' => $data['callback_url'] ?? config('app.url'),
                'customer' => [
                    'email' => $data['email'],
                    'name' => $data['customer_name'] ?? '',
                    'phonenumber' => $data['phone'] ?? '',
                ],
                'customizations' => [
                    'title' => $data['title'] ?? config('app.name'),
                    'description' => $data['description'] ?? 'Payment',
                    'logo' => $data['logo'] ?? '',
                ],
                'meta' => $data['metadata'] ?? [],
            ];

            $response = $this->makeApiCall('/payments', $payload, 'POST');

            if (isset($response['status']) && $response['status'] === 'success') {
                return [
                    'success' => true,
                    'data' => [
                        'authorization_url' => $response['data']['link'],
                        'access_code' => $response['data']['id'],
                        'reference' => $data['reference'],
                    ],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Payment initialization failed',
                'data' => $response,
            ];

        } catch (Exception $e) {
            Log::error('Flutterwave payment initialization failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Payment initialization failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Verify payment transaction.
     */
    public function verifyPayment(string $transactionId): array
    {
        try {
            $this->ensureConfigured();
            $response = $this->makeApiCall("/transactions/{$transactionId}/verify", [], 'GET');

            if (isset($response['status']) && $response['status'] === 'success') {
                $data = $response['data'];
                
                return [
                    'success' => true,
                    'data' => [
                        'reference' => $data['tx_ref'],
                        'amount' => $data['amount'],
                        'currency' => $data['currency'],
                        'status' => $data['status'],
                        'gateway_response' => $data['processor_response'],
                        'paid_at' => $data['created_at'],
                        'channel' => $data['payment_type'],
                        'fees' => $data['app_fee'] ?? 0,
                        'customer' => [
                            'email' => $data['customer']['email'] ?? '',
                            'name' => $data['customer']['name'] ?? '',
                        ],
                    ],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Payment verification failed',
                'data' => $response,
            ];

        } catch (Exception $e) {
            Log::error('Flutterwave payment verification failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Payment verification failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get list of Nigerian banks.
     */
    public function getBanks(): array
    {
        $this->ensureConfigured();
        try {
            $response = $this->makeApiCall('/banks/NG', [], 'GET');

            if (isset($response['status']) && $response['status'] === 'success') {
                return [
                    'success' => true,
                    'data' => $response['data'],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Failed to fetch banks',
                'data' => [],
            ];

        } catch (Exception $e) {
            Log::error('Flutterwave get banks failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to fetch banks: ' . $e->getMessage(),
                'data' => [],
            ];
        }
    }

    /**
     * Create transfer.
     */
    public function createTransfer(array $data): array
    {
        $this->ensureConfigured();
        try {
            $payload = [
                'account_bank' => $data['bank_code'],
                'account_number' => $data['account_number'],
                'amount' => $data['amount'],
                'narration' => $data['narration'] ?? 'Transfer',
                'currency' => $data['currency'] ?? 'NGN',
                'reference' => $data['reference'],
                'callback_url' => $data['callback_url'] ?? config('app.url'),
                'debit_currency' => $data['currency'] ?? 'NGN',
            ];

            $response = $this->makeApiCall('/transfers', $payload, 'POST');

            if (isset($response['status']) && $response['status'] === 'success') {
                return [
                    'success' => true,
                    'data' => [
                        'id' => $response['data']['id'],
                        'reference' => $response['data']['reference'],
                        'status' => $response['data']['status'],
                        'amount' => $response['data']['amount'],
                        'fee' => $response['data']['fee'],
                    ],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Transfer failed',
                'data' => $response,
            ];

        } catch (Exception $e) {
            Log::error('Flutterwave transfer failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Transfer failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Verify bank account.
     */
    public function verifyAccount(string $accountNumber, string $bankCode): array
    {
        $this->ensureConfigured();
        try {
            $response = $this->makeApiCall('/accounts/resolve', [
                'account_number' => $accountNumber,
                'account_bank' => $bankCode,
            ], 'POST');

            if (isset($response['status']) && $response['status'] === 'success') {
                return [
                    'success' => true,
                    'data' => [
                        'account_number' => $response['data']['account_number'],
                        'account_name' => $response['data']['account_name'],
                    ],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Account verification failed',
            ];

        } catch (Exception $e) {
            Log::error('Flutterwave account verification failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Account verification failed: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get transaction by reference.
     */
    public function getTransaction(string $reference): array
    {
        $this->ensureConfigured();
        try {
            $response = $this->makeApiCall('/transactions', ['tx_ref' => $reference], 'GET');

            if (isset($response['status']) && $response['status'] === 'success' && !empty($response['data'])) {
                return [
                    'success' => true,
                    'data' => $response['data'][0],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Transaction not found',
            ];

        } catch (Exception $e) {
            Log::error('Flutterwave get transaction failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Failed to get transaction: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Make API call to Flutterwave.
     */
    protected function makeApiCall(string $endpoint, array $data = [], string $method = 'GET'): array
    {
        $url = rtrim($this->getApiConfig()->effective_base_url, '/') . $endpoint;
        $timeout = $this->config['timeout'] ?? 30;
        $retryAttempts = $this->config['retry_attempts'] ?? 3;

        $headers = [
            'Authorization' => 'Bearer ' . $this->getApiConfig()->secret_key,
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
        ];

        for ($attempt = 1; $attempt <= $retryAttempts; $attempt++) {
            try {
                $request = Http::withHeaders($headers)->timeout($timeout);

                $response = match (strtoupper($method)) {
                    'GET' => $request->get($url, $data),
                    'POST' => $request->post($url, $data),
                    'PUT' => $request->put($url, $data),
                    'DELETE' => $request->delete($url, $data),
                    default => throw new Exception("Unsupported HTTP method: {$method}"),
                };

                if ($response->successful()) {
                    return $response->json();
                }

                if ($attempt === $retryAttempts) {
                    throw new Exception("API call failed with status: {$response->status()}, Body: {$response->body()}");
                }

                sleep($attempt); // Progressive delay

            } catch (Exception $e) {
                if ($attempt === $retryAttempts) {
                    throw $e;
                }
                sleep($attempt);
            }
        }

        throw new Exception('Max retry attempts exceeded');
    }

    /**
     * Verify webhook signature.
     */
    public function verifyWebhookSignature(string $payload, string $signature): bool
    {
        $secretHash = $this->getApiConfig()?->secret_key;
        if (!$secretHash) {
            return false;
        }
        
        $expectedSignature = hash_hmac('sha256', $payload, $secretHash);
        
        return hash_equals($expectedSignature, $signature);
    }

    /**
     * Check if service is configured and active.
     */
    public function isConfigured(): bool
    {
        $config = $this->getApiConfig();
        return $config && 
               $config->is_active && 
               !empty($config->secret_key);
    }

    /**
     * Get service configuration.
     */
    public function getConfig(): ?ApiConfig
    {
        return $this->getApiConfig();
    }

    /**
     * Test API connection.
     */
    public function testConnection(): array
    {
        try {
            return $this->getBanks();
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Connection test failed: ' . $e->getMessage(),
            ];
        }
    }
}