<?php

namespace Database\Seeders;

use App\Models\Admin;
use App\Models\AuditLog;
use App\Models\User;
use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;

class AuditLogSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get existing admin users or create dummy ones
        $admins = Admin::take(3)->get();
        if ($admins->isEmpty()) {
            // Create sample admin users if none exist
            $admins = collect([
                Admin::firstOrCreate([
                    'email' => 'admin@swiftpay.ng'
                ], [
                    'name' => 'Super Admin',
                    'password' => bcrypt('password'),
                    'role' => 'super_admin',
                    'is_active' => true,
                ]),
                Admin::firstOrCreate([
                    'email' => 'support@swiftpay.ng'
                ], [
                    'name' => 'Support Admin',
                    'password' => bcrypt('password'),
                    'role' => 'admin',
                    'is_active' => true,
                ]),
            ]);
        }

        // Sample audit log data
        $auditLogData = [
            [
                'action' => 'admin_login',
                'ip_address' => '192.168.1.100',
                'data' => ['success' => true, 'login_method' => 'password'],
                'metadata' => ['browser' => 'Chrome', 'os' => 'Windows'],
            ],
            [
                'action' => 'update_site_settings',
                'target_type' => 'Settings',
                'old_values' => ['business.name' => 'Old Name'],
                'new_values' => ['business.name' => 'SwiftPay'],
                'ip_address' => '192.168.1.100',
                'metadata' => ['section' => 'site_settings'],
            ],
            [
                'action' => 'user_kyc_approval',
                'target_type' => 'KycVerification',
                'target_id' => 1,
                'ip_address' => '10.0.0.50',
                'data' => ['kyc_type' => 'bvn', 'status' => 'approved'],
                'metadata' => ['reviewer_notes' => 'Documents verified successfully'],
            ],
            [
                'action' => 'send_broadcast_notification',
                'target_type' => 'Notification',
                'ip_address' => '192.168.1.100',
                'data' => [
                    'title' => 'System Maintenance',
                    'recipients' => 'all_users',
                    'channels' => ['in_app', 'email']
                ],
                'metadata' => ['user_count' => 1250],
            ],
            [
                'action' => 'admin_login',
                'ip_address' => '203.123.45.67',
                'data' => ['success' => false, 'reason' => 'invalid_password'],
                'metadata' => ['attempt_count' => 1],
            ],
            [
                'action' => 'admin_login',
                'ip_address' => '203.123.45.67',
                'data' => ['success' => false, 'reason' => 'invalid_password'],
                'metadata' => ['attempt_count' => 2],
            ],
            [
                'action' => 'admin_login',
                'ip_address' => '203.123.45.67',
                'data' => ['success' => false, 'reason' => 'invalid_password'],
                'metadata' => ['attempt_count' => 3],
            ],
            [
                'action' => 'update_fees_and_limits',
                'target_type' => 'Settings',
                'ip_address' => '192.168.1.100',
                'old_values' => ['fees.transfer.value' => 1.0],
                'new_values' => ['fees.transfer.value' => 1.5],
                'metadata' => ['section' => 'fees_and_limits'],
            ],
            [
                'action' => 'export_security_logs',
                'target_type' => 'AuditLog',
                'ip_address' => '192.168.1.100',
                'data' => ['format' => 'csv', 'record_count' => 150],
                'metadata' => ['date_range' => '2024-01-01 to 2024-01-31'],
            ],
            [
                'action' => 'view_user_details',
                'target_type' => 'User',
                'target_id' => 1,
                'ip_address' => '192.168.1.105',
                'metadata' => ['section' => 'user_management'],
            ],
        ];

        // Create audit logs with random dates and admin assignments
        foreach ($auditLogData as $index => $logData) {
            $admin = $admins->random();
            
            AuditLog::create(array_merge($logData, [
                'admin_id' => $admin->id,
                'user_agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
                'created_at' => now()->subDays(rand(0, 30))->subHours(rand(0, 23))->subMinutes(rand(0, 59)),
                'updated_at' => now()->subDays(rand(0, 30))->subHours(rand(0, 23))->subMinutes(rand(0, 59)),
            ]));
        }

        // Add some recent activity (last 24 hours)
        $recentActions = ['admin_login', 'view_user_details', 'update_notification_setting'];
        foreach ($recentActions as $action) {
            AuditLog::create([
                'admin_id' => $admins->random()->id,
                'action' => $action,
                'ip_address' => '192.168.1.' . rand(100, 200),
                'user_agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
                'data' => ['success' => true],
                'metadata' => ['timestamp' => now()],
                'created_at' => now()->subHours(rand(1, 24)),
                'updated_at' => now()->subHours(rand(1, 24)),
            ]);
        }

        $this->command->info('Audit logs seeded successfully!');
    }
}