/**
 * UI Update Service
 * Handles selective UI updates instead of full page reloads for better UX
 */
class UIUpdateService {
    constructor() {
        this.updateIntervals = {};
        this.config = {
            updateIntervals: {
                dashboard: 30000, // 30 seconds
                transactions: 15000, // 15 seconds
                orders: 10000, // 10 seconds
            },
            alertTimeout: 5000, // 5 seconds
        };
    }

    /**
     * Initialize the service with configuration from Laravel
     */
    init(config = {}) {
        this.config = { ...this.config, ...config };
    }

    /**
     * Refresh specific section of the page instead of full reload
     */
    refreshSection(sectionId, url = null, callback = null) {
        const section = document.getElementById(sectionId);
        if (!section) {
            console.warn(`Section ${sectionId} not found`);
            return;
        }

        // Add loading state
        this.addLoadingState(section);

        // Use current URL if not provided
        const refreshUrl = url || window.location.href;

        fetch(refreshUrl, {
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'Accept': 'text/html,application/xhtml+xml',
            }
        })
        .then(response => response.text())
        .then(html => {
            // Parse the response HTML safely using DOMParser
            const parser = new DOMParser();
            const doc = parser.parseFromString(html, 'text/html');
            
            // Find the corresponding section in the parsed document
            const newSection = doc.querySelector(`#${sectionId}`);
            if (newSection) {
                // Replace the section's children with the new content nodes
                section.replaceChildren(...Array.from(newSection.childNodes));
                this.removeLoadingState(section);
                
                // Execute callback if provided
                if (callback && typeof callback === 'function') {
                    callback(section);
                }
                
                // Trigger custom event
                section.dispatchEvent(new CustomEvent('sectionRefreshed', {
                    detail: { sectionId }
                }));
            } else {
                console.warn(`Section ${sectionId} not found in response`);
                this.removeLoadingState(section);
            }
        })
        .catch(error => {
            console.error('Error refreshing section:', error);
            this.removeLoadingState(section);
            this.showAlert('error', 'Failed to refresh content. Please try again.');
        });
    }

    /**
     * Refresh multiple sections simultaneously
     */
    refreshSections(sections) {
        sections.forEach(({ id, url, callback }) => {
            this.refreshSection(id, url, callback);
        });
    }

    /**
     * Start auto-refresh for a section
     */
    startAutoRefresh(sectionId, interval = null, url = null) {
        const refreshInterval = interval || this.config.updateIntervals.dashboard;
        
        // Clear existing interval if any
        if (this.updateIntervals[sectionId]) {
            clearInterval(this.updateIntervals[sectionId]);
        }

        this.updateIntervals[sectionId] = setInterval(() => {
            this.refreshSection(sectionId, url);
        }, refreshInterval);
    }

    /**
     * Stop auto-refresh for a section
     */
    stopAutoRefresh(sectionId) {
        if (this.updateIntervals[sectionId]) {
            clearInterval(this.updateIntervals[sectionId]);
            delete this.updateIntervals[sectionId];
        }
    }

    /**
     * Update statistics/counters without page reload
     */
    updateStats(stats) {
        Object.keys(stats).forEach(statId => {
            const element = document.getElementById(statId);
            if (element) {
                this.animateNumberChange(element, stats[statId]);
            }
        });
    }

    /**
     * Animate number changes in statistics
     */
    animateNumberChange(element, newValue) {
        const oldValue = parseFloat(element.textContent.replace(/[^\d.-]/g, '')) || 0;
        const difference = newValue - oldValue;
        
        if (difference === 0) return;

        const duration = 1000; // 1 second
        const steps = 30;
        const stepValue = difference / steps;
        const stepTime = duration / steps;

        let currentStep = 0;
        const interval = setInterval(() => {
            currentStep++;
            const currentValue = oldValue + (stepValue * currentStep);
            
            // Format the number appropriately
            if (element.dataset.format === 'currency') {
                element.textContent = '₦' + this.formatNumber(currentValue);
            } else if (element.dataset.format === 'percentage') {
                element.textContent = currentValue.toFixed(1) + '%';
            } else {
                element.textContent = this.formatNumber(currentValue);
            }

            if (currentStep >= steps) {
                clearInterval(interval);
                // Ensure final value is exact
                if (element.dataset.format === 'currency') {
                    element.textContent = '₦' + this.formatNumber(newValue);
                } else if (element.dataset.format === 'percentage') {
                    element.textContent = newValue.toFixed(1) + '%';
                } else {
                    element.textContent = this.formatNumber(newValue);
                }
            }
        }, stepTime);
    }

    /**
     * Format numbers with appropriate separators
     */
    formatNumber(num) {
        return new Intl.NumberFormat('en-NG').format(num);
    }

    /**
     * Add loading state to element
     */
    addLoadingState(element) {
        element.classList.add('opacity-50', 'pointer-events-none');
        
        // Add loading indicator if not present
        if (!element.querySelector('.loading-indicator')) {
            const loader = document.createElement('div');
            loader.className = 'loading-indicator absolute inset-0 flex items-center justify-center bg-white bg-opacity-75';
            loader.innerHTML = `
                <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
            `;
            element.style.position = 'relative';
            element.appendChild(loader);
        }
    }

    /**
     * Remove loading state from element
     */
    removeLoadingState(element) {
        element.classList.remove('opacity-50', 'pointer-events-none');
        
        const loader = element.querySelector('.loading-indicator');
        if (loader) {
            loader.remove();
        }
    }

    /**
     * Show alert notification
     */
    showAlert(type, message, duration = null) {
        const alertDiv = document.createElement('div');
        const alertId = 'alert-' + Date.now();
        alertDiv.id = alertId;
        alertDiv.className = `fixed top-4 right-4 z-50 p-4 rounded-lg shadow-lg max-w-sm transform translate-x-full transition-transform duration-300 ${this.getAlertClasses(type)}`;
        
        alertDiv.innerHTML = `
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    ${this.getAlertIcon(type)}
                </div>
                <div class="ml-3">
                    <p class="text-sm font-medium">${message}</p>
                </div>
                <div class="ml-4 flex-shrink-0">
                    <button onclick="uiUpdateService.removeAlert('${alertId}')" class="inline-flex text-gray-400 hover:text-gray-600">
                        <svg class="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                        </svg>
                    </button>
                </div>
            </div>
        `;
        
        document.body.appendChild(alertDiv);
        
        // Animate in
        setTimeout(() => {
            alertDiv.classList.remove('translate-x-full');
        }, 100);
        
        // Auto remove
        const timeout = duration || this.config.alertTimeout;
        setTimeout(() => {
            this.removeAlert(alertId);
        }, timeout);
    }

    /**
     * Remove alert by ID
     */
    removeAlert(alertId) {
        const alert = document.getElementById(alertId);
        if (alert) {
            alert.classList.add('translate-x-full');
            setTimeout(() => {
                if (alert.parentElement) {
                    alert.remove();
                }
            }, 300);
        }
    }

    /**
     * Get CSS classes for alert type
     */
    getAlertClasses(type) {
        switch(type) {
            case 'success': return 'bg-green-50 border border-green-200 text-green-800';
            case 'error': return 'bg-red-50 border border-red-200 text-red-800';
            case 'warning': return 'bg-yellow-50 border border-yellow-200 text-yellow-800';
            case 'info': return 'bg-blue-50 border border-blue-200 text-blue-800';
            default: return 'bg-gray-50 border border-gray-200 text-gray-800';
        }
    }

    /**
     * Get icon for alert type
     */
    getAlertIcon(type) {
        switch(type) {
            case 'success': 
                return '<svg class="h-5 w-5 text-green-400" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
            case 'error': 
                return '<svg class="h-5 w-5 text-red-400" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
            case 'warning': 
                return '<svg class="h-5 w-5 text-yellow-400" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z"></path></svg>';
            case 'info': 
                return '<svg class="h-5 w-5 text-blue-400" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
            default: 
                return '<svg class="h-5 w-5 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path></svg>';
        }
    }

    /**
     * Cleanup - stop all intervals
     */
    cleanup() {
        Object.keys(this.updateIntervals).forEach(sectionId => {
            this.stopAutoRefresh(sectionId);
        });
    }
}

// Initialize global instance
const uiUpdateService = new UIUpdateService();

// Initialize on DOM load
document.addEventListener('DOMContentLoaded', function() {
    // Initialize with Laravel config if available
    if (typeof window.laravelConfig !== 'undefined' && window.laravelConfig.ui) {
        uiUpdateService.init(window.laravelConfig.ui);
    }
});

// Cleanup on page unload
window.addEventListener('beforeunload', function() {
    uiUpdateService.cleanup();
});