<?php

namespace Tests\Feature\Admin;

use Tests\TestCase;
use App\Models\Admin;
use App\Models\ApiConfig;
use Illuminate\Foundation\Testing\RefreshDatabase;

class ApiConfigManagementTest extends TestCase
{
    use RefreshDatabase;

    protected $admin;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create an admin user for testing using the factory
        $this->admin = Admin::factory()->create([
            'id' => 1,
            'name' => 'Test Admin',
            'email' => 'admin@test.com',
            'password' => bcrypt('password'),
            'is_active' => true,
        ]);
    }

    public function test_admin_can_access_api_config_index()
    {
        $response = $this->actingAs($this->admin, 'admin')
                         ->get(route('admin.api-config.index'));

        $response->assertStatus(200);
        $response->assertViewIs('admin.api-config.index');
    }

    public function test_admin_can_access_api_config_create_form()
    {
        $response = $this->actingAs($this->admin, 'admin')
                         ->get(route('admin.api-config.create'));

        $response->assertStatus(200);
        $response->assertViewIs('admin.api-config.create');
        $response->assertViewHas('categories');
        $response->assertViewHas('cardProviderTemplates');
    }

    public function test_admin_can_create_new_api_config()
    {
        $configData = [
            'name' => 'Test Provider',
            'slug' => 'test-provider',
            'category' => 'banking',
            'description' => 'Test provider for unit testing',
            'base_url' => 'https://api.testprovider.com',
            'test_url' => 'https://sandbox.testprovider.com',
            'live_url' => 'https://live.testprovider.com',
            'api_key' => 'test-api-key',
            'secret_key' => 'test-secret-key',
            'client_id' => 'test-client-id',
            'is_active' => true,
            'is_live_mode' => false,
            'config' => [
                'timeout' => 30,
                'retry_attempts' => 3,
            ],
        ];

        $response = $this->actingAs($this->admin, 'admin')
                         ->post(route('admin.api-config.store'), $configData);

        $response->assertRedirect(route('admin.api-config.index'));
        $response->assertSessionHas('success');

        // Verify the config was created
        $this->assertDatabaseHas('api_configs', [
            'name' => 'Test Provider',
            'slug' => 'test-provider',
            'category' => 'banking',
            'base_url' => 'https://api.testprovider.com',
        ]);
    }

    public function test_api_config_creation_validates_required_fields()
    {
        $response = $this->actingAs($this->admin, 'admin')
                         ->post(route('admin.api-config.store'), []);

        $response->assertSessionHasErrors(['name', 'slug', 'category', 'base_url']);
    }

    public function test_api_config_creation_validates_unique_slug()
    {
        // Create an existing config
        ApiConfig::create([
            'name' => 'Existing Provider',
            'slug' => 'existing-provider',
            'category' => 'payment',
            'base_url' => 'https://api.existing.com',
        ]);

        $configData = [
            'name' => 'New Provider',
            'slug' => 'existing-provider', // Same slug
            'category' => 'banking',
            'base_url' => 'https://api.newprovider.com',
        ];

        $response = $this->actingAs($this->admin, 'admin')
                         ->post(route('admin.api-config.store'), $configData);

        $response->assertSessionHasErrors(['slug']);
    }

    public function test_api_config_creation_validates_slug_format()
    {
        $configData = [
            'name' => 'Test Provider',
            'slug' => 'Invalid Slug Format!', // Invalid characters
            'category' => 'banking',
            'base_url' => 'https://api.testprovider.com',
        ];

        $response = $this->actingAs($this->admin, 'admin')
                         ->post(route('admin.api-config.store'), $configData);

        $response->assertSessionHasErrors(['slug']);
    }

    public function test_api_config_creation_encrypts_sensitive_fields()
    {
        $configData = [
            'name' => 'Test Provider',
            'slug' => 'test-provider',
            'category' => 'banking',
            'base_url' => 'https://api.testprovider.com',
            'api_key' => 'test-api-key',
            'secret_key' => 'test-secret-key',
            'client_secret' => 'test-client-secret',
        ];

        $this->actingAs($this->admin, 'admin')
             ->post(route('admin.api-config.store'), $configData);

        $config = ApiConfig::where('slug', 'test-provider')->first();

        // Verify sensitive fields are encrypted (they should be different from input)
        $this->assertNotEquals('test-api-key', $config->getRawOriginal('api_key'));
        $this->assertNotEquals('test-secret-key', $config->getRawOriginal('secret_key'));
        $this->assertNotEquals('test-client-secret', $config->getRawOriginal('client_secret'));

        // But should decrypt correctly when accessed through model
        $this->assertEquals('test-api-key', $config->api_key);
        $this->assertEquals('test-secret-key', $config->secret_key);
        $this->assertEquals('test-client-secret', $config->client_secret);
    }

    public function test_guest_cannot_access_api_config_management()
    {
        $response = $this->get(route('admin.api-config.index'));
        $response->assertRedirect(route('admin.login'));

        $response = $this->get(route('admin.api-config.create'));
        $response->assertRedirect(route('admin.login'));

        $response = $this->post(route('admin.api-config.store'), []);
        $response->assertRedirect(route('admin.login'));
    }
}