<?php

namespace Tests\Feature;

use App\Models\Admin;
use App\Models\Setting;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class SmtpConnectionIntegrationTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Create admin user for testing
        $this->admin = Admin::factory()->create([
            'email' => 'admin@test.com',
            'password' => bcrypt('password'),
        ]);
    }

    public function test_smtp_connection_endpoint_requires_authentication(): void
    {
        $response = $this->postJson('/admin/settings/notifications/smtp/test');
        
        // Should be redirected or return unauthorized
        $this->assertTrue(in_array($response->getStatusCode(), [302, 401, 419]));
    }

    public function test_smtp_connection_validates_missing_configuration(): void
    {
        $this->actingAs($this->admin, 'admin');
        
        // Clear any existing SMTP settings
        Setting::where('key', 'like', 'mail.smtp.%')->delete();
        
        $response = $this->postJson('/admin/settings/notifications/smtp/test');
        
        $response->assertStatus(400)
                ->assertJson([
                    'success' => false,
                    'message' => 'SMTP configuration is incomplete. Please fill all required fields.'
                ]);
    }

    public function test_smtp_connection_validates_empty_credentials(): void
    {
        $this->actingAs($this->admin, 'admin');
        
        // Set settings with empty credentials
        Setting::set('mail.smtp.host', 'smtp.example.com');
        Setting::set('mail.smtp.port', 587);
        Setting::set('mail.smtp.username', '');
        Setting::set('mail.smtp.password', '');
        
        $response = $this->postJson('/admin/settings/notifications/smtp/test');
        
        $response->assertStatus(400)
                ->assertJson([
                    'success' => false,
                    'message' => 'SMTP username and password are required.'
                ]);
    }

    public function test_smtp_connection_fails_with_invalid_server(): void
    {
        $this->actingAs($this->admin, 'admin');
        
        // Set invalid SMTP settings
        Setting::set('mail.smtp.host', 'invalid.smtp.server.that.does.not.exist');
        Setting::set('mail.smtp.port', 587);
        Setting::set('mail.smtp.username', 'test@example.com');
        Setting::set('mail.smtp.password', 'password123');
        Setting::set('mail.smtp.encryption', 'tls');
        
        $response = $this->postJson('/admin/settings/notifications/smtp/test');
        
        $response->assertStatus(500)
                ->assertJsonStructure([
                    'success',
                    'message'
                ])
                ->assertJson([
                    'success' => false
                ]);
                
        // Message should contain "SMTP connection failed"
        $responseData = $response->json();
        $this->assertStringContainsString('SMTP connection failed', $responseData['message']);
    }

    public function test_smtp_configuration_with_special_characters(): void
    {
        $this->actingAs($this->admin, 'admin');
        
        // Test with credentials containing special characters
        Setting::set('mail.smtp.host', 'smtp.gmail.com');
        Setting::set('mail.smtp.port', 587);
        Setting::set('mail.smtp.username', 'test+user@gmail.com');
        Setting::set('mail.smtp.password', 'pass@word#123');
        Setting::set('mail.smtp.encryption', 'tls');
        
        // This should try to connect (and fail because credentials are fake)
        // but the important thing is that it properly handles the special characters
        $response = $this->postJson('/admin/settings/notifications/smtp/test');
        
        // Should return error but not due to DSN construction issues
        $this->assertFalse($response->json('success'));
        $responseMessage = $response->json('message');
        
        // Should not contain URL encoding errors or DSN parsing errors
        $this->assertStringNotContainsString('DSN', $responseMessage);
        $this->assertStringNotContainsString('URL', $responseMessage);
        $this->assertStringNotContainsString('encoding', $responseMessage);
    }

    public function test_smtp_connection_uses_default_encryption(): void
    {
        $this->actingAs($this->admin, 'admin');
        
        // Set SMTP settings without encryption
        Setting::set('mail.smtp.host', 'smtp.test.com');
        Setting::set('mail.smtp.port', 587);
        Setting::set('mail.smtp.username', 'test@example.com');
        Setting::set('mail.smtp.password', 'password');
        // Don't set encryption - should default to 'tls'
        Setting::where('key', 'mail.smtp.encryption')->delete();
        
        $response = $this->postJson('/admin/settings/notifications/smtp/test');
        
        // Should fail connection but encryption default should work
        $this->assertFalse($response->json('success'));
        
        // Verify default encryption was used in settings
        $this->assertEquals('tls', Setting::get('mail.smtp.encryption', 'tls'));
    }
}
