<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\VirtualCard;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Schema;

class VirtualCardAnalyticsTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        parent::setUp();
        
        // Ensure we have the virtual_cards table
        $this->artisan('migrate');
    }

    /** @test */
    public function test_analytics_method_returns_proper_structure()
    {
        // Make a GET request to the analytics route
        $response = $this->get('/admin/virtual-cards/analytics');

        // Assert it returns a view
        $response->assertStatus(200);
        $response->assertViewIs('admin.virtual-cards.analytics'); // Adjust view name if needed

        // Get the data passed to the view
        $data = $response->viewData();

        // Assert analytics array exists and has expected structure
        $this->assertArrayHasKey('analytics', $data);
        $analytics = $data['analytics'];

        $this->assertArrayHasKey('cards_by_status', $analytics);
        $this->assertArrayHasKey('cards_by_type', $analytics);
        $this->assertArrayHasKey('monthly_creations', $analytics);
        $this->assertArrayHasKey('transaction_volume', $analytics);
        // All should be arrays (empty if no data)
        $this->assertIsArray($analytics['cards_by_status']);
        $this->assertIsArray($analytics['cards_by_type']);
        $this->assertIsArray($analytics['monthly_creations']);
        $this->assertIsArray($analytics['transaction_volume']);
    }

    /** @test */
    public function test_analytics_with_sample_data()
    {
        // Create a user first
        $user = User::factory()->create();
        
        // Create some test virtual cards
        VirtualCard::factory()->create([
            'user_id' => $user->id,
            'status' => 'active',
            'card_type' => 'virtual',
            'created_at' => now()->subDays(5)
        ]);
        
        VirtualCard::factory()->create([
            'user_id' => $user->id,
            'status' => 'frozen',
            'card_type' => 'virtual',
            'created_at' => now()->subDays(10)
        ]);

        $controller = new \App\Http\Controllers\Admin\VirtualCardController();
        $response = $controller->analytics();
        
        $data = $response->getData();
        $analytics = $data['analytics'];
        
        // Should have status data
        $this->assertNotEmpty($analytics['cards_by_status']);
        $this->assertEquals(1, $analytics['cards_by_status']['active']);
        $this->assertEquals(1, $analytics['cards_by_status']['frozen']);
        
        // Should have type data
        $this->assertNotEmpty($analytics['cards_by_type']);
        $this->assertEquals(2, $analytics['cards_by_type']['virtual']);
    }

    /** @test */
    public function test_month_format_expression_methods()
    {
        $controller = new \App\Http\Controllers\Admin\VirtualCardController();
        
        // Use reflection to test private methods
        $reflection = new \ReflectionClass($controller);
        
        $getMonthFormatMethod = $reflection->getMethod('getMonthFormatExpression');
        $getMonthFormatMethod->setAccessible(true);
        
        $getTransactionMonthFormatMethod = $reflection->getMethod('getTransactionMonthFormatExpression');
        $getTransactionMonthFormatMethod->setAccessible(true);
        
        // Test that methods return strings
        $monthFormat = $getMonthFormatMethod->invoke($controller);
        $transactionMonthFormat = $getTransactionMonthFormatMethod->invoke($controller);
        
        $this->assertIsString($monthFormat);
        $this->assertIsString($transactionMonthFormat);
        
        // Should contain expected patterns based on SQLite (default in tests)
        $this->assertStringContainsString('strftime', $monthFormat);
        $this->assertStringContainsString('strftime', $transactionMonthFormat);
        $this->assertStringContainsString('created_at', $monthFormat);
        $this->assertStringContainsString('transactions.created_at', $transactionMonthFormat);
    }

    /** @test */
    public function test_analytics_handles_missing_transaction_table_gracefully()
    {
        $controller = new \App\Http\Controllers\Admin\VirtualCardController();
        
        // Even without transactions table, should not crash
        $response = $controller->analytics();
        
        $this->assertInstanceOf(\Illuminate\View\View::class, $response);
        
        $data = $response->getData();
        $analytics = $data['analytics'];
        
        // Transaction volume should be empty array
        $this->assertIsArray($analytics['transaction_volume']);
        $this->assertEmpty($analytics['transaction_volume']);
    }
}