<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\VirtualCard;
use App\Models\VirtualAccount;
use App\Models\CronLog;
use Illuminate\Support\Facades\Mail;

class NotifyInactiveAccountsCommand extends Command
{
    protected $signature = 'fintech:notify-inactive-accounts';
    protected $description = 'Notify users if card or account is inactive';

    public function handle()
    {
        $startTime = microtime(true);
        $startMemory = memory_get_usage(true);
        
        try {
            $this->info('Starting inactive account notification process...');
            
            $notifiedUsers = 0;
            $errorCount = 0;
            
            // Check for inactive virtual cards (no activity in 30 days)
            $inactiveCards = VirtualCard::where('status', 'active')
                ->where('updated_at', '<', now()->subDays(30))
                ->whereHas('user', function($query) {
                    $query->whereNotNull('email_verified_at');
                })
                ->with('user')
                ->get();
            
            foreach ($inactiveCards as $card) {
                try {
                    $this->notifyInactiveCard($card);
                    $notifiedUsers++;
                } catch (\Exception $e) {
                    $this->error("Error notifying user {$card->user_id} about inactive card: " . $e->getMessage());
                    $errorCount++;
                }
            }
            
            // Check for users with no recent transactions (60 days)
            $inactiveUsers = User::whereDoesntHave('transactions', function($query) {
                    $query->where('created_at', '>=', now()->subDays(60));
                })
                ->whereNotNull('email_verified_at')
                ->where('created_at', '<', now()->subDays(7)) // At least 7 days old
                ->get();
            
            foreach ($inactiveUsers as $user) {
                try {
                    $this->notifyInactiveUser($user);
                    $notifiedUsers++;
                } catch (\Exception $e) {
                    $this->error("Error notifying inactive user {$user->id}: " . $e->getMessage());
                    $errorCount++;
                }
            }
            
            // Check for users with incomplete KYC (older than 14 days)
            $incompleteKycUsers = User::where('kyc_status', '!=', 'verified')
                ->whereNotNull('email_verified_at')
                ->where('created_at', '<', now()->subDays(14))
                ->where('created_at', '>=', now()->subDays(90)) // Not too old
                ->get();
            
            foreach ($incompleteKycUsers as $user) {
                try {
                    $this->notifyIncompleteKyc($user);
                    $notifiedUsers++;
                } catch (\Exception $e) {
                    $this->error("Error notifying user {$user->id} about incomplete KYC: " . $e->getMessage());
                    $errorCount++;
                }
            }
            
            $executionTime = microtime(true) - $startTime;
            $memoryUsage = memory_get_usage(true) - $startMemory;
            
            $message = "Notified {$notifiedUsers} users about inactive accounts/cards, {$errorCount} errors";
            $this->info($message);
            
            CronLog::create([
                'task_name' => 'notify_inactive_accounts',
                'status' => $errorCount > 0 ? 'warning' : 'success',
                'message' => $message,
                'executed_at' => now(),
                'execution_time' => $executionTime,
                'memory_usage' => $memoryUsage
            ]);
            
        } catch (\Exception $e) {
            $executionTime = microtime(true) - $startTime;
            $memoryUsage = memory_get_usage(true) - $startMemory;
            
            $this->error('Inactive account notification process failed: ' . $e->getMessage());
            
            CronLog::create([
                'task_name' => 'notify_inactive_accounts',
                'status' => 'failed',
                'message' => 'Inactive account notification process failed',
                'executed_at' => now(),
                'execution_time' => $executionTime,
                'memory_usage' => $memoryUsage,
                'error_details' => [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]
            ]);
        }
    }
    
    private function notifyInactiveCard(VirtualCard $card)
    {
        $user = $card->user;
        
        // Check if we've already notified about this card recently
        $recentNotification = $user->notifications()
            ->where('type', 'inactive_card_reminder')
            ->where('created_at', '>=', now()->subDays(7))
            ->whereJsonContains('data->card_id', $card->id)
            ->exists();
        
        if ($recentNotification) {
            return; // Don't spam notifications
        }
        
        // Create in-app notification
        $user->notifications()->create([
            'type' => 'inactive_card_reminder',
            'title' => 'Inactive Virtual Card',
            'message' => "Your virtual card ending in " . substr($card->card_number, -4) . " has been inactive for 30 days. Consider using it or freezing it for security.",
            'data' => [
                'card_id' => $card->id,
                'card_last_four' => substr($card->card_number, -4),
                'inactive_days' => now()->diffInDays($card->updated_at)
            ]
        ]);
        
        // Send email notification
        try {
            Mail::to($user->email)->send(new \App\Mail\InactiveCardNotification($user, $card));
        } catch (\Exception $e) {
            $this->warn("Failed to send email to {$user->email}: " . $e->getMessage());
        }
        
        $this->info("Notified user {$user->id} about inactive card {$card->id}");
    }
    
    private function notifyInactiveUser(User $user)
    {
        // Check if we've already notified this user recently
        $recentNotification = $user->notifications()
            ->where('type', 'inactive_account_reminder')
            ->where('created_at', '>=', now()->subDays(14))
            ->exists();
        
        if ($recentNotification) {
            return; // Don't spam notifications
        }
        
        $wallet = $user->wallet;
        $balance = $wallet ? $wallet->available_balance : 0;
        
        // Create in-app notification
        $user->notifications()->create([
            'type' => 'inactive_account_reminder',
            'title' => 'Account Inactive',
            'message' => "You haven't used your account in 60 days. Your wallet balance is ₦" . number_format($balance, 2) . ". Consider making a transaction to keep your account active.",
            'data' => [
                'inactive_days' => 60,
                'wallet_balance' => $balance,
                'suggested_actions' => ['Fund wallet', 'Buy airtime', 'Send money']
            ]
        ]);
        
        // Send email notification
        try {
            Mail::to($user->email)->send(new \App\Mail\InactiveAccountNotification($user));
        } catch (\Exception $e) {
            $this->warn("Failed to send email to {$user->email}: " . $e->getMessage());
        }
        
        $this->info("Notified inactive user {$user->id}");
    }
    
    private function notifyIncompleteKyc(User $user)
    {
        // Check if we've already notified about KYC recently
        $recentNotification = $user->notifications()
            ->where('type', 'incomplete_kyc_reminder')
            ->where('created_at', '>=', now()->subDays(7))
            ->exists();
        
        if ($recentNotification) {
            return; // Don't spam notifications
        }
        
        $kycSteps = [
            'BVN verification',
            'NIN verification',
            'Document upload',
            'Selfie verification'
        ];
        
        $completedSteps = [];
        if ($user->bvn_verified_at) $completedSteps[] = 'BVN verification';
        if ($user->nin_verified_at) $completedSteps[] = 'NIN verification';
        
        $remainingSteps = array_diff($kycSteps, $completedSteps);
        
        // Create in-app notification
        $user->notifications()->create([
            'type' => 'incomplete_kyc_reminder',
            'title' => 'Complete Your KYC',
            'message' => "Complete your identity verification to unlock all features including virtual cards and higher transaction limits. Remaining steps: " . implode(', ', $remainingSteps),
            'data' => [
                'completed_steps' => $completedSteps,
                'remaining_steps' => $remainingSteps,
                'kyc_status' => $user->kyc_status
            ]
        ]);
        
        // Send email notification
        try {
            Mail::to($user->email)->send(new \App\Mail\IncompleteKycNotification($user, $remainingSteps));
        } catch (\Exception $e) {
            $this->warn("Failed to send email to {$user->email}: " . $e->getMessage());
        }
        
        $this->info("Notified user {$user->id} about incomplete KYC");
    }
}