<?php

namespace App\Http\Requests\Admin;

use App\Services\FundingSourceService;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Config;

class AdjustWalletBalanceRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return auth('admin')->check();
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        return [
            'amount' => [
                'required',
                'numeric',
                'min:' . Config::get('fintech.wallet.min_adjustment_amount', 0.01),
                'max:' . Config::get('fintech.wallet.max_adjustment_amount', 1000000)
            ],
            'type' => 'required|in:credit,debit',
            'narration' => 'required|string|max:255',
            'funding_source' => [
                'nullable',
                'string',
                'in:' . implode(',', FundingSourceService::getEnabledSourceKeys())
            ],
        ];
    }

    /**
     * Get custom validation messages.
     */
    public function messages(): array
    {
        $minAmount = Config::get('fintech.wallet.min_adjustment_amount', 0.01);
        $maxAmount = Config::get('fintech.wallet.max_adjustment_amount', 1000000);
        
        return [
            'amount.required' => 'Amount is required.',
            'amount.numeric' => 'Amount must be a valid number.',
            'amount.min' => "Amount must be at least ₦{$minAmount}.",
            'amount.max' => "Amount cannot exceed ₦" . number_format($maxAmount),
            'type.required' => 'Transaction type is required.',
            'type.in' => 'Transaction type must be either credit or debit.',
            'narration.required' => 'Narration/reason is required.',
            'narration.max' => 'Narration cannot exceed 255 characters.',
            'funding_source.in' => 'Invalid or disabled funding source selected.',
        ];
    }

    /**
     * Get custom attribute names for validation errors.
     */
    public function attributes(): array
    {
        return [
            'funding_source' => 'funding source',
            'narration' => 'reason',
        ];
    }

    /**
     * Prepare the data for validation.
     */
    protected function prepareForValidation(): void
    {
        // Set default funding source if not provided
        if (!$this->has('funding_source') || empty($this->funding_source)) {
            $this->merge([
                'funding_source' => FundingSourceService::MANUAL
            ]);
        }
        
        // Sanitize amount
        if ($this->has('amount')) {
            $this->merge([
                'amount' => floatval($this->amount)
            ]);
        }
        
        // Sanitize narration
        if ($this->has('narration')) {
            $this->merge([
                'narration' => trim(strip_tags($this->narration))
            ]);
        }
    }

    /**
     * Get the validated funding source, ensuring it's valid and enabled.
     */
    public function getFundingSource(): string
    {
        $source = $this->validated('funding_source', FundingSourceService::MANUAL);
        
        // Double-check that the source is enabled
        if (!FundingSourceService::isEnabledSource($source)) {
            return FundingSourceService::MANUAL;
        }
        
        return $source;
    }

    /**
     * Check if the funding source requires approval.
     */
    public function requiresApproval(): bool
    {
        return FundingSourceService::requiresApproval($this->getFundingSource());
    }
}