<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AirtimeProvider extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'module',
        'api_credentials',
        'supported_services',
        'supported_networks',
        'is_active',
        'is_test_mode',
        'priority',
        'success_rate',
        'avg_response_time',
        'last_used_at',
        'last_test_result',
    ];

    protected function casts(): array
    {
        return [
            'api_credentials' => 'array',
            'supported_services' => 'array',
            'supported_networks' => 'array',
            'is_active' => 'boolean',
            'is_test_mode' => 'boolean',
            'success_rate' => 'decimal:2',
            'last_used_at' => 'datetime',
            'last_test_result' => 'array',
        ];
    }

    /**
     * Scope to get active providers.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope to get providers by service.
     */
    public function scopeByService($query, $service)
    {
        return $query->whereJsonContains('supported_services', $service);
    }

    /**
     * Scope to get providers by network.
     */
    public function scopeByNetwork($query, $network)
    {
        return $query->whereJsonContains('supported_networks', $network);
    }

    /**
     * Get providers ordered by priority.
     */
    public function scopeOrderedByPriority($query)
    {
        return $query->orderBy('priority', 'desc')->orderBy('success_rate', 'desc');
    }

    /**
     * Alias for orderedByPriority scope to match controller expectations.
     */
    public function scopeOrderByPriority($query)
    {
        return $query->orderedByPriority();
    }

    /**
     * Check if provider supports a service.
     */
    public function supportsService($service): bool
    {
        return in_array($service, $this->supported_services ?? []);
    }

    /**
     * Check if provider supports a network.
     */
    public function supportsNetwork($network): bool
    {
        return in_array($network, $this->supported_networks ?? []);
    }

    /**
     * Test provider connectivity.
     */
    public function testConnectivity(): array
    {
        $startTime = microtime(true);
        
        try {
            // Mock test for now - in real implementation, make actual API call
            if (!app()->environment('production')) {
                sleep(rand(1, 3)); // Simulate network delay
            }
            
            $endTime = microtime(true);
            $responseTime = round(($endTime - $startTime) * 1000);
            
            $result = [
                'success' => true,
                'response_time' => $responseTime,
                'tested_at' => now(),
                'message' => 'Connection successful',
            ];
            
            $this->update([
                'last_test_result' => $result,
                'avg_response_time' => $responseTime,
            ]);
            
            return $result;
            
        } catch (\Exception $e) {
            $result = [
                'success' => false,
                'response_time' => 0,
                'tested_at' => now(),
                'message' => $e->getMessage(),
            ];
            
            $this->update(['last_test_result' => $result]);
            
            return $result;
        }
    }
}
