<?php

namespace App\Services;

use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AuditLogService
{
    /**
     * Log an action for audit purposes.
     */
    public static function log(
        string $action,
        ?object $model = null,
        ?array $oldValues = null,
        ?array $newValues = null,
        ?array $metadata = null,
        ?Request $request = null
    ): AuditLog {
        $request = $request ?: request();
        
        return AuditLog::create([
            'user_id' => Auth::guard('web')->id(),
            'admin_id' => Auth::guard('admin')->id(),
            'action' => $action,
            'model_type' => $model ? get_class($model) : null,
            'model_id' => $model?->getKey(),
            'old_values' => $oldValues,
            'new_values' => $newValues,
            'metadata' => $metadata,
            'ip_address' => $request?->ip(),
            'user_agent' => $request?->userAgent(),
        ]);
    }

    /**
     * Log an admin action specifically.
     */
    public static function logAdminAction(
        string $action,
        ?object $model = null,
        ?array $oldValues = null,
        ?array $newValues = null,
        ?array $metadata = null,
        ?array $data = null,
        ?Request $request = null
    ): AuditLog {
        $request = $request ?: request();
        
        return AuditLog::create([
            'admin_id' => Auth::guard('admin')->id(),
            'action' => $action,
            'target_type' => $model ? get_class($model) : null,
            'target_id' => $model?->getKey(),
            'old_values' => $oldValues,
            'new_values' => $newValues,
            'metadata' => $metadata,
            'data' => $data,
            'ip_address' => $request?->ip(),
            'user_agent' => $request?->userAgent(),
        ]);
    }

    /**
     * Log a transaction-related action.
     */
    public static function logTransaction(
        string $action,
        object $transaction,
        ?array $metadata = null
    ): AuditLog {
        return self::log(
            action: $action,
            model: $transaction,
            newValues: $transaction->toArray(),
            metadata: array_merge([
                'transaction_amount' => $transaction->amount,
                'transaction_type' => $transaction->type,
                'transaction_status' => $transaction->status,
            ], $metadata ?? [])
        );
    }

    /**
     * Log a wallet operation.
     */
    public static function logWalletOperation(
        string $action,
        object $wallet,
        ?float $previousBalance = null,
        ?array $metadata = null
    ): AuditLog {
        return self::log(
            action: $action,
            model: $wallet,
            oldValues: $previousBalance ? ['balance' => $previousBalance] : null,
            newValues: ['balance' => $wallet->balance],
            metadata: array_merge([
                'wallet_currency' => $wallet->currency,
                'balance_change' => $previousBalance ? ($wallet->balance - $previousBalance) : null,
            ], $metadata ?? [])
        );
    }

    /**
     * Log a KYC action.
     */
    public static function logKycAction(
        string $action,
        object $kycVerification,
        ?array $metadata = null
    ): AuditLog {
        return self::log(
            action: $action,
            model: $kycVerification,
            newValues: $kycVerification->toArray(),
            metadata: array_merge([
                'kyc_type' => $kycVerification->type,
                'kyc_status' => $kycVerification->status,
            ], $metadata ?? [])
        );
    }
}