<?php

namespace App\Services\CardProviders;

use App\Models\ApiConfig;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WalletsAfricaService implements CardProviderInterface
{
    protected ?ApiConfig $config;

    public function __construct()
    {
        $this->config = ApiConfig::where('slug', 'wallets-africa')->first();
    }

    /**
     * Create a new virtual card for a user.
     */
    public function createCard(array $data): array
    {
        $this->ensureConfigured();

        $payload = [
            'type' => 'virtual',
            'brand' => strtoupper($data['brand'] ?? 'VISA'),
            'currency' => $data['currency'] ?? 'USD',
            'first_name' => $data['user']['first_name'],
            'last_name' => $data['user']['last_name'],
            'email' => $data['user']['email'],
            'phone' => $data['user']['phone'] ?? '',
            'amount' => $data['amount'] ?? 0,
            'daily_limit' => $data['daily_limit'],
            'monthly_limit' => $data['monthly_limit'],
        ];

        try {
            $response = $this->makeApiCall('POST', '/api/v1/cards/create', $payload);

            if ($response['success'] !== true) {
                throw new \Exception($response['message'] ?? 'Card creation failed');
            }

            $cardData = $response['data'];

            return [
                'success' => true,
                'data' => [
                    'card_id' => $cardData['card_id'],
                    'masked_pan' => $cardData['masked_pan'],
                    'card_type' => 'virtual',
                    'brand' => strtolower($cardData['brand']),
                    'currency' => $cardData['currency'],
                    'balance' => $cardData['balance'] ?? 0,
                    'status' => $cardData['status'] ?? 'active',
                    'expires_at' => $cardData['expiry_date'] ?? null,
                    'provider_reference' => $cardData['reference'],
                    'metadata' => $cardData,
                ],
            ];
        } catch (\Exception $e) {
            Log::error('Wallets Africa card creation failed', [
                'error' => $e->getMessage(),
                'payload' => $payload,
            ]);

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Get card details from provider.
     */
    public function getCard(string $cardId): array
    {
        $this->ensureConfigured();

        try {
            $response = $this->makeApiCall('GET', "/api/v1/cards/{$cardId}");

            if ($response['success'] !== true) {
                throw new \Exception($response['message'] ?? 'Failed to fetch card details');
            }

            return [
                'success' => true,
                'data' => $response['data'],
            ];
        } catch (\Exception $e) {
            Log::error('Wallets Africa get card failed', [
                'card_id' => $cardId,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Fund a virtual card.
     */
    public function fundCard(string $cardId, float $amount): array
    {
        $this->ensureConfigured();

        $payload = [
            'card_id' => $cardId,
            'amount' => $amount,
        ];

        try {
            $response = $this->makeApiCall('POST', '/api/v1/cards/fund', $payload);

            if ($response['success'] !== true) {
                throw new \Exception($response['message'] ?? 'Card funding failed');
            }

            return [
                'success' => true,
                'data' => $response['data'],
            ];
        } catch (\Exception $e) {
            Log::error('Wallets Africa card funding failed', [
                'card_id' => $cardId,
                'amount' => $amount,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Freeze a virtual card.
     */
    public function freezeCard(string $cardId): array
    {
        $this->ensureConfigured();

        $payload = ['card_id' => $cardId];

        try {
            $response = $this->makeApiCall('POST', '/api/v1/cards/freeze', $payload);

            return [
                'success' => $response['success'] ?? false,
                'message' => $response['message'] ?? 'Card frozen successfully',
            ];
        } catch (\Exception $e) {
            Log::error('Wallets Africa card freeze failed', [
                'card_id' => $cardId,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Unfreeze a virtual card.
     */
    public function unfreezeCard(string $cardId): array
    {
        $this->ensureConfigured();

        $payload = ['card_id' => $cardId];

        try {
            $response = $this->makeApiCall('POST', '/api/v1/cards/unfreeze', $payload);

            return [
                'success' => $response['success'] ?? false,
                'message' => $response['message'] ?? 'Card unfrozen successfully',
            ];
        } catch (\Exception $e) {
            Log::error('Wallets Africa card unfreeze failed', [
                'card_id' => $cardId,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Terminate a virtual card.
     */
    public function terminateCard(string $cardId): array
    {
        $this->ensureConfigured();

        $payload = ['card_id' => $cardId];

        try {
            $response = $this->makeApiCall('POST', '/api/v1/cards/terminate', $payload);

            return [
                'success' => $response['success'] ?? false,
                'message' => $response['message'] ?? 'Card terminated successfully',
            ];
        } catch (\Exception $e) {
            Log::error('Wallets Africa card termination failed', [
                'card_id' => $cardId,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Get card transaction history.
     */
    public function getTransactions(string $cardId, array $filters = []): array
    {
        $this->ensureConfigured();

        $queryParams = array_merge(['card_id' => $cardId], $filters);

        try {
            $response = $this->makeApiCall('GET', '/api/v1/cards/transactions', $queryParams);

            return [
                'success' => $response['success'] ?? false,
                'data' => $response['data'] ?? [],
            ];
        } catch (\Exception $e) {
            Log::error('Wallets Africa get transactions failed', [
                'card_id' => $cardId,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'message' => $e->getMessage(),
            ];
        }
    }

    /**
     * Check if the provider is properly configured.
     */
    public function isConfigured(): bool
    {
        return $this->config && 
               !empty($this->config->api_key) && 
               !empty($this->config->effective_base_url);
    }

    /**
     * Get provider name.
     */
    public function getProviderName(): string
    {
        return 'Wallets Africa';
    }

    /**
     * Get provider slug.
     */
    public function getProviderSlug(): string
    {
        return 'wallets-africa';
    }

    /**
     * Make API call to Wallets Africa.
     */
    protected function makeApiCall(string $method, string $endpoint, array $data = []): array
    {
        $url = $this->config->effective_base_url . $endpoint;
        
        $headers = [
            'Authorization' => 'Bearer ' . $this->config->api_key,
            'Content-Type' => 'application/json',
            'Accept' => 'application/json',
        ];

        $response = Http::withHeaders($headers)
            ->timeout(30)
            ->retry(2, 1000);

        if ($method === 'GET') {
            $response = $response->get($url, $data);
        } elseif ($method === 'POST') {
            $response = $response->post($url, $data);
        } elseif ($method === 'PUT') {
            $response = $response->put($url, $data);
        } elseif ($method === 'DELETE') {
            $response = $response->delete($url, $data);
        }

        if (!$response->successful()) {
            throw new \Exception(
                'API call failed: ' . $response->status() . ' - ' . $response->body()
            );
        }

        try {
            $json = $response->json();
        } catch (\Throwable $e) {
            throw new \Exception(
                'Failed to decode JSON response: ' . $e->getMessage() . ' | Response body: ' . $response->body()
            );
        }

        if ($json === null) {
            throw new \Exception(
                'API response is not valid JSON. Response body: ' . $response->body()
            );
        }

        return $json;
    }

    /**
     * Ensure the provider is configured.
     */
    protected function ensureConfigured(): void
    {
        if (!$this->config) {
            throw new \Exception('Wallets Africa API configuration not found');
        }
        
        if (!$this->isConfigured()) {
            throw new \Exception('Wallets Africa provider is not properly configured');
        }
    }
}