<?php

namespace App\Services;

use Illuminate\Support\Facades\Config;

/**
 * Export Parameter Validation Service
 * 
 * Handles validation and sanitization of export parameters to prevent
 * URL injection and ensure data integrity
 */
class ExportValidationService
{
    /**
     * Validate and sanitize export parameters
     * 
     * @param array $parameters
     * @return array
     */
    public static function validateParameters(array $parameters): array
    {
        $allowedFields = Config::get('fintech.export.allowed_filter_fields', []);
        $maxDateRange = Config::get('fintech.export.max_date_range', 365);
        $allowedFormats = Config::get('fintech.export.formats', ['csv']);
        
        $validated = [];
        $errors = [];
        
        foreach ($parameters as $key => $value) {
            // Skip empty values
            if (empty($value) || is_null($value)) {
                continue;
            }
            
            // Only allow whitelisted fields
            if (!in_array($key, $allowedFields) && $key !== 'export') {
                continue;
            }
            
            // Sanitize the value
            $sanitizedValue = static::sanitizeValue($value);
            
            if ($sanitizedValue !== '') {
                $validated[$key] = $sanitizedValue;
            }
        }
        
        // Validate date range if present
        if (isset($validated['date_from']) && isset($validated['date_to'])) {
            $dateValidation = static::validateDateRange(
                $validated['date_from'], 
                $validated['date_to'], 
                $maxDateRange
            );
            
            if (!$dateValidation['valid']) {
                $errors[] = $dateValidation['error'];
            }
        }
        
        // Validate export format
        if (isset($validated['export'])) {
            if (!in_array($validated['export'], $allowedFormats)) {
                $validated['export'] = 'csv'; // Default to CSV
            }
        }
        
        return [
            'parameters' => $validated,
            'errors' => $errors,
            'valid' => empty($errors)
        ];
    }
    
    /**
     * Sanitize a parameter value
     * 
     * @param mixed $value
     * @return string
     */
    private static function sanitizeValue($value): string
    {
        if (is_array($value)) {
            $value = implode(',', $value);
        }
        
        // Remove potentially dangerous characters
        $sanitized = preg_replace(self::DANGEROUS_CHARS_REGEX, '', (string) $value);
        
        // Trim whitespace
        $sanitized = trim($sanitized);
        
        // Remove null bytes
        $sanitized = str_replace("\0", '', $sanitized);
        
        return $sanitized;
    }
    
    /**
     * Validate date range
     * 
     * @param string $dateFrom
     * @param string $dateTo
     * @param int $maxDays
     * @return array
     */
    private static function validateDateRange(string $dateFrom, string $dateTo, int $maxDays): array
    {
        try {
            $fromDate = new \DateTime($dateFrom);
            $toDate = new \DateTime($dateTo);
            
            // Check if from date is before to date
            if ($fromDate > $toDate) {
                return [
                    'valid' => false,
                    'error' => 'Start date cannot be after end date.'
                ];
            }
            
            // Check date range limit
            $daysDiff = $fromDate->diff($toDate)->days;
            
            if ($daysDiff > $maxDays) {
                return [
                    'valid' => false,
                    'error' => "Date range cannot exceed {$maxDays} days."
                ];
            }
            
            // Check if dates are not in the future (for most use cases)
            $now = new \DateTime();
            if ($fromDate > $now || $toDate > $now) {
                return [
                    'valid' => false,
                    'error' => 'Dates cannot be in the future.'
                ];
            }
            
            return ['valid' => true];
            
        } catch (\Exception $e) {
            return [
                'valid' => false,
                'error' => 'Invalid date format provided.'
            ];
        }
    }
    
    /**
     * Generate secure export URL
     * 
     * @param string $baseUrl
     * @param array $parameters
     * @return string
     */
    public static function generateExportUrl(string $baseUrl, array $parameters): string
    {
        $validation = static::validateParameters($parameters);
        
        if (!$validation['valid']) {
            throw new \InvalidArgumentException('Invalid export parameters: ' . implode(', ', $validation['errors']));
        }
        
        $queryString = http_build_query($validation['parameters']);
        
        return $baseUrl . '?' . $queryString;
    }
    
    /**
     * Check if user has exceeded export rate limit
     * 
     * @param int $userId
     * @return bool
     */
    public static function checkRateLimit(int $userId): bool
    {
        $rateLimit = Config::get('fintech.export.rate_limit', 10);
        $cacheKey = "export_rate_limit:user:{$userId}";
        
        // Get current count from cache (or implement with your preferred storage)
        $currentCount = cache()->get($cacheKey, 0);
        
        if ($currentCount >= $rateLimit) {
            return false;
        }
        
        // Increment counter with 1-hour expiry
        cache()->put($cacheKey, $currentCount + 1, now()->addHour());
        
        return true;
    }
    
    /**
     * Get remaining export attempts for user
     * 
     * @param int $userId
     * @return int
     */
    public static function getRemainingAttempts(int $userId): int
    {
        $rateLimit = Config::get('fintech.export.rate_limit', 10);
        $cacheKey = "export_rate_limit:user:{$userId}";
        $currentCount = cache()->get($cacheKey, 0);
        
        return max(0, $rateLimit - $currentCount);
    }
}