<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\DB;
use Tests\TestCase;

class VirtualCardTransactionIndexTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Test that the virtual card transactions table has the correct short index name.
     */
    public function test_virtual_card_transactions_has_short_index_name(): void
    {
        // Check if we're using SQLite (our test environment)
        if (DB::connection()->getDriverName() === 'sqlite') {
            // Get all indexes for the vc_transactions table
            $indexes = DB::select("SELECT name FROM sqlite_master WHERE type='index' AND tbl_name='vc_transactions'");
            
            $indexNames = collect($indexes)->pluck('name')->toArray();
            
            // Assert that the short index name exists
            $this->assertContains('vc_mc_td_idx', $indexNames, 'The short index name vc_mc_td_idx should exist');
            
            // Assert that the long index name does NOT exist
            $this->assertNotContains('vc_transactions_merchant_category_transaction_date_index', $indexNames, 'The long index name should not exist');
        } else {
            // For MySQL/other databases, we can use a different approach
            $indexes = DB::select("SHOW INDEX FROM vc_transactions WHERE Key_name = 'vc_mc_td_idx'");
            $this->assertNotEmpty($indexes, 'The vc_mc_td_idx index should exist on the vc_transactions table');
        }
    }

    /**
     * Test that the index works for merchant category and transaction date queries.
     */
    public function test_merchant_category_transaction_date_index_functionality(): void
    {
        // Run migration to ensure table exists
        $this->artisan('migrate');
        // Create a test user and virtual card first
        $user = \App\Models\User::factory()->create();
        $virtualCard = \App\Models\VirtualCard::factory()->create(['user_id' => $user->id]);

        // Insert test data into vc_transactions table
        DB::table('vc_transactions')->insert([
            'user_id' => $user->id,
            'virtual_card_id' => $virtualCard->id,
            'provider_transaction_id' => 'test123',
            'amount' => 100.00,
            'currency' => 'USD',
            'merchant_category' => 'grocery',
            'transaction_type' => 'debit',
            'status' => 'approved',
            'transaction_date' => now(),
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // Test query that should use the index
        $result = DB::table('vc_transactions')
            ->where('merchant_category', 'grocery')
            ->where('transaction_date', '>=', now()->subDays(30))
            ->count();

        $this->assertEquals(1, $result, 'Query using merchant_category and transaction_date should work correctly');
    }

    /**
     * Test that the migration can be rolled back and re-run successfully.
     */
    public function test_migration_rollback_and_rerun(): void
    {
        // First, ensure we're at the latest migration state
        $this->artisan('migrate');
        
        // Check that table exists
        $this->assertTrue(
            DB::getSchemaBuilder()->hasTable('vc_transactions'),
            'vc_transactions table should exist after migration'
        );

        // Rollback the virtual card transactions migration
        // Find the migration file for vc_transactions
        $migrationFiles = glob(database_path('migrations/*_create_vc_transactions_table.php'));
        $this->assertNotEmpty($migrationFiles, 'vc_transactions migration file not found');
        $vcMigrationPath = $migrationFiles[0];
        $this->artisan('migrate:rollback', ['--path' => $vcMigrationPath]);
        
        // Re-run the migration
        $this->artisan('migrate');
        
        // Verify table still exists and index is correct
        $this->assertTrue(
            DB::getSchemaBuilder()->hasTable('vc_transactions'),
            'vc_transactions table should exist after rollback and re-migration'
        );

        // Check index again after rollback/re-run
        if (DB::connection()->getDriverName() === 'sqlite') {
            $indexes = DB::select("SELECT name FROM sqlite_master WHERE type='index' AND tbl_name='vc_transactions'");
            $indexNames = collect($indexes)->pluck('name')->toArray();
            $this->assertContains('vc_mc_td_idx', $indexNames, 'Short index should exist after rollback/re-migration');
        }
    }
}