<?php

namespace Tests\Unit;

use PHPUnit\Framework\TestCase;

class SmtpConnectionTest extends TestCase
{
    public function test_rawurlencode_is_used_for_credentials(): void
    {
        // Test that special characters in username/password are properly encoded
        $username = 'user@domain.com';
        $password = 'password@#$%';
        
        $encodedUsername = rawurlencode($username);
        $encodedPassword = rawurlencode($password);
        
        // Verify encoding works as expected
        $this->assertEquals('user%40domain.com', $encodedUsername);
        $this->assertEquals('password%40%23%24%25', $encodedPassword);
        
        // Test DSN construction format (what the method should create)
        $host = 'smtp.example.com';
        $port = 587;
        $encryption = 'tls';
        
        $expectedDsn = sprintf(
            'smtp://%s:%s@%s:%d?encryption=%s',
            $encodedUsername,
            $encodedPassword,
            $host,
            $port,
            $encryption
        );
        
        $this->assertEquals('smtp://user%40domain.com:password%40%23%24%25@smtp.example.com:587?encryption=tls', $expectedDsn);
    }

    public function test_smtp_dsn_construction_logic(): void
    {
        // Test the core DSN construction logic that the method uses
        $host = 'smtp.gmail.com';
        $port = 587;
        $username = 'test+user@gmail.com';
        $password = 'pass@word#123';
        $encryption = 'tls';

        // This is the format the testSmtpConnection method should construct
        $expectedFormat = sprintf(
            'smtp://%s:%s@%s:%d?encryption=%s',
            rawurlencode($username),
            rawurlencode($password),
            $host,
            $port,
            $encryption ?: 'tls'
        );

        $this->assertEquals(
            'smtp://test%2Buser%40gmail.com:pass%40word%23123@smtp.gmail.com:587?encryption=tls',
            $expectedFormat
        );
    }

    public function test_dsn_with_various_special_characters(): void
    {
        // Test with various special characters that need URL encoding
        $testCases = [
            ['user@example.com', 'simple123', 'user%40example.com', 'simple123'],
            ['user+test@example.com', 'pass@word', 'user%2Btest%40example.com', 'pass%40word'],
            ['user.name@domain.co.uk', 'p@ss#w0rd!', 'user.name%40domain.co.uk', 'p%40ss%23w0rd%21'],
            ['test_user@company.org', 'my$ecret&key', 'test_user%40company.org', 'my%24ecret%26key'],
        ];

        foreach ($testCases as [$username, $password, $expectedUser, $expectedPass]) {
            $this->assertEquals($expectedUser, rawurlencode($username), "Username encoding failed for: $username");
            $this->assertEquals($expectedPass, rawurlencode($password), "Password encoding failed for: $password");
        }
    }

    public function test_default_encryption_fallback(): void
    {
        // Test that 'tls' is used as default when encryption is null or empty
        $host = 'smtp.example.com';
        $port = 587;
        $username = 'user@example.com';
        $password = 'password';
        
        // Test with null encryption
        $encryptionNull = null;
        $dsnWithNull = sprintf(
            'smtp://%s:%s@%s:%d?encryption=%s',
            rawurlencode($username),
            rawurlencode($password),
            $host,
            $port,
            $encryptionNull ?: 'tls'
        );
        
        $this->assertStringContainsString('?encryption=tls', $dsnWithNull);
        
        // Test with empty encryption
        $encryptionEmpty = '';
        $dsnWithEmpty = sprintf(
            'smtp://%s:%s@%s:%d?encryption=%s',
            rawurlencode($username),
            rawurlencode($password),
            $host,
            $port,
            $encryptionEmpty ?: 'tls'
        );
        
        $this->assertStringContainsString('?encryption=tls', $dsnWithEmpty);
    }

    public function test_smtp_method_exists_in_controller(): void
    {
        // Verify the method exists in the controller class
        $this->assertTrue(
            method_exists(\App\Http\Controllers\Admin\SettingsController::class, 'testSmtpConnection'),
            'testSmtpConnection method should exist in SettingsController'
        );
        
        // Verify it's a public method
        $reflection = new \ReflectionMethod(\App\Http\Controllers\Admin\SettingsController::class, 'testSmtpConnection');
        $this->assertTrue($reflection->isPublic(), 'testSmtpConnection method should be public');
    }
}
