<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\KycVerification;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;

class KycController extends Controller
{
    /**
     * Display a listing of KYC verifications.
     */
    public function index(Request $request)
    {
        $query = KycVerification::with('user');

        // Status filter
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Type filter
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Date range filter
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Search by user
        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('user', function ($q) use ($search) {
                $q->where('first_name', 'LIKE', "%{$search}%")
                  ->orWhere('last_name', 'LIKE', "%{$search}%")
                  ->orWhere('email', 'LIKE', "%{$search}%");
            });
        }

        // Sorting
        $sortField = $request->get('sort', 'created_at');
        $sortDirection = $request->get('direction', 'desc');
        
        $allowedSorts = ['created_at', 'status', 'type'];
        if (in_array($sortField, $allowedSorts)) {
            $query->orderBy($sortField, $sortDirection);
        }

        $kycVerifications = $query->paginate(20)->withQueryString();

        // Statistics for the page
        $stats = [
            'total' => KycVerification::count(),
            'pending' => KycVerification::where('status', 'pending')->count(),
            'verified' => KycVerification::where('status', 'verified')->count(),
            'failed' => KycVerification::where('status', 'failed')->count(),
            'expired' => KycVerification::where('status', 'expired')->count(),
        ];

        return view('admin.kyc.index', compact('kycVerifications', 'stats'));
    }

    /**
     * Display the specified KYC verification.
     */
    public function show(KycVerification $kycVerification)
    {
        $kycVerification->load('user');
        
        return view('admin.kyc.show', compact('kycVerification'));
    }

    /**
     * Approve a KYC verification.
     */
    public function approve(Request $request, KycVerification $kycVerification)
    {
        if ($kycVerification->status !== 'pending') {
            return back()->withErrors(['error' => 'Only pending KYC verifications can be approved.']);
        }

        $kycVerification->update([
            'status' => 'verified',
            'verified_at' => now(),
            'failure_reason' => null,
        ]);

        // Update user verification status
        $user = $kycVerification->user;
        if (!$user->email_verified_at) {
            $user->update(['email_verified_at' => now()]);
        }

        // Log admin action
        $this->logAdminAction('kyc_approved', $kycVerification, $request->notes);

        // Send notification to user (placeholder - implement actual notification)
        // $this->sendKycStatusNotification($user, 'approved');

        return back()->with('success', 'KYC verification has been approved successfully.');
    }

    /**
     * Reject a KYC verification.
     */
    public function reject(Request $request, KycVerification $kycVerification)
    {
        $request->validate([
            'reason' => 'required|string|max:1000',
        ]);

        if ($kycVerification->status !== 'pending') {
            return back()->withErrors(['error' => 'Only pending KYC verifications can be rejected.']);
        }

        $kycVerification->update([
            'status' => 'failed',
            'failure_reason' => $request->reason,
        ]);

        // Log admin action
        $this->logAdminAction('kyc_rejected', $kycVerification, $request->reason);

        // Send notification to user (placeholder)
        // $this->sendKycStatusNotification($kycVerification->user, 'rejected', $request->reason);

        return back()->with('success', 'KYC verification has been rejected.');
    }

    /**
     * Reset a KYC verification to pending status.
     */
    public function reset(KycVerification $kycVerification)
    {
        $kycVerification->update([
            'status' => 'pending',
            'verified_at' => null,
            'failure_reason' => null,
        ]);

        // Log admin action
        $this->logAdminAction('kyc_reset', $kycVerification);

        return back()->with('success', 'KYC verification has been reset to pending status.');
    }

    /**
     * Download KYC document.
     */
    public function downloadDocument(KycVerification $kycVerification)
    {
        if (!$kycVerification->document_path) {
            return back()->withErrors(['error' => 'No document available for download.']);
        }

        if (!Storage::exists($kycVerification->document_path)) {
            return back()->withErrors(['error' => 'Document file not found.']);
        }

        // Log admin action
        $this->logAdminAction('kyc_document_downloaded', $kycVerification);

        return Storage::download($kycVerification->document_path, 
            "kyc-{$kycVerification->user->id}-{$kycVerification->type}.pdf");
    }

    /**
     * Get KYC analytics data.
     */
    public function analytics()
    {
        // Calculate basic analytics
        $totalSubmissions = KycVerification::count();
        $approvedCount = KycVerification::where('status', 'verified')->count();
        $pendingCount = KycVerification::where('status', 'pending')->count();
        $weeklyGrowth = $this->getWeeklyGrowth();
        $approvalRate = $totalSubmissions > 0 ? ($approvedCount / $totalSubmissions) * 100 : 0;
        $avgProcessingTime = $this->getAverageProcessingTime();
        $urgentCount = KycVerification::where('status', 'pending')
            ->where('created_at', '<=', now()->subHours(48))
            ->count();

        // Get type distribution
        $typeDistribution = [];
        $types = KycVerification::selectRaw('type, COUNT(*) as count')
            ->groupBy('type')
            ->get();
        
        $colors = [
            'bvn' => 'bg-blue-500',
            'nin' => 'bg-green-500',
            'passport' => 'bg-purple-500',
            'drivers_license' => 'bg-orange-500',
            'voters_card' => 'bg-red-500',
        ];

        foreach ($types as $type) {
            $percentage = $totalSubmissions > 0 ? ($type->count / $totalSubmissions) * 100 : 0;
            $typeDistribution[$type->type] = [
                'count' => $type->count,
                'percentage' => $percentage,
                'color' => $colors[$type->type] ?? 'bg-gray-500',
            ];
        }

        // Mock provider performance data
        $providerPerformance = [
            [
                'name' => 'Dojah',
                'total_requests' => 2500,
                'success_rate' => 95.2,
                'avg_response_time' => 1200,
                'status' => 'active',
            ],
            [
                'name' => 'VerifyMe',
                'total_requests' => 1800,
                'success_rate' => 88.7,
                'avg_response_time' => 2300,
                'status' => 'active',
            ],
            [
                'name' => 'Manual Review',
                'total_requests' => 450,
                'success_rate' => 98.1,
                'avg_response_time' => 28800,
                'status' => 'active',
            ],
        ];

        // Get recent activity
        $recentActivity = KycVerification::with('user')
            ->whereIn('status', ['verified', 'failed'])
            ->orderBy('updated_at', 'desc')
            ->limit(10)
            ->get()
            ->map(function ($kyc) {
                return [
                    'type' => $kyc->status === 'verified' ? 'approved' : 'rejected',
                    'description' => $kyc->status === 'verified' 
                        ? "Approved {$kyc->user->full_name}'s {$kyc->type} verification"
                        : "Rejected {$kyc->user->full_name}'s {$kyc->type} verification",
                    'admin' => 'Admin User', // Replace with actual admin name when available
                    'time' => $kyc->updated_at->diffForHumans(),
                    'url' => route('admin.kyc.show', $kyc),
                ];
            });

        $analytics = [
            'total_submissions' => $totalSubmissions,
            'weekly_growth' => $weeklyGrowth,
            'approval_rate' => $approvalRate,
            'approved_count' => $approvedCount,
            'avg_processing_time' => $avgProcessingTime,
            'pending_count' => $pendingCount,
            'urgent_count' => $urgentCount,
            'type_distribution' => $typeDistribution,
            'provider_performance' => $providerPerformance,
            'recent_activity' => $recentActivity,
        ];

        return view('admin.kyc.analytics', compact('analytics'));
    }

    /**
     * Get weekly growth percentage.
     */
    private function getWeeklyGrowth()
    {
        $thisWeek = KycVerification::where('created_at', '>=', now()->startOfWeek())->count();
        $lastWeek = KycVerification::whereBetween('created_at', [
            now()->subWeek()->startOfWeek(),
            now()->subWeek()->endOfWeek()
        ])->count();

        if ($lastWeek == 0) {
            return $thisWeek > 0 ? 100 : 0;
        }

        return round((($thisWeek - $lastWeek) / $lastWeek) * 100, 1);
    }

    /**
     * Bulk approve KYC verifications.
     */
    public function bulkApprove(Request $request)
    {
        $request->validate([
            'kyc_ids' => 'required|array',
            'kyc_ids.*' => 'exists:kyc_verifications,id',
        ]);

        $kycVerifications = KycVerification::whereIn('id', $request->kyc_ids)
            ->where('status', 'pending')
            ->get();

        $approvedCount = 0;
        foreach ($kycVerifications as $kyc) {
            $kyc->update([
                'status' => 'verified',
                'verified_at' => now(),
            ]);
            
            // Update user verification
            if (!$kyc->user->email_verified_at) {
                $kyc->user->update(['email_verified_at' => now()]);
            }

            $this->logAdminAction('kyc_bulk_approved', $kyc);
            $approvedCount++;
        }

        return back()->with('success', "{$approvedCount} KYC verification(s) have been approved.");
    }

    /**
     * Bulk reject KYC verifications.
     */
    public function bulkReject(Request $request)
    {
        $request->validate([
            'kyc_ids' => 'required|array',
            'kyc_ids.*' => 'exists:kyc_verifications,id',
            'reason' => 'required|string|max:1000',
        ]);

        $kycVerifications = KycVerification::whereIn('id', $request->kyc_ids)
            ->where('status', 'pending')
            ->get();

        $rejectedCount = 0;
        foreach ($kycVerifications as $kyc) {
            $kyc->update([
                'status' => 'failed',
                'failure_reason' => $request->reason,
            ]);

            $this->logAdminAction('kyc_bulk_rejected', $kyc, $request->reason);
            $rejectedCount++;
        }

        return back()->with('success', "{$rejectedCount} KYC verification(s) have been rejected.");
    }

    /**
     * Log admin actions for audit trail.
     */
    private function logAdminAction(string $action, KycVerification $kyc, ?string $notes = null)
    {
        \App\Models\AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => $action,
            'target_type' => 'kyc_verification',
            'target_id' => $kyc->id,
            'data' => [
                'user_id' => $kyc->user_id,
                'kyc_type' => $kyc->type,
                'previous_status' => $kyc->getOriginal('status'),
                'new_status' => $kyc->status,
                'notes' => $notes,
                'ip_address' => request()->ip(),
                'user_agent' => request()->userAgent(),
            ],
        ]);
    }

    /**
     * Get average KYC processing time in hours.
     */
    private function getAverageProcessingTime()
    {
        $processed = KycVerification::whereIn('status', ['verified', 'failed'])
            ->whereNotNull('verified_at')
            ->orWhereNotNull('updated_at')
            ->get();

        if ($processed->isEmpty()) {
            return 0;
        }

        $totalHours = $processed->sum(function ($kyc) {
            $endTime = $kyc->verified_at ?: $kyc->updated_at;
            return $kyc->created_at->diffInHours($endTime);
        });

        return round($totalHours / $processed->count(), 1);
    }

    /**
     * Get monthly KYC submission trends.
     */
    private function getMonthlyTrends()
    {
        return KycVerification::selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month, COUNT(*) as count')
            ->where('created_at', '>=', now()->subMonths(12))
            ->groupBy('month')
            ->orderBy('month')
            ->get()
            ->pluck('count', 'month');
    }

    /**
     * Get processing performance metrics.
     */
    private function getProcessingPerformance()
    {
        $last30Days = now()->subDays(30);
        
        return [
            'processed_last_30_days' => KycVerification::whereIn('status', ['verified', 'failed'])
                ->where('updated_at', '>=', $last30Days)
                ->count(),
            'pending_over_24h' => KycVerification::where('status', 'pending')
                ->where('created_at', '<=', now()->subHours(24))
                ->count(),
            'pending_over_7_days' => KycVerification::where('status', 'pending')
                ->where('created_at', '<=', now()->subDays(7))
                ->count(),
        ];
    }
}