<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class VTPassService
{
    protected $username;
    protected $password;
    protected $baseUrl;

    public function __construct()
    {
        $this->username = config('services.vtpass.username');
        $this->password = config('services.vtpass.password');
        $this->baseUrl = config('services.vtpass.base_url');
    }

    /**
     * Purchase airtime.
     */
    public function purchaseAirtime(array $data): array
    {
        try {
            $response = $this->makeApiCall('/pay', [
                'serviceID' => $this->getAirtimeServiceId($data['network']),
                'amount' => $data['amount'],
                'phone' => $data['phone'],
                'request_id' => $data['request_id'],
            ]);

            if ($response['code'] === '000') {
                return [
                    'success' => true,
                    'transaction_id' => $response['content']['transactions']['transactionId'],
                    'status' => $response['content']['transactions']['status'],
                    'reference' => $response['requestId'],
                ];
            }

            return [
                'success' => false,
                'message' => $response['response_description'] ?? 'Airtime purchase failed',
            ];
        } catch (\Exception $e) {
            Log::error('VTPass airtime purchase failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Airtime purchase failed. Please try again.',
            ];
        }
    }

    /**
     * Purchase data bundle.
     */
    public function purchaseData(array $data): array
    {
        try {
            $response = $this->makeApiCall('/pay', [
                'serviceID' => $this->getDataServiceId($data['network']),
                'billersCode' => $data['phone'],
                'variation_code' => $data['plan_code'],
                'amount' => $data['amount'],
                'phone' => $data['phone'],
                'request_id' => $data['request_id'],
            ]);

            if ($response['code'] === '000') {
                return [
                    'success' => true,
                    'transaction_id' => $response['content']['transactions']['transactionId'],
                    'status' => $response['content']['transactions']['status'],
                    'reference' => $response['requestId'],
                ];
            }

            return [
                'success' => false,
                'message' => $response['response_description'] ?? 'Data purchase failed',
            ];
        } catch (\Exception $e) {
            Log::error('VTPass data purchase failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Data purchase failed. Please try again.',
            ];
        }
    }

    /**
     * Pay electricity bill.
     */
    public function payElectricity(array $data): array
    {
        try {
            $response = $this->makeApiCall('/pay', [
                'serviceID' => $data['service_id'], // e.g., 'eko-electric', 'ikeja-electric'
                'billersCode' => $data['meter_number'],
                'variation_code' => $data['meter_type'], // prepaid or postpaid
                'amount' => $data['amount'],
                'phone' => $data['phone'],
                'request_id' => $data['request_id'],
            ]);

            if ($response['code'] === '000') {
                return [
                    'success' => true,
                    'transaction_id' => $response['content']['transactions']['transactionId'],
                    'status' => $response['content']['transactions']['status'],
                    'reference' => $response['requestId'],
                    'token' => $response['content']['transactions']['Token'] ?? null,
                ];
            }

            return [
                'success' => false,
                'message' => $response['response_description'] ?? 'Electricity payment failed',
            ];
        } catch (\Exception $e) {
            Log::error('VTPass electricity payment failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Electricity payment failed. Please try again.',
            ];
        }
    }

    /**
     * Pay cable TV subscription.
     */
    public function payCableTV(array $data): array
    {
        try {
            $response = $this->makeApiCall('/pay', [
                'serviceID' => $data['service_id'], // e.g., 'dstv', 'gotv', 'startimes'
                'billersCode' => $data['decoder_number'],
                'variation_code' => $data['package_code'],
                'amount' => $data['amount'],
                'phone' => $data['phone'],
                'request_id' => $data['request_id'],
            ]);

            if ($response['code'] === '000') {
                return [
                    'success' => true,
                    'transaction_id' => $response['content']['transactions']['transactionId'],
                    'status' => $response['content']['transactions']['status'],
                    'reference' => $response['requestId'],
                ];
            }

            return [
                'success' => false,
                'message' => $response['response_description'] ?? 'Cable TV payment failed',
            ];
        } catch (\Exception $e) {
            Log::error('VTPass cable TV payment failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Cable TV payment failed. Please try again.',
            ];
        }
    }

    /**
     * Verify customer details.
     */
    public function verifyCustomer(string $serviceId, string $billersCode): array
    {
        try {
            $response = $this->makeApiCall('/merchant-verify', [
                'serviceID' => $serviceId,
                'billersCode' => $billersCode,
            ]);

            if ($response['code'] === '000') {
                return [
                    'success' => true,
                    'customer_name' => $response['content']['Customer_Name'],
                    'customer_id' => $response['content']['Customer_ID'] ?? null,
                    'address' => $response['content']['Address'] ?? null,
                ];
            }

            return [
                'success' => false,
                'message' => $response['response_description'] ?? 'Customer verification failed',
            ];
        } catch (\Exception $e) {
            Log::error('VTPass customer verification failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Customer verification failed.',
            ];
        }
    }

    /**
     * Get service variations (plans/packages).
     */
    public function getServiceVariations(string $serviceId): array
    {
        try {
            $response = $this->makeApiCall("/service-variations?serviceID={$serviceId}", [], 'GET');

            if ($response['code'] === '000') {
                return [
                    'success' => true,
                    'variations' => $response['content']['varations'], // Note: VTPass API has typo
                ];
            }

            return [
                'success' => false,
                'message' => 'Could not fetch service variations',
            ];
        } catch (\Exception $e) {
            Log::error('VTPass get service variations failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Could not fetch service variations.',
            ];
        }
    }

    /**
     * Get transaction status.
     */
    public function getTransactionStatus(string $requestId): array
    {
        try {
            $response = $this->makeApiCall("/requery", [
                'request_id' => $requestId,
            ]);

            if ($response['code'] === '000') {
                return [
                    'success' => true,
                    'status' => $response['content']['transactions']['status'],
                    'transaction_id' => $response['content']['transactions']['transactionId'],
                    'amount' => $response['content']['transactions']['amount'],
                ];
            }

            return [
                'success' => false,
                'message' => 'Transaction not found',
            ];
        } catch (\Exception $e) {
            Log::error('VTPass transaction status check failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Could not check transaction status.',
            ];
        }
    }

    /**
     * Get airtime service ID by network.
     */
    protected function getAirtimeServiceId(string $network): string
    {
        return match (strtolower($network)) {
            'mtn' => 'mtn',
            'glo' => 'glo',
            'airtel' => 'airtel',
            '9mobile' => '9mobile',
            default => 'mtn'
        };
    }

    /**
     * Get data service ID by network.
     */
    protected function getDataServiceId(string $network): string
    {
        return match (strtolower($network)) {
            'mtn' => 'mtn-data',
            'glo' => 'glo-data',
            'airtel' => 'airtel-data',
            '9mobile' => '9mobile-data',
            default => 'mtn-data'
        };
    }

    /**
     * Make API call to VTPass.
     */
    protected function makeApiCall(string $endpoint, array $data = [], string $method = 'POST'): array
    {
        if (config('app.env') === 'local') {
            // Return mock response in local environment
            Log::info("VTPass API Call (MOCK): {$method} {$endpoint}", $data);
            
            return [
                'code' => '000',
                'response_description' => 'TRANSACTION SUCCESSFUL',
                'requestId' => $data['request_id'] ?? 'mock_request_id',
                'content' => [
                    'transactions' => [
                        'transactionId' => 'VTP' . uniqid(),
                        'status' => 'delivered',
                        'amount' => $data['amount'] ?? 1000,
                        'Token' => $method === 'POST' && isset($data['serviceID']) && str_contains($data['serviceID'], 'electric') 
                            ? '1234-5678-9012-3456-7890' : null,
                    ]
                ]
            ];
        }

        $http = Http::withBasicAuth($this->username, $this->password)
            ->withHeaders(['Content-Type' => 'application/json']);

        $response = $method === 'GET' 
            ? $http->get($this->baseUrl . $endpoint)
            : $http->post($this->baseUrl . $endpoint, $data);

        if ($response->successful()) {
            return $response->json();
        }

        throw new \Exception('VTPass API call failed: ' . $response->body());
    }
}