<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use App\Models\AuditLog;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class SettingsController extends Controller
{
    /**
     * Display fees and limits settings.
     */
    public function fees()
    {
        // Mock settings data - in a real app, this would come from a settings table
        $settings = [
            'fees' => [
                'transfer_fee_type' => 'percentage', // percentage or flat
                'transfer_fee_value' => 1.5,
                'transfer_flat_fee' => 10.00,
                'withdrawal_fee_type' => 'flat',
                'withdrawal_fee_value' => 25.00,
                'card_issuance_fee' => 500.00,
                'card_maintenance_fee' => 50.00,
                'virtual_account_fee' => 0.00,
            ],
            'limits' => [
                'daily_transfer_limit' => 500000,
                'single_transfer_limit' => 100000,
                'daily_withdrawal_limit' => 200000,
                'single_withdrawal_limit' => 50000,
                'daily_airtime_limit' => 10000,
                'single_airtime_limit' => 5000,
                'card_daily_limit' => 100000,
                'card_monthly_limit' => 1000000,
                'minimum_wallet_balance' => 100,
            ],
            'requirements' => [
                'kyc_required_for_transfers' => true,
                'kyc_required_for_cards' => true,
                'kyc_required_amount_threshold' => 50000,
                'require_2fa_for_large_transfers' => true,
                'large_transfer_threshold' => 100000,
            ]
        ];

        return view('admin.settings.fees', compact('settings'));
    }

    /**
     * Display site settings.
     */
    public function site()
    {
        $settings = [
            'business' => [
                'name' => config('app.name', 'SwiftPay'),
                'tagline' => 'Your trusted fintech partner',
                'description' => 'Leading digital financial services platform in Nigeria',
                'logo' => '/images/logo.png',
                'favicon' => '/images/favicon.png',
                'support_email' => 'support@swiftpay.ng',
                'support_phone' => '+234 800 000 0000',
                'address' => '123 Business District, Lagos, Nigeria',
            ],
            'features' => [
                'enable_wallet_funding' => true,
                'enable_p2p_transfers' => true,
                'enable_bank_transfers' => true,
                'enable_bill_payments' => true,
                'enable_virtual_cards' => true,
                'enable_airtime_data' => true,
                'enable_user_registration' => true,
                'maintenance_mode' => false,
            ],
            'integrations' => [
                'sms_provider' => 'termii',
                'email_provider' => 'mailgun',
                'payment_gateway' => 'paystack',
                'identity_provider' => 'dojah',
                'card_provider' => 'anchor',
            ],
            'security' => [
                'session_lifetime' => 120,
                'password_expiry_days' => 90,
                'max_login_attempts' => 5,
                'lockout_duration' => 30,
                'require_email_verification' => true,
                'require_phone_verification' => true,
            ]
        ];

        return view('admin.settings.site', compact('settings'));
    }

    /**
     * Display notification settings.
     */
    public function notifications()
    {
        $settings = [
            'email_notifications' => [
                'user_registration' => true,
                'kyc_approval' => true,
                'kyc_rejection' => true,
                'transaction_success' => true,
                'transaction_failure' => true,
                'card_issuance' => true,
                'suspicious_activity' => true,
                'password_reset' => true,
                'low_balance_warning' => true,
            ],
            'sms_notifications' => [
                'transaction_alerts' => true,
                'otp_verification' => true,
                'security_alerts' => true,
                'balance_alerts' => false,
                'promotional_messages' => false,
            ],
            'push_notifications' => [
                'transaction_updates' => true,
                'system_maintenance' => true,
                'new_features' => false,
                'promotional_offers' => false,
            ],
            'admin_notifications' => [
                'new_user_registration' => true,
                'kyc_submissions' => true,
                'high_value_transactions' => true,
                'failed_transactions' => true,
                'system_errors' => true,
                'security_incidents' => true,
            ]
        ];

        // Recent broadcast messages
        $broadcasts = [
            [
                'id' => 1,
                'title' => 'System Maintenance Notice',
                'message' => 'Scheduled maintenance on Sunday 2AM-4AM',
                'type' => 'maintenance',
                'sent_at' => '2024-01-15 14:30:00',
                'recipients' => 'all_users',
                'status' => 'sent'
            ],
            [
                'id' => 2,
                'title' => 'New Virtual Card Feature',
                'message' => 'Experience our new virtual card service',
                'type' => 'feature',
                'sent_at' => '2024-01-10 10:00:00',
                'recipients' => 'verified_users',
                'status' => 'sent'
            ]
        ];

        return view('admin.settings.notifications', compact('settings', 'broadcasts'));
    }

    /**
     * Display security logs.
     */
    public function securityLogs(Request $request)
    {
        $query = AuditLog::with('admin');

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('action', 'like', "%{$search}%")
                  ->orWhere('target_type', 'like', "%{$search}%")
                  ->orWhere('ip_address', 'like', "%{$search}%");
            });
        }

        if ($request->filled('action')) {
            $query->where('action', $request->action);
        }

        if ($request->filled('admin_id')) {
            $query->where('admin_id', $request->admin_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        $logs = $query->latest()->paginate(20)->withQueryString();

        // Statistics
        $stats = [
            'total_logs' => AuditLog::count(),
            'today_actions' => AuditLog::whereDate('created_at', today())->count(),
            'unique_admins' => AuditLog::distinct('admin_id')->count(),
            'login_attempts' => AuditLog::where('action', 'like', '%login%')->count(),
        ];

        // Common actions for filter
        $actions = AuditLog::distinct('action')->pluck('action')->sort();

        return view('admin.settings.security-logs', compact('logs', 'stats', 'actions'));
    }

    /**
     * Display pages management.
     */
    public function pages()
    {
        // Mock pages data - in a real app, this would come from a pages table
        $pages = [
            [
                'id' => 1,
                'title' => 'Terms & Conditions',
                'slug' => 'terms-conditions',
                'content' => 'Terms and conditions content here...',
                'status' => 'published',
                'updated_at' => '2024-01-15 10:30:00',
            ],
            [
                'id' => 2,
                'title' => 'Privacy Policy',
                'slug' => 'privacy-policy',
                'content' => 'Privacy policy content here...',
                'status' => 'published',
                'updated_at' => '2024-01-12 15:45:00',
            ],
            [
                'id' => 3,
                'title' => 'About Us',
                'slug' => 'about-us',
                'content' => 'About us content here...',
                'status' => 'published',
                'updated_at' => '2024-01-10 09:20:00',
            ],
            [
                'id' => 4,
                'title' => 'Frequently Asked Questions',
                'slug' => 'faqs',
                'content' => 'FAQ content here...',
                'status' => 'published',
                'updated_at' => '2024-01-08 16:10:00',
            ],
        ];

        return view('admin.settings.pages', compact('pages'));
    }

    /**
     * Update fees and limits.
     */
    public function updateFees(Request $request)
    {
        $request->validate([
            'transfer_fee_type' => 'required|in:percentage,flat',
            'transfer_fee_value' => 'required|numeric|min:0|max:100',
            'withdrawal_fee_value' => 'required|numeric|min:0',
            'daily_transfer_limit' => 'required|numeric|min:1000',
            // Add more validation rules as needed
        ]);

        // In a real app, you would save these to a settings table
        // For now, we'll just simulate the update

        // Log admin action
        AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'update_fees_settings',
            'target_type' => 'Settings',
            'target_id' => null,
            'data' => $request->all(),
            'ip_address' => request()->ip(),
        ]);

        return back()->with('success', 'Fees and limits updated successfully.');
    }

    /**
     * Send broadcast notification.
     */
    public function sendBroadcast(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'message' => 'required|string|max:1000',
            'type' => 'required|in:info,warning,success,error',
            'recipients' => 'required|in:all_users,verified_users,unverified_users,active_users',
        ]);

        // In a real app, you would queue this notification for sending
        // For now, we'll just simulate the broadcast

        // Log admin action
        AuditLog::create([
            'admin_id' => auth('admin')->id(),
            'action' => 'send_broadcast_notification',
            'target_type' => 'Notification',
            'target_id' => null,
            'data' => [
                'title' => $request->title,
                'recipients' => $request->recipients,
                'type' => $request->type,
            ],
            'ip_address' => request()->ip(),
        ]);

        return back()->with('success', 'Broadcast notification sent successfully.');
    }
}