<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\Transaction;

// TODO: Add comprehensive error handling and retry mechanisms
// TODO: Implement webhook signature verification
// TODO: Add rate limiting for API calls
// TODO: Add caching for frequently accessed data
// TODO: Implement idempotency for payment operations

class PaystackService
{
    protected $secretKey;
    protected $publicKey;
    protected $baseUrl;

    public function __construct()
    {
        $this->secretKey = config('services.paystack.secret_key');
        $this->publicKey = config('services.paystack.public_key');
        $this->baseUrl = config('services.paystack.base_url', 'https://api.paystack.co');
        
        // TODO: Add validation to ensure API keys are configured
        // TODO: Add environment-specific configurations (sandbox vs live)
    }

    /**
     * Initialize payment transaction.
     * 
     * TODO: Add input validation and sanitization
     * TODO: Add idempotency key support
     * TODO: Add webhook URL configuration
     * TODO: Add custom fields and metadata handling
     */
    public function initializePayment(array $data): array
    {
        try {
            $response = $this->makeApiCall('/transaction/initialize', [
                'amount' => $data['amount'] * 100, // Convert to kobo
                'email' => $data['email'],
                'reference' => $data['reference'],
                'currency' => $data['currency'] ?? 'NGN',
                'callback_url' => $data['callback_url'] ?? null,
                'metadata' => $data['metadata'] ?? [],
            ]);

            if ($response['status']) {
                return [
                    'success' => true,
                    'authorization_url' => $response['data']['authorization_url'],
                    'access_code' => $response['data']['access_code'],
                    'reference' => $response['data']['reference'],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Payment initialization failed',
            ];
        } catch (\Exception $e) {
            Log::error('Paystack payment initialization failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Payment initialization failed. Please try again.',
            ];
        }
    }

    /**
     * Verify payment transaction.
     */
    public function verifyPayment(string $reference): array
    {
        try {
            $response = $this->makeApiCall("/transaction/verify/{$reference}", [], 'GET');

            if ($response['status'] && $response['data']['status'] === 'success') {
                return [
                    'success' => true,
                    'status' => 'success',
                    'amount' => $response['data']['amount'] / 100, // Convert from kobo
                    'currency' => $response['data']['currency'],
                    'reference' => $response['data']['reference'],
                    'customer' => $response['data']['customer'],
                    'authorization' => $response['data']['authorization'],
                    'fees' => $response['data']['fees'] / 100,
                    'paid_at' => $response['data']['paid_at'],
                ];
            }

            return [
                'success' => false,
                'status' => $response['data']['status'] ?? 'failed',
                'message' => $response['message'] ?? 'Payment verification failed',
            ];
        } catch (\Exception $e) {
            Log::error('Paystack payment verification failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Payment verification failed.',
            ];
        }
    }

    /**
     * Create transfer recipient.
     */
    public function createTransferRecipient(array $data): array
    {
        try {
            $response = $this->makeApiCall('/transferrecipient', [
                'type' => 'nuban',
                'name' => $data['name'],
                'account_number' => $data['account_number'],
                'bank_code' => $data['bank_code'],
                'currency' => $data['currency'] ?? 'NGN',
            ]);

            if ($response['status']) {
                return [
                    'success' => true,
                    'recipient_code' => $response['data']['recipient_code'],
                    'details' => $response['data'],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Recipient creation failed',
            ];
        } catch (\Exception $e) {
            Log::error('Paystack recipient creation failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Recipient creation failed.',
            ];
        }
    }

    /**
     * Initiate transfer.
     */
    public function initiateTransfer(array $data): array
    {
        try {
            $response = $this->makeApiCall('/transfer', [
                'source' => 'balance',
                'amount' => $data['amount'] * 100, // Convert to kobo
                'recipient' => $data['recipient_code'],
                'reason' => $data['reason'] ?? 'Transfer',
                'reference' => $data['reference'],
            ]);

            if ($response['status']) {
                return [
                    'success' => true,
                    'transfer_code' => $response['data']['transfer_code'],
                    'reference' => $response['data']['reference'],
                    'status' => $response['data']['status'],
                ];
            }

            return [
                'success' => false,
                'message' => $response['message'] ?? 'Transfer initiation failed',
            ];
        } catch (\Exception $e) {
            Log::error('Paystack transfer initiation failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Transfer initiation failed.',
            ];
        }
    }

    /**
     * Get list of banks.
     */
    public function getBanks(): array
    {
        try {
            $response = $this->makeApiCall('/bank', [], 'GET');

            if ($response['status']) {
                return [
                    'success' => true,
                    'banks' => $response['data'],
                ];
            }

            return [
                'success' => false,
                'message' => 'Could not fetch banks',
            ];
        } catch (\Exception $e) {
            Log::error('Paystack get banks failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Could not fetch banks.',
            ];
        }
    }

    /**
     * Resolve account number.
     */
    public function resolveAccount(string $accountNumber, string $bankCode): array
    {
        try {
            $response = $this->makeApiCall("/bank/resolve?account_number={$accountNumber}&bank_code={$bankCode}", [], 'GET');

            if ($response['status']) {
                return [
                    'success' => true,
                    'account_name' => $response['data']['account_name'],
                    'account_number' => $response['data']['account_number'],
                ];
            }

            return [
                'success' => false,
                'message' => 'Could not resolve account',
            ];
        } catch (\Exception $e) {
            Log::error('Paystack account resolution failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Could not resolve account.',
            ];
        }
    }

    /**
     * Make API call to Paystack.
     */
    protected function makeApiCall(string $endpoint, array $data = [], string $method = 'POST'): array
    {
        if (config('app.env') === 'local') {
            // Return mock response in local environment
            Log::info("Paystack API Call (MOCK): {$method} {$endpoint}", $data);
            
            // Return mock successful response based on endpoint
            if (str_contains($endpoint, 'initialize')) {
                return [
                    'status' => true,
                    'data' => [
                        'authorization_url' => 'https://checkout.paystack.com/mock',
                        'access_code' => 'mock_access_code',
                        'reference' => $data['reference'] ?? 'mock_reference',
                    ]
                ];
            }
            
            return ['status' => true, 'data' => ['mock' => true]];
        }

        $http = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->secretKey,
            'Content-Type' => 'application/json',
        ]);

        $response = $method === 'GET' 
            ? $http->get($this->baseUrl . $endpoint)
            : $http->post($this->baseUrl . $endpoint, $data);

        if ($response->successful()) {
            return $response->json();
        }

        throw new \Exception('Paystack API call failed: ' . $response->body());
    }
}