<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\VirtualAccount;
use App\Services\MonnifyService;

/**
 * Virtual Account Web Controller
 * 
 * TODO: Add comprehensive input validation
 * TODO: Add CSRF protection for all forms
 * TODO: Add user-friendly error messages
 * TODO: Add success/error flash messages
 */
class VirtualAccountController extends Controller
{
    protected $monnifyService;

    public function __construct(MonnifyService $monnifyService)
    {
        $this->monnifyService = $monnifyService;
    }

    /**
     * Display virtual accounts page
     * 
     * TODO: Add pagination for large account lists
     * TODO: Add account filtering options
     */
    public function index(Request $request)
    {
        try {
            $user = $request->user();
            
            $accounts = VirtualAccount::where('user_id', $user->id)
                ->with(['bank'])
                ->orderBy('created_at', 'desc')
                ->paginate(10);

            return view('virtual-accounts.index', compact('accounts'));

        } catch (\Exception $e) {
            // TODO: Add proper error logging
            return back()->with('error', 'Failed to load virtual accounts');
        }
    }

    /**
     * Show create virtual account form
     * 
     * TODO: Add bank options from database
     * TODO: Add form pre-population
     */
    public function create()
    {
        try {
            // TODO: Get available banks from database
            $banks = [
                ['code' => '011', 'name' => 'First Bank'],
                ['code' => '058', 'name' => 'GTBank'],
                ['code' => '030', 'name' => 'Heritage Bank'],
                ['code' => '070', 'name' => 'Fidelity Bank'],
                // TODO: Load from banks table
            ];

            return view('virtual-accounts.create', compact('banks'));

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to load create form');
        }
    }

    /**
     * Store new virtual account
     * 
     * TODO: Add comprehensive validation
     * TODO: Add account creation limits
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'bank_code' => 'required|string|max:10',
                'account_name' => 'required|string|max:100',
            ]);

            if ($validator->fails()) {
                return back()
                    ->withErrors($validator)
                    ->withInput();
            }

            $user = $request->user();

            // TODO: Check user's account creation limits
            // TODO: Verify user's KYC status

            // Create virtual account through service
            $accountData = [
                'account_name' => $request->account_name,
                'bank_code' => $request->bank_code,
                'user_id' => $user->id,
                'email' => $user->email,
                'phone' => $user->phone
            ];

            $response = $this->monnifyService->createVirtualAccount($accountData);

            if (!$response['success']) {
                return back()
                    ->with('error', 'Failed to create virtual account: ' . $response['message'])
                    ->withInput();
            }

            // Save account to database
            $virtualAccount = VirtualAccount::create([
                'user_id' => $user->id,
                'account_number' => $response['data']['account_number'],
                'account_name' => $response['data']['account_name'],
                'bank_code' => $request->bank_code,
                'bank_name' => $response['data']['bank_name'],
                'provider' => 'monnify',
                'provider_reference' => $response['data']['account_reference'],
                'status' => 'active'
            ]);

            // TODO: Send creation notification
            // TODO: Log account creation

            return redirect()
                ->route('virtual-account.index')
                ->with('success', 'Virtual account created successfully');

        } catch (\Exception $e) {
            return back()
                ->with('error', 'Failed to create virtual account')
                ->withInput();
        }
    }

    /**
     * Show virtual account details
     * 
     * TODO: Add transaction history
     * TODO: Add account analytics
     */
    public function show(Request $request, $accountId)
    {
        try {
            $user = $request->user();
            
            $account = VirtualAccount::where('id', $accountId)
                ->where('user_id', $user->id)
                ->with(['bank'])
                ->first();

            if (!$account) {
                return redirect()
                    ->route('virtual-account.index')
                    ->with('error', 'Virtual account not found');
            }

            // TODO: Get transaction history for this account
            $transactions = collect(); // Placeholder

            return view('virtual-accounts.show', compact('account', 'transactions'));

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to load account details');
        }
    }

    /**
     * Show edit form
     * 
     * TODO: Add editable account properties
     */
    public function edit(Request $request, $accountId)
    {
        try {
            $user = $request->user();
            
            $account = VirtualAccount::where('id', $accountId)
                ->where('user_id', $user->id)
                ->first();

            if (!$account) {
                return redirect()
                    ->route('virtual-account.index')
                    ->with('error', 'Virtual account not found');
            }

            return view('virtual-accounts.edit', compact('account'));

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to load edit form');
        }
    }

    /**
     * Update virtual account
     * 
     * TODO: Add updateable fields validation
     */
    public function update(Request $request, $accountId)
    {
        try {
            $validator = Validator::make($request->all(), [
                'account_name' => 'required|string|max:100',
                'status' => 'required|string|in:active,inactive',
            ]);

            if ($validator->fails()) {
                return back()
                    ->withErrors($validator)
                    ->withInput();
            }

            $user = $request->user();
            
            $account = VirtualAccount::where('id', $accountId)
                ->where('user_id', $user->id)
                ->first();

            if (!$account) {
                return redirect()
                    ->route('virtual-account.index')
                    ->with('error', 'Virtual account not found');
            }

            $account->update([
                'account_name' => $request->account_name,
                'status' => $request->status,
            ]);

            // TODO: Update account with provider if needed
            // TODO: Log account update

            return redirect()
                ->route('virtual-account.show', $account->id)
                ->with('success', 'Virtual account updated successfully');

        } catch (\Exception $e) {
            return back()
                ->with('error', 'Failed to update virtual account')
                ->withInput();
        }
    }

    /**
     * Deactivate virtual account
     * 
     * TODO: Add confirmation modal
     * TODO: Add balance check before deactivation
     */
    public function destroy(Request $request, $accountId)
    {
        try {
            $user = $request->user();
            
            $account = VirtualAccount::where('id', $accountId)
                ->where('user_id', $user->id)
                ->first();

            if (!$account) {
                return redirect()
                    ->route('virtual-account.index')
                    ->with('error', 'Virtual account not found');
            }

            // TODO: Check for pending transactions
            // TODO: Deactivate with provider

            $account->update(['status' => 'inactive']);

            // TODO: Log account deactivation

            return redirect()
                ->route('virtual-account.index')
                ->with('success', 'Virtual account deactivated successfully');

        } catch (\Exception $e) {
            return back()->with('error', 'Failed to deactivate virtual account');
        }
    }
}